/*
 * Decompiled with CFR 0.152.
 */
package org.hsqldb.persist;

import java.io.File;
import java.io.IOException;
import org.hsqldb.Database;
import org.hsqldb.HsqlException;
import org.hsqldb.HsqlNameManager;
import org.hsqldb.NumberSequence;
import org.hsqldb.Row;
import org.hsqldb.Session;
import org.hsqldb.Table;
import org.hsqldb.error.Error;
import org.hsqldb.lib.FileAccess;
import org.hsqldb.persist.Crypto;
import org.hsqldb.persist.DataFileCache;
import org.hsqldb.persist.DataFileDefrag;
import org.hsqldb.persist.HsqlDatabaseProperties;
import org.hsqldb.persist.ScriptRunner;
import org.hsqldb.scriptio.ScriptReaderBase;
import org.hsqldb.scriptio.ScriptReaderDecode;
import org.hsqldb.scriptio.ScriptReaderText;
import org.hsqldb.scriptio.ScriptWriterBase;
import org.hsqldb.scriptio.ScriptWriterEncode;
import org.hsqldb.scriptio.ScriptWriterText;

public class Log {
    private HsqlDatabaseProperties properties;
    private String baseFileName;
    private Database database;
    private FileAccess fa;
    ScriptWriterBase dbLogWriter;
    private String scriptFileName;
    private String logFileName;
    private boolean filesReadOnly;
    private long maxLogSize;
    private int writeDelay;
    private DataFileCache cache;
    private boolean isModified;

    Log(Database db) {
        this.database = db;
        this.fa = db.logger.getFileAccess();
        this.baseFileName = db.getPath();
        this.properties = db.getProperties();
    }

    void initParams() {
        this.maxLogSize = (long)this.database.logger.getLogSize() * 1024L * 1024L;
        this.writeDelay = this.database.logger.getWriteDelay();
        this.filesReadOnly = this.database.isFilesReadOnly();
        this.scriptFileName = this.baseFileName + ".script";
        this.logFileName = this.baseFileName + ".log";
    }

    void open() {
        this.initParams();
        int state = this.properties.getDBModified();
        switch (state) {
            case 3: {
                break;
            }
            case 1: {
                this.database.logger.logInfoEvent("open start - state modified");
                this.deleteNewAndOldFiles();
                this.deleteOldTempFiles();
                if (this.properties.isVersion18()) {
                    if (this.fa.isStreamElement(this.scriptFileName)) {
                        this.processScript();
                    } else {
                        this.database.schemaManager.createPublicSchema();
                    }
                    HsqlNameManager.HsqlName name = this.database.schemaManager.findSchemaHsqlName("PUBLIC");
                    if (name != null) {
                        this.database.schemaManager.setDefaultSchemaHsqlName(name);
                    }
                } else {
                    this.processScript();
                }
                this.processLog();
                this.checkpoint();
                break;
            }
            case 2: {
                this.database.logger.logInfoEvent("open start - state new files");
                this.renameNewDataFile();
                this.renameNewScript();
                this.deleteLog();
                this.backupData();
                this.properties.setDBModified(0);
            }
            case 0: {
                this.fa.removeElement(this.logFileName);
                this.database.logger.logInfoEvent("open start - state not modified");
                this.processScript();
                if (this.filesReadOnly || !this.isAnyCacheModified()) break;
                this.properties.setDBModified(1);
                this.checkpoint();
            }
        }
        if (!this.filesReadOnly) {
            this.openLog();
        }
    }

    void close(boolean script) {
        this.closeLog();
        this.deleteOldDataFiles();
        this.deleteOldTempFiles();
        this.deleteTempFileDirectory();
        this.writeScript(script);
        this.database.logger.textTableManager.closeAllTextCaches(script);
        if (this.cache != null) {
            this.cache.close();
        }
        this.properties.setProperty("hsqldb.script_format", this.database.logger.propScriptFormat);
        this.properties.setDBModified(2);
        this.deleteLog();
        boolean complete = true;
        if (this.cache != null) {
            if (script) {
                this.cache.deleteFile();
                this.cache.deleteBackup();
                if (this.fa.isStreamElement(this.cache.dataFileName)) {
                    this.database.logger.logInfoEvent("delete .data file failed ");
                    complete = false;
                }
                if (this.fa.isStreamElement(this.cache.backupFileName)) {
                    this.database.logger.logInfoEvent("delete .backup file failed ");
                    complete = false;
                }
            } else {
                this.cache.backupDataFile(false);
            }
        }
        if (this.fa.isStreamElement(this.logFileName)) {
            this.database.logger.logInfoEvent("delete .log file failed ");
            complete = false;
        }
        this.renameNewScript();
        if (complete) {
            this.properties.setDBModified(0);
        }
    }

    void shutdown() {
        if (this.cache != null) {
            this.cache.release();
        }
        this.database.logger.textTableManager.closeAllTextCaches(false);
        this.closeLog();
    }

    void deleteNewAndOldFiles() {
        this.deleteOldDataFiles();
        this.fa.removeElement(this.baseFileName + ".data" + ".new");
        this.fa.removeElement(this.baseFileName + ".backup" + ".new");
        this.fa.removeElement(this.scriptFileName + ".new");
    }

    void deleteBackup() {
        this.fa.removeElement(this.baseFileName + ".backup");
    }

    void backupData() {
        DataFileCache.backupFile(this.database, this.baseFileName + ".data", this.baseFileName + ".backup", false);
    }

    void renameNewDataFile() {
        DataFileCache.renameDataFile(this.database, this.baseFileName + ".data");
    }

    void renameNewBackup() {
        DataFileCache.renameBackupFile(this.database, this.baseFileName + ".backup");
    }

    void renameNewScript() {
        if (this.fa.isStreamElement(this.scriptFileName + ".new")) {
            this.fa.removeElement(this.scriptFileName);
            this.fa.renameElement(this.scriptFileName + ".new", this.scriptFileName);
        }
    }

    boolean renameNewDataFileDone() {
        return this.fa.isStreamElement(this.baseFileName + ".data") && !this.fa.isStreamElement(this.baseFileName + ".data" + ".new");
    }

    boolean renameNewScriptDone() {
        return this.fa.isStreamElement(this.scriptFileName) && !this.fa.isStreamElement(this.scriptFileName + ".new");
    }

    void deleteNewScript() {
        this.fa.removeElement(this.scriptFileName + ".new");
    }

    void deleteNewBackup() {
        this.fa.removeElement(this.baseFileName + ".backup" + ".new");
    }

    void deleteLog() {
        this.fa.removeElement(this.logFileName);
    }

    boolean isAnyCacheModified() {
        if (this.cache != null && this.cache.isModified()) {
            return true;
        }
        return this.database.logger.textTableManager.isAnyTextCacheModified();
    }

    private boolean checkpoint() {
        if (this.filesReadOnly) {
            return true;
        }
        boolean result = this.checkpointClose();
        boolean reopenResult = this.checkpointReopen();
        if (result) {
            this.database.lobManager.deleteUnusedLobs();
        } else {
            this.database.logger.logSevereEvent("checkpoint failed - see previous error", null);
        }
        return reopenResult;
    }

    void checkpoint(boolean defrag) {
        if (this.filesReadOnly) {
            return;
        }
        if (this.cache == null) {
            defrag = false;
        } else if (this.forceDefrag()) {
            defrag = true;
        }
        if (defrag) {
            this.defrag();
        } else {
            this.checkpoint();
        }
    }

    boolean checkpointClose() {
        if (this.filesReadOnly) {
            return true;
        }
        this.database.logger.logInfoEvent("checkpointClose start");
        this.synchLog();
        this.database.lobManager.synch();
        this.database.logger.logInfoEvent("checkpointClose synched");
        this.deleteOldDataFiles();
        try {
            this.writeScript(false);
            this.database.logger.logInfoEvent("checkpointClose script done");
            if (this.cache != null) {
                this.cache.reset();
                this.cache.backupDataFile(true);
            }
            this.properties.setProperty("hsqldb.script_format", this.database.logger.propScriptFormat);
            this.properties.setDBModified(2);
        }
        catch (Throwable t) {
            this.deleteNewScript();
            this.deleteNewBackup();
            this.database.logger.logSevereEvent("checkpoint failed - recovered", t);
            return false;
        }
        this.closeLog();
        this.deleteLog();
        this.renameNewScript();
        this.renameNewBackup();
        try {
            this.properties.setDBModified(0);
        }
        catch (Throwable e) {
            this.database.logger.logSevereEvent("logger.checkpointClose properties file save failed", e);
        }
        this.database.logger.logInfoEvent("checkpointClose end");
        return true;
    }

    boolean checkpointReopen() {
        if (this.filesReadOnly) {
            return true;
        }
        this.database.sessionManager.resetLoggedSchemas();
        try {
            if (this.cache != null) {
                this.cache.reopen();
            }
            if (this.dbLogWriter != null) {
                this.openLog();
            }
        }
        catch (Throwable e) {
            return false;
        }
        return true;
    }

    public void defrag() {
        this.database.logger.logInfoEvent("defrag start");
        try {
            this.synchLog();
            this.database.lobManager.synch();
            this.deleteOldDataFiles();
            DataFileDefrag dfd = this.cache.defrag();
            this.database.persistentStoreCollection.setNewTableSpaces();
            this.database.schemaManager.setIndexRoots(dfd.getIndexRoots());
            this.database.sessionManager.resetLoggedSchemas();
        }
        catch (HsqlException e) {
            throw e;
        }
        catch (Throwable e) {
            this.database.logger.logSevereEvent("defrag failure", e);
            throw Error.error(466, e);
        }
        this.database.logger.logInfoEvent("defrag end");
    }

    boolean forceDefrag() {
        long lostSize;
        long limit = (long)this.database.logger.propCacheDefragLimit * this.cache.getFileFreePos() / 100L;
        if (limit == 0L) {
            return false;
        }
        long floor = (long)this.database.logger.propFileSpaceValue * 1024L * 1024L;
        if (floor > limit) {
            limit = floor;
        }
        return (lostSize = this.cache.getLostBlockSize()) > limit;
    }

    boolean hasCache() {
        return this.cache != null;
    }

    DataFileCache getCache() {
        if (this.cache == null) {
            this.cache = new DataFileCache(this.database, this.baseFileName);
            this.cache.open(this.filesReadOnly);
        }
        return this.cache;
    }

    void setLogSize(int megas) {
        this.maxLogSize = (long)megas * 1024L * 1024L;
    }

    int getWriteDelay() {
        return this.writeDelay;
    }

    void setWriteDelay(int delay) {
        this.writeDelay = delay;
        if (this.dbLogWriter != null && this.dbLogWriter.getWriteDelay() != delay) {
            this.dbLogWriter.forceSync();
            this.dbLogWriter.stop();
            this.dbLogWriter.setWriteDelay(delay);
            this.dbLogWriter.start();
        }
    }

    public void setIncrementBackup(boolean val) {
        if (this.cache != null) {
            this.cache.setIncrementBackup(val);
        }
    }

    void writeOtherStatement(Session session, String s) {
        try {
            this.dbLogWriter.writeOtherStatement(session, s);
        }
        catch (IOException e) {
            throw Error.error(452, this.logFileName);
        }
        if (this.maxLogSize > 0L && this.dbLogWriter.size() > this.maxLogSize) {
            this.database.logger.setCheckpointRequired();
        }
        this.setModified();
    }

    void writeInsertStatement(Session session, Row row, Table t) {
        try {
            this.dbLogWriter.writeInsertStatement(session, row, t);
        }
        catch (IOException e) {
            throw Error.error(452, this.logFileName);
        }
    }

    void writeDeleteStatement(Session session, Table t, Object[] row) {
        try {
            this.dbLogWriter.writeDeleteStatement(session, t, row);
        }
        catch (IOException e) {
            throw Error.error(452, this.logFileName);
        }
    }

    void writeSequenceStatement(Session session, NumberSequence s) {
        try {
            this.dbLogWriter.writeSequenceStatement(session, s);
        }
        catch (IOException e) {
            throw Error.error(452, this.logFileName);
        }
        this.setModified();
    }

    void writeCommitStatement(Session session) {
        try {
            this.dbLogWriter.writeCommitStatement(session);
        }
        catch (IOException e) {
            throw Error.error(452, this.logFileName);
        }
        if (this.maxLogSize > 0L && this.dbLogWriter.size() > this.maxLogSize) {
            this.database.logger.setCheckpointRequired();
        }
        this.setModified();
    }

    private void setModified() {
        if (!this.isModified) {
            this.database.databaseProperties.setDBModified(1);
            this.isModified = true;
        }
    }

    void synchLog() {
        if (this.dbLogWriter != null) {
            this.dbLogWriter.forceSync();
        }
    }

    void openLog() {
        if (this.filesReadOnly) {
            return;
        }
        Crypto crypto = this.database.logger.getCrypto();
        try {
            this.dbLogWriter = crypto == null ? new ScriptWriterText(this.database, this.logFileName, false, false, false) : new ScriptWriterEncode(this.database, this.logFileName, crypto);
            this.dbLogWriter.setWriteDelay(this.writeDelay);
            this.dbLogWriter.start();
            this.isModified = false;
        }
        catch (Throwable e) {
            throw Error.error(452, this.logFileName);
        }
    }

    void closeLog() {
        if (this.dbLogWriter != null) {
            this.database.logger.logDetailEvent("log close size: " + this.dbLogWriter.size());
            this.dbLogWriter.close();
        }
    }

    void writeScript(boolean full) {
        ScriptWriterText scw;
        this.deleteNewScript();
        Crypto crypto = this.database.logger.getCrypto();
        if (crypto == null) {
            boolean compressed = this.database.logger.propScriptFormat == 3;
            scw = new ScriptWriterText(this.database, this.scriptFileName + ".new", full, compressed);
        } else {
            scw = new ScriptWriterEncode(this.database, this.scriptFileName + ".new", full, crypto);
        }
        scw.writeAll();
        scw.close();
        scw = null;
    }

    private void processScript() {
        ScriptReaderBase scr = null;
        try {
            Crypto crypto = this.database.logger.getCrypto();
            if (crypto == null) {
                boolean compressed = this.database.logger.propScriptFormat == 3;
                scr = new ScriptReaderText(this.database, this.scriptFileName, compressed);
            } else {
                scr = new ScriptReaderDecode(this.database, this.scriptFileName, crypto, false);
            }
            Session session = this.database.sessionManager.getSysSessionForScript(this.database);
            scr.readAll(session);
            scr.close();
        }
        catch (Throwable e) {
            if (scr != null) {
                scr.close();
                if (this.cache != null) {
                    this.cache.release();
                }
                this.database.logger.textTableManager.closeAllTextCaches(false);
            }
            this.database.logger.logWarningEvent("Script processing failure", e);
            if (e instanceof HsqlException) {
                throw (HsqlException)e;
            }
            if (e instanceof IOException) {
                throw Error.error(452, e);
            }
            if (e instanceof OutOfMemoryError) {
                throw Error.error(460);
            }
            throw Error.error(458, e);
        }
    }

    private void processLog() {
        if (this.fa.isStreamElement(this.logFileName)) {
            ScriptRunner.runScript(this.database, this.logFileName);
        }
    }

    void deleteOldDataFiles() {
        if (this.database.logger.isStoredFileAccess()) {
            return;
        }
        try {
            File file = new File(this.database.getCanonicalPath());
            File[] list = file.getParentFile().listFiles();
            if (list == null) {
                return;
            }
            for (int i = 0; i < list.length; ++i) {
                if (!list[i].getName().startsWith(file.getName()) || !list[i].getName().endsWith(".old")) continue;
                list[i].delete();
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
    }

    void deleteOldTempFiles() {
        try {
            if (this.database.logger.tempDirectoryPath == null) {
                return;
            }
            File file = new File(this.database.logger.tempDirectoryPath);
            File[] list = file.listFiles();
            if (list == null) {
                return;
            }
            for (int i = 0; i < list.length; ++i) {
                list[i].delete();
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
    }

    void deleteTempFileDirectory() {
        try {
            if (this.database.logger.tempDirectoryPath == null) {
                return;
            }
            File file = new File(this.database.logger.tempDirectoryPath);
            file.delete();
        }
        catch (Throwable throwable) {
            // empty catch block
        }
    }

    String getLogFileName() {
        return this.logFileName;
    }
}

