/*
 * Decompiled with CFR 0.152.
 */
package com.caucho.db.block;

import com.caucho.db.block.BlockReadWrite;
import com.caucho.db.block.BlockStore;
import com.caucho.util.FreeList;
import com.caucho.util.L10N;
import com.caucho.util.SyncCacheListener;
import java.io.IOException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class Block
implements SyncCacheListener {
    private static final L10N L = new L10N(Block.class);
    private static final Logger log = Logger.getLogger(Block.class.getName());
    private static final long INIT_DIRTY = 8192L;
    private static final FreeList<byte[]> _freeBuffers = new FreeList(256);
    private final BlockStore _store;
    private final long _blockId;
    private final Lock _readLock;
    private final Lock _writeLock;
    private final AtomicInteger _useCount = new AtomicInteger(1);
    private final AtomicBoolean _isWriteQueued = new AtomicBoolean();
    private final AtomicLong _dirty = new AtomicLong(8192L);
    private boolean _isFlushDirtyOnCommit;
    private boolean _isValid;
    private boolean _isDeallocate;
    private byte[] _buffer;
    private boolean _isRemoved;

    Block(BlockStore store, long blockId) {
        store.validateBlockId(blockId);
        this._store = store;
        this._blockId = blockId;
        ReentrantReadWriteLock rwLock = new ReentrantReadWriteLock();
        this._readLock = rwLock.readLock();
        this._writeLock = rwLock.writeLock();
        this._isFlushDirtyOnCommit = this._store.isFlushDirtyBlocksOnCommit();
        this._buffer = Block.allocateBuffer();
        if (log.isLoggable(Level.FINEST)) {
            log.finest(this + " create");
        }
    }

    public boolean isFlushDirtyOnCommit() {
        return this._isFlushDirtyOnCommit;
    }

    public void setFlushDirtyOnCommit(boolean isFlush) {
        this._isFlushDirtyOnCommit = isFlush;
    }

    public boolean isIndex() {
        long blockIndex = BlockStore.blockIdToIndex(this.getBlockId());
        return this.getStore().getAllocation(blockIndex) == 4;
    }

    public void validateIsIndex() {
        long blockIndex = BlockStore.blockIdToIndex(this.getBlockId());
        int allocCode = this.getStore().getAllocation(blockIndex);
        if (allocCode != 4) {
            throw new IllegalStateException(L.l("block {0} is not an index code={1}", (Object)this, allocCode));
        }
    }

    public BlockStore getStore() {
        return this._store;
    }

    public long getBlockId() {
        return this._blockId;
    }

    public final Lock getReadLock() {
        return this._readLock;
    }

    public final Lock getWriteLock() {
        return this._writeLock;
    }

    public final boolean isValid() {
        return this._isValid;
    }

    public boolean isDirty() {
        return this._dirty.get() != 8192L;
    }

    public final boolean allocate() {
        int useCount;
        do {
            if ((useCount = this._useCount.get()) >= 1) continue;
            return false;
        } while (!this._useCount.compareAndSet(useCount, useCount + 1));
        if (this.getBuffer() == null) {
            this._useCount.decrementAndGet();
            Thread.dumpStack();
            log.fine(this + " null buffer " + this + " " + this._useCount.get());
            return false;
        }
        if (log.isLoggable(Level.FINEST)) {
            log.finest(this + " allocate (" + useCount + ")");
        }
        if (useCount > 32 && log.isLoggable(Level.FINE)) {
            Thread.dumpStack();
            log.fine("using " + this + " " + useCount + " times");
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void read() throws IOException {
        if (this._isValid) {
            return;
        }
        Block block = this;
        synchronized (block) {
            if (!this._isValid) {
                if (this._store.getBlockManager().copyDirtyBlock(this)) {
                    this._isValid = true;
                    this.clearDirty();
                } else {
                    if (log.isLoggable(Level.FINEST)) {
                        log.finest("read db-block " + this);
                    }
                    this.clearDirty();
                    BlockReadWrite readWrite = this._store.getReadWrite();
                    readWrite.readBlock(this._blockId & 0xFFFFFFFFFFFFE000L, this.getBuffer(), 0, 8192);
                    this._isValid = true;
                }
            }
        }
    }

    public final byte[] getBuffer() {
        return this._buffer;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void invalidate() {
        Block block = this;
        synchronized (block) {
            if (this._dirty.get() != 8192L) {
                throw new IllegalStateException();
            }
            this._isValid = false;
            this.clearDirty();
        }
    }

    void validate() {
        this._isValid = true;
    }

    public void setDirty(int min, int max) {
        int dirtyMin;
        int dirtyMax;
        long newDirty;
        long oldDirty;
        if (8192 < max || min < 0 || max < min) {
            throw new IllegalStateException("min=" + min + ", max=" + max);
        }
        do {
            oldDirty = this._dirty.get();
            dirtyMax = (int)(oldDirty >> 32);
            dirtyMin = (int)oldDirty;
            if (min < dirtyMin) {
                dirtyMin = min;
            }
            if (dirtyMax >= max) continue;
            dirtyMax = max;
        } while (!this._dirty.compareAndSet(oldDirty, newDirty = ((long)dirtyMax << 32) + (long)dirtyMin));
    }

    private void clearDirty() {
        this._dirty.set(8192L);
    }

    public void commit() throws IOException {
        if (!this._isFlushDirtyOnCommit) {
            return;
        }
        this.save();
    }

    public int getUseCount() {
        return this._useCount.get();
    }

    public boolean isRemoved() {
        return this._isRemoved;
    }

    public void deallocate() throws IOException {
        this._isDeallocate = true;
    }

    public void saveAllocation() throws IOException {
        this.getStore().saveAllocation();
    }

    public final void free() {
        int useCount = this._useCount.decrementAndGet();
        if (log.isLoggable(Level.FINEST)) {
            log.finest(this + " free (" + useCount + ")");
        }
        if (useCount < 2 && this._isDeallocate) {
            this._isDeallocate = false;
            try {
                this.getStore().freeBlock(this.getBlockId());
            }
            catch (Exception e) {
                log.log(Level.FINER, e.toString(), e);
            }
        }
        if (useCount < 1) {
            this.freeImpl();
        }
    }

    public boolean startLruRemove() {
        this.save();
        if (this._useCount.compareAndSet(1, 0)) {
            this.save();
            return true;
        }
        return false;
    }

    public final void syncLruRemoveEvent() {
        this._isRemoved = true;
        if (!this.isDirty()) {
            this.freeImpl();
        }
    }

    public final void syncRemoveEvent() {
        this._useCount.set(0);
        this._isRemoved = true;
        if (!this._isWriteQueued.get()) {
            this.freeImpl();
        }
    }

    private boolean save() {
        if (this._dirty.get() == 8192L) {
            return false;
        }
        if (this._isWriteQueued.compareAndSet(false, true)) {
            this._store.getWriter().addDirtyBlock(this);
        }
        return true;
    }

    void writeFromBlockWriter() throws IOException {
        while (true) {
            int use;
            if ((use = this._useCount.get()) >= 0 && !this._useCount.compareAndSet(use, use + 1)) {
                continue;
            }
            if (use >= 0) {
                int dirtyMax;
                long dirty = this._dirty.getAndSet(8192L);
                int dirtyMin = (int)dirty;
                if (dirtyMin < (dirtyMax = (int)(dirty >> 32))) {
                    if (log.isLoggable(Level.FINEST)) {
                        log.finest("write db-block " + this + " [" + dirtyMin + ", " + dirtyMax + "]");
                    }
                    boolean isPriority = false;
                    this.writeImpl(dirtyMin, dirtyMax - dirtyMin, isPriority);
                }
                this._useCount.decrementAndGet();
            }
            this._isWriteQueued.set(false);
            if (use < 0 || this._dirty.get() == 8192L) break;
        }
        if (this._useCount.get() == 0) {
            this.freeImpl();
        }
    }

    private void writeImpl(int offset, int length, boolean isPriority) throws IOException {
        BlockReadWrite readWrite = this._store.getReadWrite();
        readWrite.writeBlock((this._blockId & 0xFFFFFFFFFFFFE000L) + (long)offset, this.getBuffer(), offset, length, isPriority);
    }

    boolean copyToBlock(Block block) {
        int use;
        if (block == this) {
            return true;
        }
        while ((use = this._useCount.get()) >= 0 && !this._useCount.compareAndSet(use, use + 1)) {
        }
        if (use < 0) {
            return false;
        }
        byte[] buffer = this._buffer;
        boolean isValid = this.isValid();
        if (isValid) {
            System.arraycopy(buffer, 0, block.getBuffer(), 0, buffer.length);
            block.validate();
        }
        if (this._useCount.decrementAndGet() == 0 && !this.isDirty()) {
            this.freeImpl();
        }
        return isValid;
    }

    void freeImpl() {
        if (this._useCount.compareAndSet(0, -1)) {
            byte[] buffer = this._buffer;
            this._buffer = null;
            if (buffer != null) {
                _freeBuffers.free(buffer);
            }
            if (this._isDeallocate) {
                try {
                    this._store.freeBlock(this.getBlockId());
                }
                catch (Exception e) {
                    log.log(Level.WARNING, e.toString(), e);
                }
            }
        }
    }

    private static byte[] allocateBuffer() {
        byte[] buffer = _freeBuffers.allocate();
        if (buffer == null) {
            buffer = new byte[8192];
        }
        return buffer;
    }

    public String toString() {
        return this.getClass().getSimpleName() + "[" + this._store + "," + Long.toHexString(this._blockId) + "]";
    }
}

