/*
 * Decompiled with CFR 0.152.
 */
package com.caucho.db.index;

import com.caucho.db.Database;
import com.caucho.db.block.Block;
import com.caucho.db.block.BlockStore;
import com.caucho.db.index.KeyCompare;
import com.caucho.db.index.SqlIndexAlreadyExistsException;
import com.caucho.db.index.StringKeyCompare;
import com.caucho.sql.SQLExceptionWrapper;
import com.caucho.util.L10N;
import com.caucho.vfs.Path;
import java.io.IOException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.logging.Level;
import java.util.logging.Logger;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public final class BTree {
    private static final L10N L = new L10N(BTree.class);
    private static final Logger log = Logger.getLogger(BTree.class.getName());
    public static final long FAIL = 0L;
    private static final int BLOCK_SIZE = 8192;
    private static final int PTR_SIZE = 8;
    private static final int FLAGS_OFFSET = 0;
    private static final int LENGTH_OFFSET = 4;
    private static final int PARENT_OFFSET = 8;
    private static final int NEXT_OFFSET = 16;
    private static final int HEADER_SIZE = 24;
    private static final int LEAF_MASK = 3;
    private static final int IS_LEAF = 1;
    private static final int IS_NODE = 2;
    private BlockStore _store;
    private long _rootBlockId;
    private Block _rootBlock;
    private int _keySize;
    private int _tupleSize;
    private int _n;
    private int _minN;
    private KeyCompare _keyCompare;
    private long _timeout = 120000L;
    private volatile boolean _isStarted;

    public BTree(BlockStore store, long rootBlockId, int keySize, KeyCompare keyCompare) throws IOException {
        if (keyCompare == null) {
            throw new NullPointerException();
        }
        this._store = store;
        this._store.getBlockManager();
        this._rootBlockId = rootBlockId;
        this._rootBlock = store.readBlock(rootBlockId);
        if (8192 < keySize + 24) {
            throw new IOException(L.l("BTree key size '{0}' is too large.", keySize));
        }
        this._keySize = keySize;
        this._tupleSize = keySize + 8;
        this._n = 8168 / this._tupleSize;
        this._minN = (this._n + 1) / 2;
        if (this._minN < 0) {
            this._minN = 1;
        }
        this._keyCompare = keyCompare;
        byte[] rootBuffer = this._rootBlock.getBuffer();
        if (this.getInt(rootBuffer, 0) == 0) {
            this.setLeaf(rootBuffer, true);
        }
    }

    public long getIndexRoot() {
        return this._rootBlockId;
    }

    public void create() throws IOException {
    }

    public long lookup(byte[] keyBuffer, int keyOffset, int keyLength) throws IOException, SQLException {
        try {
            return this.lookup(keyBuffer, keyOffset, keyLength, this._rootBlockId);
        }
        catch (InterruptedException e) {
            throw new IllegalStateException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private long lookup(byte[] keyBuffer, int keyOffset, int keyLength, long blockId) throws IOException, SQLException, InterruptedException {
        Block block;
        if (blockId == this._rootBlockId) {
            block = this._rootBlock;
            block.allocate();
        } else {
            block = this._store.loadBlock(blockId);
        }
        try {
            long value;
            Lock blockLock;
            block7: {
                long l;
                blockLock = block.getReadLock();
                blockLock.tryLock(this._timeout, TimeUnit.MILLISECONDS);
                try {
                    this.validateIndex(block);
                    block.read();
                    byte[] buffer = block.getBuffer();
                    boolean isLeaf = this.isLeaf(buffer, block);
                    value = this.lookupTuple(blockId, buffer, keyBuffer, keyOffset, keyLength, isLeaf);
                    if (!isLeaf && value != 0L) break block7;
                    l = value;
                    Object var15_12 = null;
                    blockLock.unlock();
                }
                catch (Throwable throwable) {
                    Object var15_14 = null;
                    blockLock.unlock();
                    throw throwable;
                }
                Object var17_15 = null;
                block.free();
                return l;
            }
            long l = this.lookup(keyBuffer, keyOffset, keyLength, value);
            Object var15_13 = null;
            blockLock.unlock();
            Object var17_16 = null;
            block.free();
            return l;
        }
        catch (Throwable throwable) {
            Object var17_17 = null;
            block.free();
            throw throwable;
        }
    }

    public void insert(byte[] keyBuffer, int keyOffset, int keyLength, long value, boolean isOverride) throws SQLException {
        try {
            while (!this.insert(keyBuffer, keyOffset, keyLength, value, isOverride, true, this._rootBlockId)) {
                this.splitRoot(this._rootBlockId);
            }
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (SQLException e) {
            throw e;
        }
        catch (Exception e) {
            log.log(Level.FINE, e.toString(), e);
            throw new SQLExceptionWrapper(e.toString(), e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean insert(byte[] keyBuffer, int keyOffset, int keyLength, long value, boolean isOverride, boolean isRead, long blockId) throws IOException, SQLException, InterruptedException {
        Block block;
        block5: {
            if (blockId == this._rootBlockId) {
                block = this._rootBlock;
                block.allocate();
            } else {
                block = this._store.loadBlock(blockId);
            }
            try {
                this.validateIndex(block);
                if (!isRead || !this.insertReadChild(keyBuffer, keyOffset, keyLength, value, isOverride, block)) break block5;
                boolean bl = true;
                Object var13_11 = null;
                block.free();
                return bl;
            }
            catch (Throwable throwable) {
                Object var13_13 = null;
                block.free();
                throw throwable;
            }
        }
        boolean bl = this.insertWriteChild(keyBuffer, keyOffset, keyLength, value, isOverride, block);
        Object var13_12 = null;
        block.free();
        return bl;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean insertReadChild(byte[] keyBuffer, int keyOffset, int keyLength, long value, boolean isOverride, Block block) throws IOException, SQLException, InterruptedException {
        byte[] buffer;
        long blockId;
        Lock blockLock;
        block5: {
            block4: {
                blockLock = block.getReadLock();
                blockLock.tryLock(this._timeout, TimeUnit.MILLISECONDS);
                try {
                    this.validateIndex(block);
                    block.read();
                    blockId = block.getBlockId();
                    buffer = block.getBuffer();
                    int length = this.getLength(buffer);
                    if (length != this._n) break block4;
                    boolean bl = false;
                    Object var17_14 = null;
                    blockLock.unlock();
                    return bl;
                }
                catch (Throwable throwable) {
                    Object var17_17 = null;
                    blockLock.unlock();
                    throw throwable;
                }
            }
            if (!this.isLeaf(buffer, block)) break block5;
            boolean bl = false;
            Object var17_15 = null;
            blockLock.unlock();
            return bl;
        }
        long childBlockId = this.lookupTuple(blockId, buffer, keyBuffer, keyOffset, keyLength, false);
        boolean bl = this.insert(keyBuffer, keyOffset, keyLength, value, isOverride, true, childBlockId);
        Object var17_16 = null;
        blockLock.unlock();
        return bl;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean insertWriteChild(byte[] keyBuffer, int keyOffset, int keyLength, long value, boolean isOverride, Block block) throws IOException, SQLException, InterruptedException {
        byte[] buffer;
        long blockId;
        Lock blockLock;
        block6: {
            block5: {
                blockLock = block.getWriteLock();
                blockLock.tryLock(this._timeout, TimeUnit.MILLISECONDS);
                block.read();
                this.validate(block);
                blockId = block.getBlockId();
                buffer = block.getBuffer();
                int length = this.getLength(buffer);
                if (length != this._n) break block5;
                boolean bl = false;
                Object var17_14 = null;
                blockLock.unlock();
                return bl;
            }
            if (!this.isLeaf(buffer, block)) break block6;
            this.insertValue(keyBuffer, keyOffset, keyLength, value, isOverride, block);
            this.validate(block);
            boolean bl = true;
            Object var17_15 = null;
            blockLock.unlock();
            return bl;
        }
        try {
            long childBlockId = this.lookupTuple(blockId, buffer, keyBuffer, keyOffset, keyLength, false);
            while (!this.insert(keyBuffer, keyOffset, keyLength, value, isOverride, true, childBlockId)) {
                this.split(block, childBlockId);
                childBlockId = this.lookupTuple(blockId, buffer, keyBuffer, keyOffset, keyLength, false);
            }
            this.validate(block);
            boolean bl = true;
            Object var17_16 = null;
            blockLock.unlock();
            return bl;
        }
        catch (Throwable throwable) {
            Object var17_17 = null;
            blockLock.unlock();
            throw throwable;
        }
    }

    private void insertValue(byte[] keyBuffer, int keyOffset, int keyLength, long value, boolean isOverride, Block block) throws IOException, SQLException {
        byte[] buffer = block.getBuffer();
        this.insertLeafBlock(block.getBlockId(), buffer, keyBuffer, keyOffset, keyLength, value, isOverride);
        block.setFlushDirtyOnCommit(false);
        block.setDirty(0, 8192);
    }

    private long insertLeafBlock(long blockId, byte[] buffer, byte[] keyBuffer, int keyOffset, int keyLength, long value, boolean isOverride) throws IOException, SQLException {
        int length;
        int tupleSize = this._tupleSize;
        int sublen = length = this.getLength(buffer);
        int min = 0;
        int max = length;
        int offset = 24;
        while (min < max) {
            int i = (min + max) / 2;
            offset = 24 + i * tupleSize;
            int cmp = this._keyCompare.compare(keyBuffer, keyOffset, buffer, offset + 8, keyLength);
            if (cmp == 0) {
                long oldValue;
                if (!isOverride && value != (oldValue = this.getPointer(buffer, offset))) {
                    throw new SqlIndexAlreadyExistsException(L.l("'{0}' insert of key '{1}' fails index uniqueness.", (Object)this._store, this._keyCompare.toString(keyBuffer, keyOffset, keyLength)));
                }
                this.setPointer(buffer, offset, value);
                return 0L;
            }
            if (0 < cmp) {
                min = i + 1;
                continue;
            }
            if (cmp >= 0) continue;
            max = i;
        }
        if (length < this._n) {
            offset = 24 + min * tupleSize;
            return this.addKey(blockId, buffer, offset, min, length, keyBuffer, keyOffset, keyLength, value);
        }
        throw new IllegalStateException("ran out of key space");
    }

    private long addKey(long blockId, byte[] buffer, int offset, int index, int length, byte[] keyBuffer, int keyOffset, int keyLength, long value) throws IOException {
        int tupleSize = this._tupleSize;
        if (index < length) {
            if (offset + tupleSize < 24) {
                throw new IllegalStateException();
            }
            System.arraycopy(buffer, offset, buffer, offset + tupleSize, (length - index) * tupleSize);
        }
        this.setPointer(buffer, offset, value);
        this.setLength(buffer, length + 1);
        if (log.isLoggable(Level.FINEST)) {
            log.finest("btree insert at " + this.debugId(blockId) + ":" + offset + " value:" + this.debugId(value));
        }
        if (offset + 8 < 24) {
            throw new IllegalStateException();
        }
        System.arraycopy(keyBuffer, keyOffset, buffer, offset + 8, keyLength);
        for (int j = 8 + keyLength; j < tupleSize; ++j) {
            buffer[offset + j] = 0;
        }
        return -value;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void split(Block parent, long blockId) throws IOException, SQLException, InterruptedException {
        Block block = this._store.readBlock(blockId);
        try {
            this.validate(block);
            Lock blockLock = block.getWriteLock();
            blockLock.tryLock(this._timeout, TimeUnit.MILLISECONDS);
            try {
                this.split(parent, block);
                this.validate(block);
                Object var7_5 = null;
                blockLock.unlock();
            }
            catch (Throwable throwable) {
                Object var7_6 = null;
                blockLock.unlock();
                throw throwable;
            }
            Object var9_8 = null;
            block.free();
        }
        catch (Throwable throwable) {
            Object var9_9 = null;
            block.free();
            throw throwable;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void split(Block parentBlock, Block block) throws IOException, SQLException {
        long parentId = parentBlock.getBlockId();
        long blockId = block.getBlockId();
        log.finest("btree splitting " + this.debugId(blockId));
        block.setFlushDirtyOnCommit(false);
        byte[] buffer = block.getBuffer();
        int length = this.getLength(buffer);
        if (length < this._n / 2) {
            return;
        }
        if (length < 2) {
            throw new IllegalStateException(L.l("illegal length '{0}' for block {1}", (Object)length, this.debugId(blockId)));
        }
        Block leftBlock = null;
        try {
            parentBlock.setFlushDirtyOnCommit(false);
            byte[] parentBuffer = parentBlock.getBuffer();
            int parentLength = this.getLength(parentBuffer);
            this.validate(parentId, parentBuffer);
            this.validate(blockId, buffer);
            leftBlock = this._store.allocateIndexBlock();
            leftBlock.setFlushDirtyOnCommit(false);
            byte[] leftBuffer = leftBlock.getBuffer();
            long leftBlockId = leftBlock.getBlockId();
            int pivot = length / 2;
            int pivotSize = pivot * this._tupleSize;
            int pivotEnd = 24 + pivotSize;
            int blockEnd = 24 + length * this._tupleSize;
            System.arraycopy(buffer, 24, leftBuffer, 24, pivotSize);
            this.setInt(leftBuffer, 0, this.getInt(buffer, 0));
            this.setLength(leftBuffer, pivot);
            this.setPointer(leftBuffer, 16, 0L);
            this.setPointer(leftBuffer, 8, parentId);
            System.arraycopy(buffer, pivotEnd, buffer, 24, blockEnd - pivotEnd);
            this.setLength(buffer, length - pivot);
            this.insertLeafBlock(parentId, parentBuffer, leftBuffer, pivotEnd - this._tupleSize + 8, this._keySize, leftBlockId, true);
            this.validate(parentId, parentBuffer);
            this.validate(leftBlockId, leftBuffer);
            this.validate(blockId, buffer);
            this.validate(block);
            this.validate(parentBlock);
            this.validate(leftBlock);
            leftBlock.setDirty(0, 8192);
            parentBlock.setDirty(0, 8192);
            Object var20_16 = null;
            if (leftBlock != null) {
                leftBlock.free();
            }
            block.setDirty(0, 8192);
        }
        catch (Throwable throwable) {
            Object var20_17 = null;
            if (leftBlock != null) {
                leftBlock.free();
            }
            block.setDirty(0, 8192);
            throw throwable;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void splitRoot(long rootBlockId) throws IOException, SQLException, InterruptedException {
        Block rootBlock = this._rootBlock;
        rootBlock.allocate();
        try {
            Lock rootLock = rootBlock.getWriteLock();
            rootLock.tryLock(this._timeout, TimeUnit.MILLISECONDS);
            try {
                this.splitRoot(rootBlock);
                this.validate(rootBlock);
                Object var6_4 = null;
                rootLock.unlock();
            }
            catch (Throwable throwable) {
                Object var6_5 = null;
                rootLock.unlock();
                throw throwable;
            }
            Object var8_7 = null;
            rootBlock.free();
        }
        catch (Throwable throwable) {
            Object var8_8 = null;
            rootBlock.free();
            throw throwable;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void splitRoot(Block parentBlock) throws IOException {
        long parentId = parentBlock.getBlockId();
        log.finest("btree splitting root " + parentId / 8192L);
        Block leftBlock = null;
        Block rightBlock = null;
        try {
            byte[] parentBuffer = parentBlock.getBuffer();
            int length = this.getLength(parentBuffer);
            if (length == 1) {
                Object var20_7 = null;
                if (leftBlock != null) {
                    leftBlock.free();
                }
                if (rightBlock != null) {
                    rightBlock.free();
                }
                return;
            }
            parentBlock.setFlushDirtyOnCommit(false);
            int parentFlags = this.getInt(parentBuffer, 0);
            leftBlock = this._store.allocateIndexBlock();
            leftBlock.setFlushDirtyOnCommit(false);
            long leftBlockId = leftBlock.getBlockId();
            rightBlock = this._store.allocateIndexBlock();
            rightBlock.setFlushDirtyOnCommit(false);
            long rightBlockId = rightBlock.getBlockId();
            int pivot = (length - 1) / 2;
            if (length <= 2 || this._n < length || pivot < 1 || length <= pivot) {
                throw new IllegalStateException(Long.toHexString(parentBlock.getBlockId()) + ": " + length + " is an illegal length, or pivot " + pivot + " is bad, with n=" + this._n);
            }
            int pivotOffset = 24 + pivot * this._tupleSize;
            long pivotValue = this.getPointer(parentBuffer, pivotOffset);
            byte[] leftBuffer = leftBlock.getBuffer();
            System.arraycopy(parentBuffer, 24, leftBuffer, 24, pivotOffset + this._tupleSize - 24);
            this.setInt(leftBuffer, 0, parentFlags);
            this.setLength(leftBuffer, pivot + 1);
            this.setPointer(leftBuffer, 8, parentId);
            this.setPointer(leftBuffer, 16, 0L);
            byte[] rightBuffer = rightBlock.getBuffer();
            if (length - pivot - 1 < 0) {
                throw new IllegalStateException("illegal length " + pivot + " " + length);
            }
            System.arraycopy(parentBuffer, pivotOffset + this._tupleSize, rightBuffer, 24, (length - pivot - 1) * this._tupleSize);
            this.setInt(rightBuffer, 0, parentFlags);
            this.setLength(rightBuffer, length - pivot - 1);
            this.setPointer(rightBuffer, 8, parentId);
            this.setPointer(rightBuffer, 16, this.getPointer(parentBuffer, 16));
            System.arraycopy(parentBuffer, pivotOffset, parentBuffer, 24, this._tupleSize);
            this.setPointer(parentBuffer, 24, leftBlockId);
            this.setLeaf(parentBuffer, false);
            this.setLength(parentBuffer, 1);
            this.setPointer(parentBuffer, 16, rightBlockId);
            parentBlock.setDirty(0, 8192);
            leftBlock.setDirty(0, 8192);
            rightBlock.setDirty(0, 8192);
            this.validate(parentBlock);
            this.validate(leftBlock);
            this.validate(rightBlock);
        }
        catch (Throwable throwable) {
            Object var20_9 = null;
            if (leftBlock != null) {
                leftBlock.free();
            }
            if (rightBlock != null) {
                rightBlock.free();
            }
            throw throwable;
        }
        Object var20_8 = null;
        if (leftBlock != null) {
            leftBlock.free();
        }
        if (rightBlock != null) {
            rightBlock.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void remove(byte[] keyBuffer, int keyOffset, int keyLength) throws SQLException {
        try {
            Block rootBlock = this._rootBlock;
            rootBlock.allocate();
            try {
                if (!this.removeRead(rootBlock, keyBuffer, keyOffset, keyLength)) {
                    this.removeWrite(rootBlock, keyBuffer, keyOffset, keyLength);
                }
                Object var6_8 = null;
                rootBlock.free();
            }
            catch (Throwable throwable) {
                Object var6_9 = null;
                rootBlock.free();
                throw throwable;
            }
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (SQLException e) {
            throw e;
        }
        catch (Exception e) {
            throw new SQLExceptionWrapper(e.toString(), e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * WARNING - Removed back jump from a try to a catch block - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean removeRead(Block block, byte[] keyBuffer, int keyOffset, int keyLength) throws IOException, SQLException, InterruptedException {
        boolean bl;
        Lock blockLock;
        block12: {
            boolean bl2;
            block11: {
                boolean bl3;
                block10: {
                    blockLock = block.getReadLock();
                    blockLock.tryLock(this._timeout, TimeUnit.MILLISECONDS);
                    try {
                        this.validateIndex(block);
                        byte[] buffer = block.getBuffer();
                        long blockId = block.getBlockId();
                        if (this.isLeaf(buffer, block)) {
                            boolean bl4 = false;
                            Object var16_10 = null;
                            blockLock.unlock();
                            return bl4;
                        }
                        long childId = this.lookupTuple(blockId, buffer, keyBuffer, keyOffset, keyLength, false);
                        if (childId == 0L) {
                            bl3 = true;
                            break block10;
                        }
                        Block childBlock = this._store.readBlock(childId);
                        try {
                            this.validateIndex(childBlock);
                            if (this.removeRead(childBlock, keyBuffer, keyOffset, keyLength)) {
                                bl2 = true;
                                Object var14_19 = null;
                                childBlock.free();
                                break block11;
                            }
                        }
                        catch (Throwable throwable) {
                            Object var14_21 = null;
                            childBlock.free();
                            throw throwable;
                        }
                        {
                            bl = this.removeWrite(childBlock, keyBuffer, keyOffset, keyLength);
                            Object var14_20 = null;
                            childBlock.free();
                        }
                        break block12;
                    }
                    catch (Throwable throwable) {
                        Object var16_14 = null;
                        blockLock.unlock();
                        throw throwable;
                    }
                }
                Object var16_11 = null;
                blockLock.unlock();
                return bl3;
            }
            Object var16_12 = null;
            blockLock.unlock();
            return bl2;
        }
        Object var16_13 = null;
        blockLock.unlock();
        return bl;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean removeWrite(Block block, byte[] keyBuffer, int keyOffset, int keyLength) throws IOException, SQLException, InterruptedException {
        boolean bl;
        byte[] buffer = block.getBuffer();
        long blockId = block.getBlockId();
        Lock blockLock = block.getWriteLock();
        blockLock.tryLock(this._timeout, TimeUnit.MILLISECONDS);
        try {
            boolean isLeaf = this.isLeaf(buffer, block);
            if (isLeaf) {
                block.setFlushDirtyOnCommit(false);
                this.removeLeafEntry(blockId, buffer, keyBuffer, keyOffset, keyLength);
                block.setDirty(0, 8192);
            } else {
                long childId = this.lookupTuple(blockId, buffer, keyBuffer, keyOffset, keyLength, isLeaf);
                if (childId == 0L) {
                    boolean bl2 = true;
                    Object var17_13 = null;
                    blockLock.unlock();
                    return bl2;
                }
                Block childBlock = this._store.readBlock(childId);
                try {
                    boolean isJoin;
                    this.validateIndex(childBlock);
                    boolean bl3 = isJoin = !this.removeWrite(childBlock, keyBuffer, keyOffset, keyLength);
                    if (isJoin && this.joinBlocks(block, childBlock)) {
                        if (childBlock.getUseCount() > 2) {
                            System.out.println("USE: " + childBlock.getUseCount() + " " + block);
                        }
                        childBlock.deallocate();
                    }
                    this.validate(block);
                    Object var15_17 = null;
                    childBlock.free();
                }
                catch (Throwable throwable) {
                    Object var15_18 = null;
                    childBlock.free();
                    throw throwable;
                }
            }
            bl = this._minN <= this.getLength(buffer);
        }
        catch (Throwable throwable) {
            Object var17_15 = null;
            blockLock.unlock();
            throw throwable;
        }
        Object var17_14 = null;
        blockLock.unlock();
        return bl;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean joinBlocks(Block parent, Block block) throws IOException, SQLException, InterruptedException {
        Lock blockLock;
        byte[] rightBuffer;
        Block rightBlock;
        int leftLength;
        Lock leftLock;
        byte[] leftBuffer;
        Block leftBlock;
        long parentBlockId = parent.getBlockId();
        byte[] parentBuffer = parent.getBuffer();
        int parentLength = this.getLength(parentBuffer);
        long blockId = block.getBlockId();
        byte[] buffer = block.getBuffer();
        long leftBlockId = this.getLeftBlockId(parent, blockId);
        long rightBlockId = this.getRightBlockId(parent, blockId);
        if (leftBlockId > 0L) {
            leftBlock = this._store.readBlock(leftBlockId);
            try {
                leftBuffer = leftBlock.getBuffer();
                leftLock = leftBlock.getWriteLock();
                leftLock.tryLock(this._timeout, TimeUnit.MILLISECONDS);
                try {
                    Lock blockLock2;
                    block36: {
                        boolean bl;
                        leftLength = this.getLength(leftBuffer);
                        blockLock2 = block.getWriteLock();
                        blockLock2.tryLock(this._timeout, TimeUnit.MILLISECONDS);
                        try {
                            if (this._minN >= leftLength) break block36;
                            this.validateEqualLeaf(buffer, leftBuffer, block, leftBlock);
                            parent.setFlushDirtyOnCommit(false);
                            leftBlock.setFlushDirtyOnCommit(false);
                            this.validate(parentBlockId, parentBuffer);
                            this.validate(leftBlockId, leftBuffer);
                            this.validate(blockId, buffer);
                            this.moveFromLeft(parentBuffer, leftBuffer, buffer, blockId);
                            this.validate(parentBlockId, parentBuffer);
                            this.validate(leftBlockId, leftBuffer);
                            this.validate(blockId, buffer);
                            parent.setDirty(0, 8192);
                            leftBlock.setDirty(0, 8192);
                            bl = false;
                            Object var21_23 = null;
                            blockLock2.unlock();
                        }
                        catch (Throwable throwable) {
                            Object var21_25 = null;
                            blockLock2.unlock();
                            throw throwable;
                        }
                        Object var23_26 = null;
                        leftLock.unlock();
                        Object var25_29 = null;
                        leftBlock.free();
                        return bl;
                    }
                    Object var21_24 = null;
                    blockLock2.unlock();
                    Object var23_27 = null;
                    leftLock.unlock();
                }
                catch (Throwable throwable) {
                    Object var23_28 = null;
                    leftLock.unlock();
                    throw throwable;
                }
                Object var25_30 = null;
                leftBlock.free();
            }
            catch (Throwable throwable) {
                Object var25_31 = null;
                leftBlock.free();
                throw throwable;
            }
        }
        if (rightBlockId > 0L) {
            rightBlock = this._store.readBlock(rightBlockId);
            try {
                rightBuffer = rightBlock.getBuffer();
                blockLock = block.getWriteLock();
                blockLock.tryLock(this._timeout, TimeUnit.MILLISECONDS);
                try {
                    Lock rightLock;
                    block37: {
                        boolean bl;
                        rightLock = rightBlock.getWriteLock();
                        rightLock.tryLock(this._timeout, TimeUnit.MILLISECONDS);
                        try {
                            int rightLength = this.getLength(rightBuffer);
                            if (this._minN >= rightLength) break block37;
                            this.validateEqualLeaf(buffer, rightBuffer, block, rightBlock);
                            parent.setFlushDirtyOnCommit(false);
                            rightBlock.setFlushDirtyOnCommit(false);
                            this.moveFromRight(parentBuffer, buffer, rightBuffer, blockId);
                            this.validate(parentBlockId, parentBuffer);
                            this.validate(blockId, buffer);
                            this.validate(rightBlockId, rightBuffer);
                            parent.setDirty(0, 8192);
                            rightBlock.setDirty(0, 8192);
                            bl = false;
                            Object var27_37 = null;
                            rightLock.unlock();
                        }
                        catch (Throwable throwable) {
                            Object var27_39 = null;
                            rightLock.unlock();
                            throw throwable;
                        }
                        Object var29_40 = null;
                        blockLock.unlock();
                        Object var31_43 = null;
                        rightBlock.free();
                        return bl;
                    }
                    Object var27_38 = null;
                    rightLock.unlock();
                    Object var29_41 = null;
                    blockLock.unlock();
                }
                catch (Throwable throwable) {
                    Object var29_42 = null;
                    blockLock.unlock();
                    throw throwable;
                }
                Object var31_44 = null;
                rightBlock.free();
            }
            catch (Throwable throwable) {
                Object var31_45 = null;
                rightBlock.free();
                throw throwable;
            }
        }
        if (parentLength < 2) {
            return false;
        }
        if (leftBlockId > 0L) {
            leftBlock = this._store.readBlock(leftBlockId);
            try {
                leftBuffer = leftBlock.getBuffer();
                leftLock = leftBlock.getWriteLock();
                leftLock.tryLock(this._timeout, TimeUnit.MILLISECONDS);
                try {
                    Lock blockLock3;
                    block38: {
                        boolean bl;
                        leftLength = this.getLength(leftBuffer);
                        blockLock3 = block.getWriteLock();
                        blockLock3.tryLock(this._timeout, TimeUnit.MILLISECONDS);
                        try {
                            int length = this.getLength(buffer);
                            if (length + leftLength > this._n) break block38;
                            this.validateEqualLeaf(leftBuffer, buffer, leftBlock, block);
                            parent.setFlushDirtyOnCommit(false);
                            leftBlock.setFlushDirtyOnCommit(false);
                            this.mergeLeft(parentBuffer, leftBuffer, leftBlockId, buffer, blockId);
                            this.validate(parentBlockId, parentBuffer);
                            this.validate(leftBlockId, leftBuffer);
                            parent.setDirty(0, 8192);
                            leftBlock.setDirty(0, 8192);
                            bl = true;
                            Object var33_49 = null;
                            blockLock3.unlock();
                        }
                        catch (Throwable throwable) {
                            Object var33_51 = null;
                            blockLock3.unlock();
                            throw throwable;
                        }
                        Object var35_52 = null;
                        leftLock.unlock();
                        Object var37_55 = null;
                        leftBlock.free();
                        return bl;
                    }
                    Object var33_50 = null;
                    blockLock3.unlock();
                    Object var35_53 = null;
                    leftLock.unlock();
                }
                catch (Throwable throwable) {
                    Object var35_54 = null;
                    leftLock.unlock();
                    throw throwable;
                }
                Object var37_56 = null;
                leftBlock.free();
            }
            catch (Throwable throwable) {
                Object var37_57 = null;
                leftBlock.free();
                throw throwable;
            }
        }
        if (rightBlockId <= 0L) return false;
        rightBlock = this._store.readBlock(rightBlockId);
        try {
            rightBuffer = rightBlock.getBuffer();
            blockLock = block.getWriteLock();
            blockLock.tryLock(this._timeout, TimeUnit.MILLISECONDS);
            try {
                Lock rightLock;
                block39: {
                    boolean bl;
                    rightLock = rightBlock.getWriteLock();
                    rightLock.tryLock(this._timeout, TimeUnit.MILLISECONDS);
                    try {
                        int length = this.getLength(buffer);
                        int rightLength = this.getLength(rightBuffer);
                        if (length + rightLength > this._n) break block39;
                        this.validateEqualLeaf(rightBuffer, buffer, rightBlock, block);
                        rightBlock.setFlushDirtyOnCommit(false);
                        parent.setFlushDirtyOnCommit(false);
                        this.validate(blockId, buffer);
                        this.validate(parentBlockId, parentBuffer);
                        this.validate(rightBlockId, rightBuffer);
                        this.mergeRight(parentBuffer, buffer, rightBuffer, blockId);
                        this.validate(parentBlockId, parentBuffer);
                        this.validate(rightBlockId, rightBuffer);
                        rightBlock.setDirty(0, 8192);
                        parent.setDirty(0, 8192);
                        bl = true;
                        Object var39_61 = null;
                        rightLock.unlock();
                    }
                    catch (Throwable throwable) {
                        Object var39_63 = null;
                        rightLock.unlock();
                        throw throwable;
                    }
                    Object var41_64 = null;
                    blockLock.unlock();
                    Object var43_67 = null;
                    rightBlock.free();
                    return bl;
                }
                Object var39_62 = null;
                rightLock.unlock();
                Object var41_65 = null;
                blockLock.unlock();
            }
            catch (Throwable throwable) {
                Object var41_66 = null;
                blockLock.unlock();
                throw throwable;
            }
            Object var43_68 = null;
            rightBlock.free();
            return false;
        }
        catch (Throwable throwable) {
            Object var43_69 = null;
            rightBlock.free();
            throw throwable;
        }
    }

    private void validateEqualLeaf(byte[] leftBuffer, byte[] rightBuffer, Block left, Block right) {
        if (this.isLeaf(leftBuffer, left) != this.isLeaf(rightBuffer, right)) {
            throw new IllegalStateException(L.l("leaf mismatch {0} {1} and {2} {3}", this.isLeaf(leftBuffer, left), this.isLeaf(rightBuffer, right), left, right));
        }
    }

    private long getLeftBlockId(Block parent, long blockId) {
        long pointer;
        int offset;
        byte[] buffer = parent.getBuffer();
        int length = this.getLength(buffer);
        if (length < 1) {
            throw new IllegalStateException("zero length for " + this.debugId(parent.getBlockId()));
        }
        int tupleSize = this._tupleSize;
        int end = offset + length * tupleSize;
        for (offset = 24; offset < end; offset += tupleSize) {
            pointer = this.getPointer(buffer, offset);
            if (pointer != blockId) continue;
            if (24 < offset) {
                return this.getPointer(buffer, offset - tupleSize);
            }
            return -1L;
        }
        pointer = this.getPointer(buffer, 16);
        if (pointer == blockId) {
            return this.getPointer(buffer, 24 + (length - 1) * tupleSize);
        }
        throw new IllegalStateException("Can't find " + this.debugId(blockId) + " in parent " + this.debugId(parent.getBlockId()));
    }

    private void moveFromLeft(byte[] parentBuffer, byte[] leftBuffer, byte[] buffer, long blockId) {
        int parentLength = this.getLength(parentBuffer);
        int tupleSize = this._tupleSize;
        int parentEnd = 24 + parentLength * tupleSize;
        int parentOffset = 24;
        int leftLength = this.getLength(leftBuffer);
        int length = this.getLength(buffer);
        int parentLeftOffset = -1;
        if (blockId == this.getPointer(parentBuffer, 16)) {
            parentLeftOffset = parentEnd - tupleSize;
        } else {
            for (parentOffset = 24 + tupleSize; parentOffset < parentEnd; parentOffset += tupleSize) {
                long pointer = this.getPointer(parentBuffer, parentOffset);
                if (pointer != blockId) continue;
                parentLeftOffset = parentOffset - tupleSize;
                break;
            }
        }
        if (parentLeftOffset < 0) {
            log.warning("Can't find parent left in deletion borrow left ");
            return;
        }
        System.arraycopy(buffer, 24, buffer, 24 + tupleSize, length * tupleSize);
        int leftEnd = 24 + leftLength * tupleSize;
        System.arraycopy(leftBuffer, leftEnd - tupleSize, buffer, 24, tupleSize);
        this.setLength(buffer, length + 1);
        this.setLength(leftBuffer, --leftLength);
        leftEnd = 24 + leftLength * tupleSize;
        System.arraycopy(leftBuffer, leftEnd - tupleSize + 8, parentBuffer, parentLeftOffset + 8, tupleSize - 8);
    }

    private void mergeLeft(byte[] parentBuffer, byte[] leftBuffer, long leftBlockId, byte[] buffer, long blockId) {
        long pointer;
        if (this.isLeaf(leftBuffer) != this.isLeaf(buffer)) {
            throw new IllegalStateException("leaf does not match " + this.isLeaf(leftBuffer) + " " + this.isLeaf(buffer) + this.debugId(blockId));
        }
        int tupleSize = this._tupleSize;
        int parentLength = this.getLength(parentBuffer);
        int parentEnd = 24 + parentLength * tupleSize;
        int parentOffset = 24;
        int leftLength = this.getLength(leftBuffer);
        int leftEnd = 24 + leftLength * tupleSize;
        int blockLength = this.getLength(buffer);
        int blockSize = blockLength * tupleSize;
        parentOffset += tupleSize;
        while (parentOffset < parentEnd) {
            pointer = this.getPointer(parentBuffer, parentOffset);
            if (pointer == blockId) {
                System.arraycopy(parentBuffer, parentOffset, parentBuffer, parentOffset - tupleSize, parentEnd - parentOffset);
                this.setPointer(parentBuffer, parentOffset - tupleSize, leftBlockId);
                this.setLength(parentBuffer, parentLength - 1);
                this.setPointer(leftBuffer, 16, this.getPointer(buffer, 16));
                System.arraycopy(buffer, 24, leftBuffer, leftEnd, blockSize);
                this.setLength(leftBuffer, leftLength + blockLength);
                return;
            }
            parentOffset += tupleSize;
        }
        pointer = this.getPointer(parentBuffer, 16);
        if (pointer != blockId) {
            throw new IllegalStateException("BTree remove can't find matching block: " + this.debugId(blockId));
        }
        this.setPointer(parentBuffer, 16, leftBlockId);
        this.setLength(parentBuffer, parentLength - 1);
        this.setPointer(leftBuffer, 16, this.getPointer(buffer, 16));
        System.arraycopy(buffer, 24, leftBuffer, leftEnd, blockSize);
        this.setLength(leftBuffer, leftLength + blockLength);
    }

    private long getRightBlockId(Block parent, long blockId) {
        int offset;
        byte[] buffer = parent.getBuffer();
        int length = this.getLength(buffer);
        int tupleSize = this._tupleSize;
        int end = offset + length * tupleSize;
        for (offset = 24; offset < end; offset += tupleSize) {
            long pointer = this.getPointer(buffer, offset);
            if (pointer != blockId) continue;
            if (offset + tupleSize < end) {
                return this.getPointer(buffer, offset + tupleSize);
            }
            return this.getPointer(buffer, 16);
        }
        return -1L;
    }

    private void moveFromRight(byte[] parentBuffer, byte[] buffer, byte[] rightBuffer, long blockId) {
        long pointer;
        int parentOffset;
        int parentLength = this.getLength(parentBuffer);
        int tupleSize = this._tupleSize;
        int parentEnd = 24 + parentLength * tupleSize;
        int rightLength = this.getLength(rightBuffer);
        int rightSize = rightLength * tupleSize;
        int blockLength = this.getLength(buffer);
        int blockEnd = 24 + blockLength * tupleSize;
        for (parentOffset = 24; parentOffset < parentEnd && (pointer = this.getPointer(parentBuffer, parentOffset)) != blockId; parentOffset += tupleSize) {
        }
        if (parentEnd <= parentOffset) {
            log.warning("Can't find buffer in deletion borrow right ");
            return;
        }
        System.arraycopy(rightBuffer, 24, buffer, blockEnd, tupleSize);
        this.setLength(buffer, blockLength + 1);
        System.arraycopy(rightBuffer, 24 + tupleSize, rightBuffer, 24, rightSize - tupleSize);
        this.setLength(rightBuffer, rightLength - 1);
        System.arraycopy(buffer, blockEnd + 8, parentBuffer, parentOffset + 8, tupleSize - 8);
    }

    private void mergeRight(byte[] parentBuffer, byte[] buffer, byte[] rightBuffer, long blockId) {
        if (this.isLeaf(buffer) != this.isLeaf(rightBuffer)) {
            throw new IllegalStateException("leaf does not match " + this.isLeaf(buffer) + " " + this.isLeaf(rightBuffer) + this.debugId(blockId));
        }
        int tupleSize = this._tupleSize;
        int parentLength = this.getLength(parentBuffer);
        int parentEnd = 24 + parentLength * tupleSize;
        int rightLength = this.getLength(rightBuffer);
        int rightSize = rightLength * tupleSize;
        int blockLength = this.getLength(buffer);
        int blockSize = blockLength * tupleSize;
        for (int parentOffset = 24; parentOffset < parentEnd; parentOffset += tupleSize) {
            long pointer = this.getPointer(parentBuffer, parentOffset);
            if (pointer != blockId) continue;
            System.arraycopy(parentBuffer, parentOffset + tupleSize, parentBuffer, parentOffset, parentEnd - parentOffset - tupleSize);
            this.setLength(parentBuffer, parentLength - 1);
            System.arraycopy(rightBuffer, 24, rightBuffer, 24 + blockSize, rightSize);
            System.arraycopy(buffer, 24, rightBuffer, 24, blockSize);
            this.setLength(rightBuffer, blockLength + rightLength);
            return;
        }
        throw new IllegalStateException("BTree merge right can't find matching index: " + this.debugId(blockId));
    }

    private long lookupTuple(long blockId, byte[] buffer, byte[] keyBuffer, int keyOffset, int keyLength, boolean isLeaf) throws IOException {
        int length = this.getLength(buffer);
        int offset = 24;
        int tupleSize = this._tupleSize;
        int end = 24 + length * tupleSize;
        while (length > 0) {
            int tail = offset + tupleSize * length;
            int delta = tupleSize * (length / 2);
            int newOffset = offset + delta;
            if (newOffset < 0) {
                System.out.println("UNDERFLOW: " + this.debugId(blockId) + " LENGTH:" + length + " STU:" + this.getLength(buffer) + " DELTA:" + delta);
                throw new IllegalStateException("lookupTuple underflow newOffset:" + newOffset);
            }
            if (newOffset > 65536) {
                System.out.println("OVERFLOW: " + this.debugId(blockId) + " LENGTH:" + length + " STU:" + this.getLength(buffer) + " DELTA:" + delta);
                throw new IllegalStateException("lookupTuple overflow newOffset:" + newOffset);
            }
            int cmp = this._keyCompare.compare(keyBuffer, keyOffset, buffer, 8 + newOffset, keyLength);
            if (cmp == 0) {
                long value = this.getPointer(buffer, newOffset);
                if (value == 0L && !isLeaf) {
                    throw new IllegalStateException("illegal 0 value at " + newOffset + " for block " + this.debugId(blockId));
                }
                return value;
            }
            if (cmp > 0) {
                offset = newOffset + tupleSize;
                length = (tail - offset) / tupleSize;
            } else if (cmp < 0) {
                length /= 2;
            }
            if (length > 0) continue;
            if (isLeaf) {
                return 0L;
            }
            if (cmp < 0) {
                long value = this.getPointer(buffer, newOffset);
                if (value == 0L && !isLeaf) {
                    throw new IllegalStateException("illegal 0 value at " + newOffset + " for block " + this.debugId(blockId));
                }
                return value;
            }
            if (offset == end) {
                long value = this.getPointer(buffer, 16);
                if (value != 0L || isLeaf) {
                    return value;
                }
                return this.getPointer(buffer, end - tupleSize);
            }
            long value = this.getPointer(buffer, offset);
            if (value == 0L && !isLeaf) {
                throw new IllegalStateException("illegal 0 value at " + newOffset + " for block " + this.debugId(blockId));
            }
            return value;
        }
        if (isLeaf) {
            return 0L;
        }
        long value = this.getPointer(buffer, 16);
        if (value == 0L && !isLeaf) {
            throw new IllegalStateException("illegal 0 value at NEXT_OFFSET for block " + this.debugId(blockId));
        }
        return value;
    }

    private long removeLeafEntry(long blockIndex, byte[] buffer, byte[] keyBuffer, int keyOffset, int keyLength) throws IOException {
        int offset = 24;
        int tupleSize = this._tupleSize;
        int length = this.getLength(buffer);
        for (int i = 0; i < length; ++i) {
            int cmp = this._keyCompare.compare(keyBuffer, keyOffset, buffer, offset + 8, keyLength);
            if (0 < cmp) {
                offset += tupleSize;
                continue;
            }
            if (cmp == 0) {
                int blockEnd = 24 + length * tupleSize;
                if (offset + tupleSize < blockEnd) {
                    if (offset < 24) {
                        throw new IllegalStateException();
                    }
                    System.arraycopy(buffer, offset + tupleSize, buffer, offset, blockEnd - offset - tupleSize);
                }
                this.setLength(buffer, length - 1);
                return i;
            }
            return 0L;
        }
        return 0L;
    }

    private void validate(long blockId, byte[] buffer) {
        boolean isLeaf = this.isLeaf(buffer);
        if (isLeaf) {
            return;
        }
        int tupleSize = this._tupleSize;
        int length = this.getLength(buffer);
        int end = 24 + tupleSize * length;
        if (length < 0 || 8192 < end) {
            throw new IllegalStateException("illegal length " + length + " for " + this.debugId(blockId));
        }
        for (int offset = 24; offset < end; offset += tupleSize) {
            if (this.getPointer(buffer, offset) != 0L) continue;
            throw new IllegalStateException("Null pointer at " + offset + " for " + this.debugId(blockId) + " tupleSize=" + tupleSize);
        }
    }

    private boolean isLeaf(byte[] buffer, Block block) {
        int flags = this.getInt(buffer, 0) & 3;
        if (flags == 1) {
            return true;
        }
        if (flags == 2) {
            return false;
        }
        if (!block.isIndex()) {
            throw new IllegalStateException(L.l("block {0} is not an index block", block));
        }
        if (!block.isValid()) {
            throw new IllegalStateException(L.l("block {0} is not valid", block));
        }
        throw new IllegalStateException(L.l("leaf value is invalid: {0} for {1}", (Object)flags, block));
    }

    private void validate(Block block) {
        this.isLeaf(block.getBuffer(), block);
    }

    private void validateIndex(Block block) {
        if (block == this._rootBlock) {
            return;
        }
        block.validateIsIndex();
    }

    private boolean isLeaf(byte[] buffer) {
        int flags = this.getInt(buffer, 0) & 3;
        if (flags == 1) {
            return true;
        }
        if (flags == 2) {
            return false;
        }
        throw new IllegalStateException(L.l("leaf value is invalid: {0}", flags));
    }

    private void setLeaf(byte[] buffer, boolean isLeaf) {
        int flags = this.getInt(buffer, 0) & 0xFFFFFFFC;
        if (isLeaf) {
            this.setInt(buffer, 0, flags + 1);
        } else {
            this.setInt(buffer, 0, flags + 2);
        }
    }

    private int getInt(byte[] buffer, int offset) {
        return ((buffer[offset + 0] & 0xFF) << 24) + ((buffer[offset + 1] & 0xFF) << 16) + ((buffer[offset + 2] & 0xFF) << 8) + (buffer[offset + 3] & 0xFF);
    }

    private long getPointer(byte[] buffer, int offset) {
        return (((long)buffer[offset + 0] & 0xFFL) << 56) + (((long)buffer[offset + 1] & 0xFFL) << 48) + (((long)buffer[offset + 2] & 0xFFL) << 40) + (((long)buffer[offset + 3] & 0xFFL) << 32) + (((long)buffer[offset + 4] & 0xFFL) << 24) + (((long)buffer[offset + 5] & 0xFFL) << 16) + (((long)buffer[offset + 6] & 0xFFL) << 8) + ((long)buffer[offset + 7] & 0xFFL);
    }

    private void setInt(byte[] buffer, int offset, int value) {
        buffer[offset + 0] = (byte)(value >> 24);
        buffer[offset + 1] = (byte)(value >> 16);
        buffer[offset + 2] = (byte)(value >> 8);
        buffer[offset + 3] = (byte)value;
    }

    private void setLength(byte[] buffer, int value) {
        if (value < 0 || 8192 / this._tupleSize < value) {
            System.out.println("BAD-LENGTH: " + value);
            throw new IllegalArgumentException("BTree: bad length " + value);
        }
        this.setInt(buffer, 4, value);
    }

    private int getLength(byte[] buffer) {
        int value = this.getInt(buffer, 4);
        if (value < 0 || value > 65536) {
            System.out.println("BAD-LENGTH: " + value);
            throw new IllegalArgumentException("BTree: bad length " + value);
        }
        return value;
    }

    private void setPointer(byte[] buffer, int offset, long value) {
        if (offset <= 4) {
            System.out.println("BAD_POINTER: " + offset);
        }
        buffer[offset + 0] = (byte)(value >> 56);
        buffer[offset + 1] = (byte)(value >> 48);
        buffer[offset + 2] = (byte)(value >> 40);
        buffer[offset + 3] = (byte)(value >> 32);
        buffer[offset + 4] = (byte)(value >> 24);
        buffer[offset + 5] = (byte)(value >> 16);
        buffer[offset + 6] = (byte)(value >> 8);
        buffer[offset + 7] = (byte)value;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void start() throws IOException {
        BTree bTree = this;
        synchronized (bTree) {
            if (this._isStarted) {
                return;
            }
            this._isStarted = true;
        }
    }

    public ArrayList<String> getBlockKeys(long blockIndex) throws IOException {
        long blockId = this._store.addressToBlockId(blockIndex * 8192L);
        if (!this._store.isIndexBlock(blockId)) {
            return null;
        }
        Block block = this._store.readBlock(blockId);
        block.read();
        byte[] buffer = block.getBuffer();
        int length = this.getInt(buffer, 4);
        int offset = 24;
        int tupleSize = this._tupleSize;
        ArrayList<String> keys = new ArrayList<String>();
        for (int i = 0; i < length; ++i) {
            keys.add(this._keyCompare.toString(buffer, offset + i * tupleSize + 8, tupleSize - 8));
        }
        block.free();
        return keys;
    }

    public long getBlockNext(long blockIndex) throws IOException {
        long blockId = this._store.addressToBlockId(blockIndex * 8192L);
        if (!this._store.isIndexBlock(blockId)) {
            return -1L;
        }
        Block block = this._store.readBlock(blockId);
        block.read();
        byte[] buffer = block.getBuffer();
        long next = this.getPointer(buffer, 16);
        block.free();
        return next / 8192L;
    }

    public static BTree createTest(Path path, int keySize) throws IOException, SQLException {
        Database db = new Database();
        db.setPath(path);
        db.init();
        BlockStore store = new BlockStore(db, "test", null);
        store.create();
        Block block = store.allocateIndexBlock();
        long blockId = block.getBlockId();
        block.free();
        return new BTree(store, blockId, keySize, new KeyCompare());
    }

    public static BTree createStringTest(Path path, int keySize) throws IOException, SQLException {
        BlockStore store = BlockStore.create(path);
        Block block = store.allocateIndexBlock();
        long blockId = block.getBlockId();
        block.free();
        return new BTree(store, blockId, keySize, new StringKeyCompare());
    }

    private String debugId(long blockId) {
        return Long.toHexString(blockId);
    }

    public void close() {
        Block rootBlock = this._rootBlock;
        this._rootBlock = null;
        if (rootBlock != null) {
            rootBlock.free();
        }
    }

    public String toString() {
        return this.getClass().getSimpleName() + "[" + this._store + "," + this._rootBlockId / 8192L + "]";
    }
}

