/**
 * Copyright (c) 2000-2012 Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.portal.kernel.servlet.liferaypackage;

import com.liferay.portal.kernel.cluster.ClusterExecutorUtil;
import com.liferay.portal.kernel.cluster.ClusterNode;
import com.liferay.portal.kernel.cluster.ClusterNodeResponse;
import com.liferay.portal.kernel.cluster.ClusterNodeResponses;
import com.liferay.portal.kernel.cluster.ClusterRequest;
import com.liferay.portal.kernel.cluster.FutureClusterResponses;
import com.liferay.portal.kernel.dao.jdbc.DataAccess;
import com.liferay.portal.kernel.deploy.hot.LiferayPackageHotDeployException;
import com.liferay.portal.kernel.io.unsync.UnsyncByteArrayOutputStream;
import com.liferay.portal.kernel.servlet.ServletContextPool;
import com.liferay.portal.kernel.servlet.filters.LiferayPackageFilter;
import com.liferay.portal.kernel.servlet.filters.invoker.FilterMapping;
import com.liferay.portal.kernel.servlet.filters.invoker.InvokerFilterConfig;
import com.liferay.portal.kernel.servlet.filters.invoker.InvokerFilterHelper;
import com.liferay.portal.kernel.util.GetterUtil;
import com.liferay.portal.kernel.util.MethodHandler;
import com.liferay.portal.kernel.util.MethodKey;
import com.liferay.portal.kernel.util.PortalClassLoaderUtil;
import com.liferay.portal.kernel.util.StreamUtil;
import com.liferay.portal.kernel.util.StringPool;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.portal.kernel.workflow.WorkflowConstants;
import com.liferay.portal.license.util.LicenseManagerUtil;

import java.io.InputStream;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.servlet.Filter;
import javax.servlet.FilterConfig;
import javax.servlet.ServletContext;

/**
 * @author Amos Fong
 */
public class LiferayPackageUtil {

	public static void checkPackage() throws Exception {
		if (_getProductType() == _productTypeEE) {
			int licenseState = LicenseManagerUtil.getLicenseState(
				_productIdPortal);

			if (licenseState != _stateGood) {
				LicenseManagerUtil.checkLicense(_productIdPortal);

				licenseState = LicenseManagerUtil.getLicenseState(
					_productIdPortal);
			}

			if (licenseState != _stateGood) {
				throw new LiferayPackageHotDeployException(
					"This application requires a valid Liferay Portal EE " +
						"license.");
			}
		}

		if (Validator.isNull(_getProductId())) {
			return;
		}

		int licenseState = _getLicenseState(
			_getProductId(), _getProductVersion());

		if (licenseState != _stateGood) {
			LicenseManagerUtil.checkLicense(_getProductId());

			licenseState = _getLicenseState(
				_getProductId(), _getProductVersion());
		}

		if (licenseState != _stateGood) {
			throw new LiferayPackageHotDeployException(
				"This application does not have a valid license");
		}

		Map<String, String> licenseProperties =
			LicenseManagerUtil.getLicenseProperties(_getProductId());

		if (licenseProperties == null) {
			throw new LiferayPackageHotDeployException(
				"This Liferay version does not support this application.");
		}

		if (licenseProperties != null) {
			int maxValidProductVersion = GetterUtil.getInteger(
				licenseProperties.get("productVersion"));

			if ((_getProductVersion() > 0) &&
				(_getProductVersion() > maxValidProductVersion)) {

				throw new LiferayPackageHotDeployException(
					"The version of your application is not compatible with " +
						"the registered license");
			}
		}

		List<ClusterNode> clusterNodes = ClusterExecutorUtil.getClusterNodes();

		if (clusterNodes.size() <= 1) {
			return;
		}

		clusterNodes.remove(ClusterExecutorUtil.getLocalClusterNode());

		for (ClusterNode clusterNode : clusterNodes) {
			MethodHandler methodHandler = new MethodHandler(
				_getLicenseStateMethodKey, _getProductId());

			ClusterRequest clusterRequest = ClusterRequest.createUnicastRequest(
				methodHandler, clusterNode.getClusterNodeId());

			FutureClusterResponses futureClusterResponses =
				ClusterExecutorUtil.execute(clusterRequest);

			ClusterNodeResponses clusterNodeResponses =
				futureClusterResponses.get(10000, TimeUnit.MILLISECONDS);

			ClusterNodeResponse clusterNodeResponse =
				clusterNodeResponses.getClusterResponse(clusterNode);

			Object result = clusterNodeResponse.getResult();

			if (result == null) {
				return;
			}

			Integer remoteLicenseState = (Integer)result;

			if (remoteLicenseState != _stateGood) {
				throw new LiferayPackageHotDeployException(
					"A clustered server has an invalid license.");
			}
		}
	}

	public static void registerFilter(
			ServletContext servletContext, String pathContext)
		throws LiferayPackageHotDeployException {

		_pathContext = pathContext;

		Map<String, String> licenseProperties =
			LicenseManagerUtil.getLicenseProperties(_getProductId());

		String description = GetterUtil.getString(
			licenseProperties.get("description"));

		if (!description.startsWith("Developer License")) {
			return;
		}

		try {
			InvokerFilterHelper invokerFilterHelper = _getInvokerFilterHelper(
				pathContext);

			Filter liferayPackageFilter = _getFilter(
				servletContext.getServletContextName(),
				pathContext + "/c/portal/license");

			FilterConfig filterConfig = new InvokerFilterConfig(
				servletContext, _getFilterName(),
				new HashMap<String, String>());

			liferayPackageFilter.init(filterConfig);

			invokerFilterHelper.registerFilter(
				_getFilterName(), liferayPackageFilter);

			List<String> urlPatterns = new ArrayList<String>();

			urlPatterns.add("/*");

			_filterMapping = new FilterMapping(
				liferayPackageFilter, filterConfig, urlPatterns,
				new ArrayList<String>());

			invokerFilterHelper.registerFilterMapping(
				_filterMapping, null, false);
		}
		catch (Exception e) {
			throw new LiferayPackageHotDeployException(
				"Unable to intialize Liferay package filter");
		}
	}

	public static void unregisterFilter() {
		if (_pathContext == null) {
			return;
		}

		InvokerFilterHelper invokerFilterHelper = _getInvokerFilterHelper(
			_pathContext);

		if (invokerFilterHelper == null) {
			return;
		}

		Filter filter = invokerFilterHelper.registerFilter(
			_getFilterName(), null);

		if (filter != null) {
			filter.destroy();
		}

		if (_filterMapping != null) {
			invokerFilterHelper.unregisterFilterMapping(_filterMapping);
		}
	}

	private static Filter _getFilter(
			String servletContextName, String licensePageURL)
		throws Exception {

		ClassLoader classLoader = LiferayPackageUtil.class.getClassLoader();

		String className = LiferayPackageFilter.class.getName();

		InputStream inputStream = classLoader.getResourceAsStream(
			className.replace(StringPool.PERIOD, StringPool.SLASH) + ".class");

		UnsyncByteArrayOutputStream unsyncByteArrayOutputStream =
			new UnsyncByteArrayOutputStream();

		StreamUtil.transfer(inputStream, unsyncByteArrayOutputStream, true);

		byte[] bytes = unsyncByteArrayOutputStream.toByteArray();

		ClassLoader portalClassLoader = PortalClassLoaderUtil.getClassLoader();

		Method defineClassMethod = ClassLoader.class.getDeclaredMethod(
			"defineClass", String.class, byte[].class, int.class, int.class);

		defineClassMethod.setAccessible(true);

		Class<?> liferayPackageFilterClass = (Class<?>)defineClassMethod.invoke(
			portalClassLoader, LiferayPackageFilter.class.getName(), bytes, 0,
			bytes.length);

		Constructor<?> liferayPackageFilterConstructor =
			liferayPackageFilterClass.getConstructor(
				String.class, String.class);

		return (Filter)liferayPackageFilterConstructor.newInstance(
			servletContextName, licensePageURL);
	}

	private static String _getFilterName() {
		return "Liferay Package Filter - " + _getProductId();
	}

	private static InvokerFilterHelper _getInvokerFilterHelper(
		String pathContext) {

		ServletContext portalServletContext = ServletContextPool.get(
			pathContext);

		if (portalServletContext == null) {
			return null;
		}

		InvokerFilterHelper invokerFilterHelper =
			(InvokerFilterHelper)portalServletContext.getAttribute(
				InvokerFilterHelper.class.getName());

		return invokerFilterHelper;
	}

	private static int _getLicenseState(String productId, int productVersion)
		throws Exception {

		Map<String, String> licenseProperties = new HashMap<String, String>();

		licenseProperties.put("productId", productId);
		licenseProperties.put("productVersion", String.valueOf(productVersion));
		licenseProperties.put("userCount", String.valueOf(_getUserCount()));

		return LicenseManagerUtil.getLicenseState(licenseProperties);
	}

	private static String _getProductId() {
		return _productId;
	}

	private static int _getProductType() {
		return GetterUtil.getInteger(_productType);
	}

	private static int _getProductVersion() {
		return GetterUtil.getInteger(_productVersion);
	}

	private static long _getUserCount() throws Exception {
		Connection con = null;
		PreparedStatement ps = null;
		ResultSet rs = null;

		try {
			con = DataAccess.getConnection();

			ps = con.prepareStatement(
				"select count(*) from User_ where (defaultUser = ?) and " +
					"(status = ?)");

			ps.setBoolean(1, false);
			ps.setLong(2, WorkflowConstants.STATUS_APPROVED);

			rs = ps.executeQuery();

			while (rs.next()) {
				long count = rs.getLong(1);

				if (count > 0) {
					return count;
				}
			}
		}
		finally {
			DataAccess.cleanUp(con, ps, rs);
		}

		throw new Exception("Unable to count number of users on server");
	}

	private static final String _productId = "_PRODUCT_ID_";
	private static final String _productIdPortal = "Portal";
	private static final String _productType = "_PRODUCT_TYPE_";
	private static final int _productTypeEE = 2;
	private static final String _productVersion = "_PRODUCT_VERSION_";
	private static final int _stateGood = 3;

	private static FilterMapping _filterMapping;
	private static MethodKey _getLicenseStateMethodKey = new MethodKey(
		LicenseManagerUtil.class, "getLicenseState", String.class);
	private static String _pathContext;

}