(function(mod) {
  if (typeof exports == "object" && typeof module == "object") // CommonJS
    return mod(require("tern/lib/infer"), require("tern/lib/tern"));
  if (typeof define == "function" && define.amd) // AMD
    return define([ "tern/lib/infer", "tern/lib/tern" ], mod);
  mod(tern, tern);
})(function(infer, tern) {
  "use strict";

  function flattenPath(path) {
    if (!/(^|\/)(\.\/|[^\/]+\/\.\.\/)/.test(path)) return path;
    var parts = path.split("/");
    for (var i = 0; i < parts.length; ++i) {
      if (parts[i] == "." || !parts[i]) parts.splice(i--, 1);
      else if (i && parts[i] == "..") parts.splice(i-- - 1, 2);
    }
    return parts.join("/");
  }

  function resolveName(name, data) {
    var excl = name.indexOf("!");
    if (excl > -1) name = name.slice(0, excl);

    var opts = data.options;
    var hasExt = /\.js$/.test(name);
    if (hasExt || /^(?:\w+:|\/)/.test(name))
      return name + (hasExt ? "" : ".js");

    var base = opts.baseURL || "";
    if (base && base.charAt(base.length - 1) != "/") base += "/";
    if (opts.paths) {
      var known = opts.paths[name];
      if (known) return flattenPath(base + known + ".js");
      var dir = name.match(/^([^\/]+)(\/.*)$/);
      if (dir) {
        var known = opts.paths[dir[1]];
        if (known) return flattenPath(base + known + dir[2] + ".js");
      }
    }
    return flattenPath(base + name + ".js");
  }
  
  function getInterface(name, data) {
    var module = infer.cx().definitions["dojotoolkit"][name];
    if (module) return module;
 
    if (name == "require") return getRequire(data);
    if (name == "module") return infer.cx().definitions["dojotoolkit"].module;

    if (data.options.override && Object.prototype.hasOwnProperty.call(data.options.override, name)) {
      var over = data.options.override[name];
      if (typeof over == "string" && over.charAt(0) == "=") return infer.def.parsePath(over.slice(1));
      if (typeof over == "object") {
        var known = getKnownModule(name, data);
        if (known) return known;
        var scope = data.interfaces[stripJSExt(name)] = new infer.Obj(null, stripJSExt(name));
        infer.def.load(over, scope);
        return scope;
      }
      name = over;
    }

    if (!/^(https?:|\/)|\.js$/.test(name))
      name = resolveName(name, data);
    name = flattenPath(name);

    var known = getKnownModule(name, data);

    if (!known) {
      known = getModule(name, data);
      data.server.addFile(name, null, data.currentFile);
    }
    return known;
  }
  
  function getKnownModule(name, data) {
    return data.interfaces[stripJSExt(name)];
  }

  function getModule(name, data) {
    var known = getKnownModule(name, data);
    if (!known) {
      known = data.interfaces[stripJSExt(name)] = new infer.AVal;
      known.origin = name;
    }
    return known;
  }

  var EXPORT_OBJ_WEIGHT = 50;

  function stripJSExt(f) {
    return f.replace(/\.js$/, '');
  }
  
  var path = {
    dirname: function(path) {
      var lastSep = path.lastIndexOf("/");
      return lastSep == -1 ? "" : path.slice(0, lastSep);
    },
    relative: function(from, to) {
      if (to.indexOf(from) == 0) return to.slice(from.length);
      else return to;
    },
    join: function(a, b) {
      if (a && b) return a + "/" + b;
      else return (a || "") + (b || "");
    },
  };
    
  infer.registerFunction("requireDojo", function(_self, args, argNodes) {
    var server = infer.cx().parent, data = server && server._dojotoolkit;
    if (!data || !args.length) return infer.ANull;

    var name = data.currentFile;
    var out = getModule(name, data);

    var deps = [], fn;
    if (argNodes && args.length > 1) {
      var node = argNodes[args.length == 2 ? 0 : 1];
      var base = path.relative(server.options.projectDir, path.dirname(node.sourceFile.name));
      if (node.type == "Literal" && typeof node.value == "string") {
        deps.push(getInterface(path.join(base, node.value), data));
      } else if (node.type == "ArrayExpression") for (var i = 0; i < node.elements.length; ++i) {
        var elt = node.elements[i];
        if (elt.type == "Literal" && typeof elt.value == "string") {
          if (elt.value == "exports") {
            var exports = new infer.Obj(true);
            deps.push(exports);
            out.addType(exports, EXPORT_OBJ_WEIGHT);
          } else {
            deps.push(getInterface(path.join(base, elt.value), data));
          }
        }
      }
    } else if (argNodes && args.length == 1 && argNodes[0].type == "FunctionExpression" && argNodes[0].params.length) {
      // Simplified CommonJS call
      var exports = new infer.Obj(true);
      deps.push(getInterface("require", data), exports);
      out.addType(exports, EXPORT_OBJ_WEIGHT);
      fn = args[0];
    }

    if (!fn) {
      fn = args[Math.min(args.length - 1, 2)];
      if (!fn.isEmpty() && !fn.getFunctionType()) fn = null;
    }

    if (fn) fn.propagate(new infer.IsCallee(infer.ANull, deps, null, out));
    else if (args.length) args[0].propagate(out);

    return infer.ANull;
  });

  
  tern.registerPlugin("dojotoolkit1.10", function(server, options) {
    server._dojotoolkit = {
      interfaces: Object.create(null),
      options: options || {},
      currentFile: null,
      server: server
    };

    server.on("beforeLoad", function(file) {
      this._dojotoolkit.currentFile = file.name;
    });
    server.on("reset", function() {
      this._dojotoolkit.interfaces = Object.create(null);
      this._dojotoolkit.require = null;
    });
     return {
      defs : defs,
      passes: {completion: findCompletions,
               typeAt: findTypeAt}
    };
  });

  function findCompletions(file, query) {
    var wordEnd = tern.resolvePos(file, query.end);
    var callExpr = infer.findExpressionAround(file.ast, null, wordEnd, file.scope, "CallExpression");
    if (!callExpr) return;
    var callNode = callExpr.node;
    if (callNode.callee.type != "Identifier" || callNode.callee.name != "require" ||
        callNode.arguments.length < 1 || callNode.arguments[0].type != "ArrayExpression") return;
    var argNode = findNodeModule(callNode.arguments[0].elements, wordEnd);
    if (!argNode) return;
    var word = argNode.raw.slice(1, wordEnd - argNode.start), quote = argNode.raw.charAt(0);
    if (word && word.charAt(word.length - 1) == quote)
      word = word.slice(0, word.length - 1);
    var completions = completeModuleName(query, file, word);
    if (argNode.end == wordEnd + 1 && file.text.charAt(wordEnd) == quote)
      ++wordEnd;
    return {
      start: tern.outputPos(query, file, argNode.start),
      end: tern.outputPos(query, file, wordEnd),
      isProperty: false,
      isObjectKey: false,
      completions: completions.map(function(rec) {
        var name = typeof rec == "string" ? rec : rec.name;
        var string = JSON.stringify(name);
        if (quote == "'") string = quote + string.slice(1, string.length -1).replace(/'/g, "\\'") + quote;
        if (typeof rec == "string") return string;
        rec.displayName = name;
        rec.name = string;
        return rec;
      })
    };
  }
  
  function findNodeModule(argsNode, wordEnd) {
    for (var i = 0; i < argsNode.length; i++) {
      var argNode = argsNode[i];
      if (argNode.type == "Literal" && typeof argNode.value == "string" &&
          argNode.start < wordEnd && argNode.end > wordEnd) return argNode;
    }
  }
  
  function completeModuleName(query, file, word) {
    var completions = [];
    var cx = infer.cx(), server = cx.parent, data = server._dojotoolkit;
    var currentFile = null; //data.currentFile || resolveProjectPath(server, file.name);
    var wrapAsObjs = query.types || query.depths || query.docs || query.urls || query.origins;

    function maybeSet(obj, prop, val) {
      if (val != null) obj[prop] = val;
    }
    
    function gather(modules) {
      for (var name in modules) {
        if (name == currentFile) continue;

        var moduleName = name; //resolveModulePath(name, currentFile);
        if (moduleName &&
            !(query.filter !== false && word &&
              (query.caseInsensitive ? moduleName.toLowerCase() : moduleName).indexOf(word) !== 0)) {
          var rec = wrapAsObjs ? {name: moduleName} : moduleName;
          completions.push(rec);

          if (query.types || query.docs || query.urls || query.origins) {
            var val = modules[name];
            infer.resetGuessing();
            var type = val.getType();
            rec.guess = infer.didGuess();
            if (query.types)
              rec.type = infer.toString(val);
            if (query.docs)
              maybeSet(rec, "doc", val.doc || type && type.doc);
            if (query.urls)
              maybeSet(rec, "url", val.url || type && type.url);
            if (query.origins)
              maybeSet(rec, "origin", val.origin || type && type.origin);
          }
        }
      }
    }

    if (query.caseInsensitive) word = word.toLowerCase();
    gather(cx.definitions.dojotoolkit);
    gather(data.modules);
    return completions;
  }
  
  function findTypeAt(_file, _pos, expr, type) {
    if (!expr) return type;
    var isStringLiteral = expr.node.type === "Literal" &&
       typeof expr.node.value === "string";
    var isRequireArg = !!expr.node.required;

    if (isStringLiteral && isRequireArg) {
      // The `type` is a value shared for all string literals.
      // We must create a copy before modifying `origin` and `originNode`.
      // Otherwise all string literals would point to the last jump location
      type = Object.create(type);

      // Provide a custom origin location pointing to the require()d file
      var exportedType;
      if (expr.node.required && (exportedType = expr.node.required.getType())) {
        type.origin = exportedType.origin;
        type.originNode = exportedType.originNode;
      }
    }

    return type;
  }
  
  var defs = {
 "!name": "dojotoolkit",
 "!define": {
  "dojo/AdapterRegistry": {
   "!type": "fn(returnWrappers?: bool)",
   "!doc": "<p>A registry to make contextual calling/searching easier.</p>\n",
   "pairs": {},
   "returnWrappers": {},
   "prototype": {
    "match": {
     "!type": "fn()",
     "!doc": "<p>Find an adapter for the given arguments. If no suitable adapter\nis found, throws an exception. match() accepts any number of\narguments, all of which are passed to all matching functions\nfrom the registered pairs.</p>\n"
    },
    "register": {
     "!type": "fn(name: string, check: fn(), wrap: fn(), directReturn?: bool, override?: bool)",
     "!doc": "<p>register a check function to determine if the wrap function or\nobject gets selected</p>\n"
    },
    "unregister": {
     "!type": "fn(name: string) -> bool",
     "!doc": "<p>Remove a named adapter from the registry</p>\n"
    }
   }
  },
  "dojo/_base/kernel": {
   "_blockAsync": {},
   "_contentHandlers": {
    "!doc": "<p>A map of available XHR transport handle types. Name matches the\n<code>handleAs</code> attribute passed to XHR calls.</p>\n"
   },
   "_defaultContext": {},
   "_hasResource": {},
   "_initFired": {},
   "_name": {},
   "_nodeDataCache": {},
   "_NodeListCtor": {},
   "_postLoad": {},
   "_windowUnloaders": {},
   "back": {
    "!doc": "<p>Browser history management resources</p>\n"
   },
   "baseUrl": {},
   "behavior": {},
   "cldr": {},
   "colors": {},
   "config": {
    "!doc": "<p>This module defines the user configuration during bootstrap.</p>\n"
   },
   "connectPublisher": {},
   "contentHandlers": {
    "!doc": "<p>A map of available XHR transport handle types. Name matches the\n<code>handleAs</code> attribute passed to XHR calls.</p>\n"
   },
   "currency": {
    "!doc": "<p>localized formatting and parsing routines for currencies</p>\n"
   },
   "data": {},
   "date": {},
   "dijit": {},
   "dnd": {},
   "doc": {
    "!doc": "<p>Alias for the current document. &#39;doc&#39; can be modified\nfor temporary context shifting. See also withDoc().</p>\n"
   },
   "dojox": {},
   "fx": {
    "!doc": "<p>Effects library on top of Base animations</p>\n"
   },
   "gears": {
    "!doc": "<p>TODOC</p>\n"
   },
   "global": {
    "!doc": "<p>Alias for the current window. &#39;global&#39; can be modified\nfor temporary context shifting. See also withGlobal().</p>\n"
   },
   "html": {
    "!doc": "<p>TODOC</p>\n"
   },
   "i18n": {
    "!doc": "<p>This module implements the dojo/i18n! plugin and the v1.6- i18n API</p>\n"
   },
   "io": {},
   "isAir": {
    "!doc": "<p>True if client is Adobe Air</p>\n"
   },
   "isAndroid": {
    "!doc": "<p>Version as a Number if client is android browser. undefined otherwise.</p>\n"
   },
   "isAsync": {},
   "isBrowser": {},
   "isChrome": {
    "!doc": "<p>Version as a Number if client is Chrome browser. undefined otherwise.</p>\n"
   },
   "isCopyKey": {},
   "isFF": {},
   "isIE": {
    "!doc": "<p>Version as a Number if client is MSIE(PC). undefined otherwise. Corresponds to\nmajor detected IE version (6, 7, 8, etc.)</p>\n"
   },
   "isIos": {
    "!doc": "<p>Version as a Number if client is iPhone, iPod, or iPad. undefined otherwise.</p>\n"
   },
   "isKhtml": {
    "!doc": "<p>Version as a Number if client is a KHTML browser. undefined otherwise. Corresponds to major\ndetected version.</p>\n"
   },
   "isMac": {
    "!doc": "<p>True if the client runs on Mac</p>\n"
   },
   "isMoz": {},
   "isMozilla": {},
   "isOpera": {
    "!doc": "<p>Version as a Number if client is Opera. undefined otherwise. Corresponds to\nmajor detected version.</p>\n"
   },
   "isQuirks": {},
   "isSafari": {
    "!doc": "<p>Version as a Number if client is Safari or iPhone. undefined otherwise.</p>\n"
   },
   "isSpidermonkey": {},
   "isWebKit": {
    "!doc": "<p>Version as a Number if client is a WebKit-derived browser (Konqueror,\nSafari, Chrome, etc.). undefined otherwise.</p>\n"
   },
   "isWii": {
    "!doc": "<p>True if client is Wii</p>\n"
   },
   "keys": {
    "!doc": "<p>Definitions for common key values.  Client code should test keyCode against these named constants,\nas the actual codes can vary by browser.</p>\n"
   },
   "locale": {},
   "mouseButtons": {},
   "number": {
    "!doc": "<p>localized formatting and parsing routines for Number</p>\n"
   },
   "parser": {},
   "publish": {},
   "query": {},
   "regexp": {
    "!doc": "<p>Regular expressions and Builder resources</p>\n"
   },
   "rpc": {},
   "scopeMap": {},
   "store": {},
   "string": {
    "!doc": "<p>String utilities for Dojo</p>\n"
   },
   "subscribe": {},
   "tests": {
    "!doc": "<p>D.O.H. Test files for Dojo unit testing.</p>\n"
   },
   "toJsonIndentStr": {},
   "touch": {
    "!doc": "<p>This module provides unified touch event handlers by exporting\npress, move, release and cancel which can also run well on desktop.\nBased on <a href=\"http://dvcs.w3.org/hg/webevents/raw-file/tip/touchevents.html\">http://dvcs.w3.org/hg/webevents/raw-file/tip/touchevents.html</a>\nAlso, if the dojoClick property is set to truthy on a DOM node, dojo/touch generates\nclick events immediately for this node and its descendants (except for descendants that\nhave a dojoClick property set to falsy), to avoid the delay before native browser click events,\nand regardless of whether evt.preventDefault() was called in a touch.press event listener.\n</p>\n"
   },
   "version": {
    "!doc": "<p>Version number of the Dojo Toolkit</p>\n"
   },
   "window": {
    "!doc": "<p>TODOC</p>\n"
   },
   "AdapterRegistry": {
    "!type": "fn(returnWrappers?: bool)",
    "!doc": "<p>A registry to make contextual calling/searching easier.</p>\n"
   },
   "addClass": {
    "!type": "fn(node: string|+Node, classStr: string|[])",
    "!doc": "<p>Adds the specified classes to the end of the class list on the\npassed node. Will not re-apply duplicate classes.\n</p>\n"
   },
   "addOnLoad": {
    "!type": "fn(priority?: +Integer, context: ?, callback?: fn())",
    "!doc": "<p>Add a function to execute on DOM content loaded and all requested modules have arrived and been evaluated.\nIn most cases, the <code>domReady</code> plug-in should suffice and this method should not be needed.\n\n</p>\n<p>When called in a non-browser environment, just checks that all requested modules have arrived and been\nevaluated.</p>\n"
   },
   "addOnUnload": {
    "!type": "fn(obj?: ?|fn(), functionName?: string|fn())",
    "!doc": "<p>Registers a function to be triggered when the page unloads.\nDeprecated, use on(window, &quot;beforeunload&quot;, lang.hitch(obj, functionName)) instead.</p>\n"
   },
   "addOnWindowUnload": {
    "!type": "fn(obj?: ?, functionName?: string|fn())",
    "!doc": "<p>registers a function to be triggered when window.onunload fires.\nBe careful trying to modify the DOM or access JavaScript properties\nduring this phase of page unloading: they may not always be available.\nConsider dojo.addOnUnload() if you need to modify the DOM or do heavy\nJavaScript work.</p>\n"
   },
   "anim": {
    "!type": "fn(node: +Node|string, properties: ?, duration?: +Integer, easing?: fn(), onEnd?: fn(), delay?: +Integer)",
    "!doc": "<p>A simpler interface to <code>animateProperty()</code>, also returns\nan instance of <code>Animation</code> but begins the animation\nimmediately, unlike nearly every other Dojo animation API.</p>\n"
   },
   "animateProperty": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Returns an animation that will transition the properties of\nnode defined in <code>args</code> depending how they are defined in\n<code>args.properties</code>\n</p>\n"
   },
   "Animation": {
    "!type": "fn(args: ?)",
    "!doc": "<p>A generic animation class that fires callbacks into its handlers\nobject at various states.</p>\n"
   },
   "attr": {
    "!type": "fn(node: +Node|string, name: string|?, value?: string)",
    "!doc": "<p>Gets or sets an attribute on an HTML element.</p>\n"
   },
   "blendColors": {
    "!type": "fn(start: +dojo, end: +dojo, weight: number, obj?: +dojo)",
    "!doc": "<p>Blend colors end and start with weight from 0 to 1, 0.5 being a 50/50 blend,\ncan reuse a previously allocated Color object for the result</p>\n"
   },
   "body": {
    "!type": "fn(doc?: +Document)",
    "!doc": "<p>Return the body element of the specified document or of dojo/_base/window::doc.</p>\n"
   },
   "byId": {
    "!type": "fn(id: string|+Node, doc?: +Document)",
    "!doc": "<p>Returns DOM node with matching <code>id</code> attribute or falsy value (ex: null or undefined)\nif not found.  If <code>id</code> is a DomNode, this function is a no-op.\n</p>\n"
   },
   "cache": {
    "!type": "fn(module: string|?, url: string, value?: string|?)",
    "!doc": "<p>A getter and setter for storing the string content associated with the\nmodule and url arguments.</p>\n"
   },
   "clearCache": {
    "!type": "fn()"
   },
   "Color": {
    "!type": "fn(color: []|string|?)",
    "!doc": "<p>Takes a named string, hex string, array of rgb or rgba values,\nan object with r, g, b, and a properties, or another <code>Color</code> object\nand creates a new Color instance to work from.\n</p>\n"
   },
   "colorFromArray": {
    "!type": "fn(a: [], obj?: +dojo)",
    "!doc": "<p>Builds a <code>Color</code> from a 3 or 4 element array, mapping each\nelement in sequence to the rgb(a) values of the color.</p>\n"
   },
   "colorFromHex": {
    "!type": "fn(color: string, obj?: +dojo)",
    "!doc": "<p>Converts a hex string with a &#39;#&#39; prefix to a color object.\nSupports 12-bit #rgb shorthand. Optionally accepts a\n<code>Color</code> object to update with the parsed value.\n</p>\n"
   },
   "colorFromRgb": {
    "!type": "fn(color: string, obj?: +dojo)",
    "!doc": "<p>get rgb(a) array from css-style color declarations</p>\n"
   },
   "colorFromString": {
    "!type": "fn(str: string, obj?: +dojo)",
    "!doc": "<p>Parses <code>str</code> for a color value. Accepts hex, rgb, and rgba\nstyle color values.</p>\n"
   },
   "connect": {
    "!type": "fn(obj?: ?, event: string, context: ?|?, method: string|fn(), dontFix?: bool)",
    "!doc": "<p><code>dojo.connect</code> is a deprecated event handling and delegation method in\nDojo. It allows one function to &quot;listen in&quot; on the execution of\nany other, triggering the second whenever the first is called. Many\nlisteners may be attached to a function, and source functions may\nbe either regular function calls or DOM events.\n</p>\n"
   },
   "contentBox": {
    "!type": "fn(node: +Node|string, box?: ?)",
    "!doc": "<p>Getter/setter for the content-box of node.</p>\n"
   },
   "cookie": {
    "!type": "fn(name: string, value?: string, props?: ?)",
    "!doc": "<p>Get or set a cookie.</p>\n"
   },
   "coords": {
    "!type": "fn(node: +Node|string, includeScroll?: bool)",
    "!doc": "<p>Deprecated: Use position() for border-box x/y/w/h\nor marginBox() for margin-box w/h/l/t.\n\n</p>\n<p>Returns an object that measures margin-box (w)idth/(h)eight\nand absolute position x/y of the border-box. Also returned\nis computed (l)eft and (t)op values in pixels from the\nnode&#39;s offsetParent as returned from marginBox().\nReturn value will be in the form:\n\n</p>\n<pre><code>{ l: 50, t: 200, w: 300: h: 150, x: 100, y: 300 }</code></pre>\n<p>Does not act as a setter. If includeScroll is passed, the x and\n\n</p>\n<p>y params are affected as one would expect in dojo.position().</p>\n"
   },
   "create": {
    "!type": "fn(tag: +Node|string, attrs: ?, refNode?: +Node|string, pos?: string)",
    "!doc": "<p>Create an element, allowing for optional attribute decoration\nand placement.</p>\n"
   },
   "declare": {
    "!type": "fn(className?: string, superclass: fn()|fn(), props: ?) -> +dojo",
    "!doc": "<p>Create a feature-rich constructor from compact notation.</p>\n"
   },
   "Deferred": {
    "!type": "fn(canceller?: fn())",
    "!doc": "<p>Deprecated.   This module defines the legacy dojo/_base/Deferred API.\nNew code should use dojo/Deferred instead.</p>\n"
   },
   "DeferredList": {
    "!type": "fn(list: [], fireOnOneCallback?: bool, fireOnOneErrback?: bool, consumeErrors?: bool, canceller?: fn())",
    "!doc": "<p>Deprecated, use dojo/promise/all instead.\nProvides event handling for a group of Deferred objects.</p>\n"
   },
   "deprecated": {
    "!type": "fn(behaviour: string, extra?: string, removal?: string)",
    "!doc": "<p>Log a debug message to indicate that a behavior has been\ndeprecated.</p>\n"
   },
   "destroy": {
    "!type": "fn(node: ?)"
   },
   "disconnect": {
    "!type": "fn(handle: +Handle)",
    "!doc": "<p>Remove a link created by dojo.connect.</p>\n"
   },
   "docScroll": {
    "!type": "fn(doc?: +Document)",
    "!doc": "<p>Returns an object with {node, x, y} with corresponding offsets.</p>\n"
   },
   "empty": {
    "!type": "fn(node: ?)"
   },
   "eval": {
    "!type": "fn(scriptText: ?)",
    "!doc": "<p>A legacy method created for use exclusively by internal Dojo methods. Do not use this method\ndirectly unless you understand its possibly-different implications on the platforms your are targeting.</p>\n"
   },
   "every": {
    "!type": "fn(arr: []|string, callback: fn()|string, thisObject?: ?) -> bool",
    "!doc": "<p>Determines whether or not every item in arr satisfies the\ncondition implemented by callback.</p>\n"
   },
   "exit": {
    "!type": "fn(exitcode: ?)"
   },
   "experimental": {
    "!type": "fn(moduleName: string, extra?: string)",
    "!doc": "<p>Marks code as experimental.</p>\n"
   },
   "fadeIn": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Returns an animation that will fade node defined in &#39;args&#39; from\nits current opacity to fully opaque.</p>\n"
   },
   "fadeOut": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Returns an animation that will fade node defined in &#39;args&#39;\nfrom its current opacity to fully transparent.</p>\n"
   },
   "fieldToObject": {
    "!type": "fn(inputNode: +Node|string)",
    "!doc": "<p>Serialize a form field to a JavaScript object.</p>\n"
   },
   "filter": {
    "!type": "fn(arr: [], callback: fn()|string, thisObject?: ?) -> []",
    "!doc": "<p>Returns a new Array with those items from arr that match the\ncondition implemented by callback.</p>\n"
   },
   "fixEvent": {
    "!type": "fn(evt: +Event, sender: +Node) -> +Event",
    "!doc": "<p>normalizes properties on the event object including event\nbubbling methods, keystroke normalization, and x/y positions</p>\n"
   },
   "fixIeBiDiScrollLeft": {
    "!type": "fn(scrollLeft: number, doc?: +Document) -> number",
    "!doc": "<p>In RTL direction, scrollLeft should be a negative value, but IE\nreturns a positive one. All codes using documentElement.scrollLeft\nmust call this function to fix this error, otherwise the position\nwill offset to right when there is a horizontal scrollbar.</p>\n"
   },
   "forEach": {
    "!type": "fn(arr: []|string, callback: fn()|string, thisObject?: ?)",
    "!doc": "<p>for every item in arr, callback is invoked. Return values are ignored.\nIf you want to break out of the loop, consider using array.every() or array.some().\nforEach does not allow breaking out of the loop over the items in arr.</p>\n"
   },
   "formToJson": {
    "!type": "fn(formNode: +Node|string, prettyPrint?: bool) -> string",
    "!doc": "<p>Create a serialized JSON string from a form node or string\nID identifying the form to serialize</p>\n"
   },
   "formToObject": {
    "!type": "fn(formNode: +Node|string)",
    "!doc": "<p>Serialize a form node to a JavaScript object.</p>\n"
   },
   "formToQuery": {
    "!type": "fn(formNode: +Node|string) -> string",
    "!doc": "<p>Returns a URL-encoded string representing the form passed as either a\nnode or string ID identifying the form to serialize</p>\n"
   },
   "fromJson": {
    "!type": "fn(js: string)",
    "!doc": "<p>Parses a JavaScript expression and returns a JavaScript value.</p>\n"
   },
   "getAttr": {
    "!type": "fn(node: +Node|string, name: string)",
    "!doc": "<p>Gets an attribute on an HTML element.</p>\n"
   },
   "getBorderExtents": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>returns an object with properties useful for noting the border\ndimensions.</p>\n"
   },
   "getComputedStyle": {
    "!type": "fn(node: +Node)",
    "!doc": "<p>Returns a &quot;computed style&quot; object.\n</p>\n"
   },
   "getContentBox": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>Returns an object that encodes the width, height, left and top\npositions of the node&#39;s content box, irrespective of the\ncurrent box model.</p>\n"
   },
   "getIeDocumentElementOffset": {
    "!type": "fn(doc?: +Document)",
    "!doc": "<p>returns the offset in x and y from the document body to the\nvisual edge of the page for IE</p>\n"
   },
   "getL10nName": {
    "!type": "fn(moduleName: ?, bundleName: ?, locale: ?) -> string"
   },
   "getMarginBox": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>returns an object that encodes the width, height, left and top\npositions of the node&#39;s margin box.</p>\n"
   },
   "getMarginExtents": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>returns object with properties useful for box fitting with\nregards to box margins (i.e., the outer-box).\n\n</p>\n<ul>\n<li>l/t = marginLeft, marginTop, respectively</li>\n<li>w = total width, margin inclusive</li>\n<li>h = total height, margin inclusive</li>\n</ul>\n<p>The w/h are used for calculating boxes.\nNormally application code will not need to invoke this\ndirectly, and will use the ...box... functions instead.</p>\n"
   },
   "getMarginSize": {
    "!type": "fn(node: +Node|string, computedStyle?: ?)",
    "!doc": "<p>returns an object that encodes the width and height of\nthe node&#39;s margin box</p>\n"
   },
   "getNodeProp": {
    "!type": "fn(node: +Node|string, name: string)",
    "!doc": "<p>Returns an effective value of a property or an attribute.</p>\n"
   },
   "getPadBorderExtents": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>Returns object with properties useful for box fitting with\nregards to padding.</p>\n"
   },
   "getPadExtents": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>Returns object with special values specifically useful for node\nfitting.</p>\n"
   },
   "getProp": {
    "!type": "fn(node: +Node|string, name: string)",
    "!doc": "<p>Gets a property on an HTML element.</p>\n"
   },
   "getStyle": {
    "!type": "fn(node: +Node|string, name?: string)",
    "!doc": "<p>Accesses styles on a node.</p>\n"
   },
   "hasAttr": {
    "!type": "fn(node: +Node|string, name: string) -> bool",
    "!doc": "<p>Returns true if the requested attribute is specified on the\ngiven element, and false otherwise.</p>\n"
   },
   "hasClass": {
    "!type": "fn(node: string|+Node, classStr: string) -> bool",
    "!doc": "<p>Returns whether or not the specified classes are a portion of the\nclass list currently applied to the node.</p>\n"
   },
   "hash": {
    "!type": "fn(hash?: string, replace?: bool)",
    "!doc": "<p>Gets or sets the hash string in the browser URL.</p>\n"
   },
   "indexOf": {
    "!type": "fn(arr: [], value: ?, fromIndex?: +Integer, findLast?: bool) -> number",
    "!doc": "<p>locates the first index of the provided value in the\npassed array. If the value is not found, -1 is returned.</p>\n"
   },
   "isBodyLtr": {
    "!type": "fn(doc?: +Document) -> bool",
    "!doc": "<p>Returns true if the current language is left-to-right, and false otherwise.</p>\n"
   },
   "isDescendant": {
    "!type": "fn(node: +Node|string, ancestor: +Node|string) -> bool",
    "!doc": "<p>Returns true if node is a descendant of ancestor</p>\n"
   },
   "lastIndexOf": {
    "!type": "fn(arr: ?, value: ?, fromIndex?: +Integer) -> number",
    "!doc": "<p>locates the last index of the provided value in the passed\narray. If the value is not found, -1 is returned.</p>\n"
   },
   "loadInit": {
    "!type": "fn(f: ?)"
   },
   "map": {
    "!type": "fn(arr: []|string, callback: fn()|string, thisObject?: ?, Ctr: ?) -> []",
    "!doc": "<p>applies callback to each element of arr and returns\nan Array with the results</p>\n"
   },
   "marginBox": {
    "!type": "fn(node: +Node|string, box?: ?)",
    "!doc": "<p>Getter/setter for the margin-box of node.</p>\n"
   },
   "moduleUrl": {
    "!type": "fn(module: string, url?: string) -> string",
    "!doc": "<p>Returns a URL relative to a module.</p>\n"
   },
   "NodeList": {
    "!type": "fn(array: ?) -> []",
    "!doc": "<p>Array-like object which adds syntactic\nsugar for chaining, common iteration operations, animation, and\nnode manipulation. NodeLists are most often returned as the\nresult of dojo/query() calls.</p>\n"
   },
   "objectToQuery": {
    "!type": "fn(map: ?)",
    "!doc": "<p>takes a name/value mapping object and returns a string representing\na URL-encoded version of that object.</p>\n"
   },
   "place": {
    "!type": "fn(node: +Node|+DocumentFragment|string, refNode: +Node|string, position?: string|number) -> +Node",
    "!doc": "<p>Attempt to insert node into the DOM, choosing from various positioning options.\nReturns the first argument resolved to a DOM node.</p>\n"
   },
   "platformRequire": {
    "!type": "fn(modMap: ?)",
    "!doc": "<p>require one or more modules based on which host environment\nDojo is currently operating in</p>\n"
   },
   "popContext": {
    "!type": "fn()",
    "!doc": "<p>If the context stack contains elements, ensure that\nsubsequent code executes in the <em>previous</em> context to the\ncurrent context. The current context set ([global,\ndocument]) is returned.</p>\n"
   },
   "position": {
    "!type": "fn(node: +Node|string, includeScroll?: bool)",
    "!doc": "<p>Gets the position and size of the passed element relative to\nthe viewport (if includeScroll==false), or relative to the\ndocument root (if includeScroll==true).\n</p>\n"
   },
   "prop": {
    "!type": "fn(node: +Node|string, name: string|?, value?: string)",
    "!doc": "<p>Gets or sets a property on an HTML element.</p>\n"
   },
   "provide": {
    "!type": "fn(mid: ?)"
   },
   "pushContext": {
    "!type": "fn(g?: ?|string, d?: +MDocumentElement)",
    "!doc": "<p>causes subsequent calls to Dojo methods to assume the\npassed object and, optionally, document as the default\nscopes to use. A 2-element array of the previous global and\ndocument are returned.</p>\n"
   },
   "queryToObject": {
    "!type": "fn(str: string)",
    "!doc": "<p>Create an object representing a de-serialized query section of a\nURL. Query keys with multiple values are returned in an array.\n</p>\n"
   },
   "rawXhrPost": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP POST request to the server. In addition to the properties\nlisted for the dojo.__XhrArgs type, the following property is allowed:</p>\n"
   },
   "rawXhrPut": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP PUT request to the server. In addition to the properties\nlisted for the dojo.__XhrArgs type, the following property is allowed:</p>\n"
   },
   "ready": {
    "!type": "fn(priority?: +Integer, context: ?, callback?: fn())",
    "!doc": "<p>Add a function to execute on DOM content loaded and all requested modules have arrived and been evaluated.\nIn most cases, the <code>domReady</code> plug-in should suffice and this method should not be needed.\n\n</p>\n<p>When called in a non-browser environment, just checks that all requested modules have arrived and been\nevaluated.</p>\n"
   },
   "registerModulePath": {
    "!type": "fn(moduleName: string, prefix: string)",
    "!doc": "<p>Maps a module name to a path</p>\n"
   },
   "removeAttr": {
    "!type": "fn(node: +Node|string, name: string)",
    "!doc": "<p>Removes an attribute from an HTML element.</p>\n"
   },
   "removeClass": {
    "!type": "fn(node: string|+Node, classStr?: string|[])",
    "!doc": "<p>Removes the specified classes from node. No <code>contains()</code>\ncheck is required.\n</p>\n"
   },
   "replaceClass": {
    "!type": "fn(node: string|+Node, addClassStr: string|[], removeClassStr?: string|[])",
    "!doc": "<p>Replaces one or more classes on a node if not present.\nOperates more quickly than calling dojo.removeClass and dojo.addClass\n</p>\n"
   },
   "require": {
    "!type": "fn(moduleName: string, omitModuleCheck?: bool)",
    "!doc": "<p>loads a Javascript module from the appropriate URI\n</p>\n"
   },
   "requireAfterIf": {
    "!type": "fn(condition: bool, moduleName: string, omitModuleCheck?: bool)",
    "!doc": "<p>If the condition is true then call <code>dojo.require()</code> for the specified\nresource\n</p>\n"
   },
   "requireIf": {
    "!type": "fn(condition: bool, moduleName: string, omitModuleCheck?: bool)",
    "!doc": "<p>If the condition is true then call <code>dojo.require()</code> for the specified\nresource\n</p>\n"
   },
   "requireLocalization": {
    "!type": "fn(moduleName: string, bundleName: string, locale?: string)"
   },
   "safeMixin": {
    "!type": "fn(target: ?, source: ?)",
    "!doc": "<p>Mix in properties skipping a constructor and decorating functions\nlike it is done by declare().</p>\n"
   },
   "setAttr": {
    "!type": "fn(node: +Node|string, name: string|?, value?: string)",
    "!doc": "<p>Sets an attribute on an HTML element.</p>\n"
   },
   "setContentSize": {
    "!type": "fn(node: +Node, box: ?, computedStyle?: ?)",
    "!doc": "<p>Sets the size of the node&#39;s contents, irrespective of margins,\npadding, or borders.</p>\n"
   },
   "setContext": {
    "!type": "fn(globalObject: ?, globalDocument: +DocumentElement)",
    "!doc": "<p>changes the behavior of many core Dojo functions that deal with\nnamespace and DOM lookup, changing them to work in a new global\ncontext (e.g., an iframe). The varibles dojo.global and dojo.doc\nare modified as a result of calling this function and the result of\n<code>dojo.body()</code> likewise differs.</p>\n"
   },
   "setMarginBox": {
    "!type": "fn(node: +Node, box: ?, computedStyle?: ?)",
    "!doc": "<p>sets the size of the node&#39;s margin box and placement\n(left/top), irrespective of box model. Think of it as a\npassthrough to setBox that handles box-model vagaries for\nyou.</p>\n"
   },
   "setProp": {
    "!type": "fn(node: +Node|string, name: string|?, value?: string)",
    "!doc": "<p>Sets a property on an HTML element.</p>\n"
   },
   "setSelectable": {
    "!type": "fn(node: ?, selectable: ?)"
   },
   "setStyle": {
    "!type": "fn(node: +Node|string, name: string|?, value?: string) -> string",
    "!doc": "<p>Sets styles on a node.</p>\n"
   },
   "some": {
    "!type": "fn(arr: []|string, callback: fn()|string, thisObject?: ?) -> bool",
    "!doc": "<p>Determines whether or not any item in arr satisfies the\ncondition implemented by callback.</p>\n"
   },
   "Stateful": {
    "!type": "fn()"
   },
   "stopEvent": {
    "!type": "fn(evt: +Event)",
    "!doc": "<p>prevents propagation and clobbers the default action of the\npassed event</p>\n"
   },
   "style": {
    "!type": "fn(node: +Node|string, name?: string|?, value?: string)",
    "!doc": "<p>Accesses styles on a node. If 2 arguments are\npassed, acts as a getter. If 3 arguments are passed, acts\nas a setter.</p>\n"
   },
   "toDom": {
    "!type": "fn(frag: string, doc?: +DocumentNode)",
    "!doc": "<p>instantiates an HTML fragment returning the corresponding DOM.</p>\n"
   },
   "toggleClass": {
    "!type": "fn(node: string|+Node, classStr: string|[], condition?: bool) -> bool",
    "!doc": "<p>Adds a class to node if not present, or removes if present.\nPass a boolean condition if you want to explicitly add or remove.\nReturns the condition that was specified directly or indirectly.\n</p>\n"
   },
   "toJson": {
    "!type": "fn(it: ?, prettyPrint?: bool)",
    "!doc": "<p>Returns a <a href=\"http://json.org\">JSON</a> serialization of an object.</p>\n"
   },
   "toPixelValue": {
    "!type": "fn(node: +Node, value: string) -> number",
    "!doc": "<p>converts style value to pixels on IE or return a numeric value.</p>\n"
   },
   "unsubscribe": {
    "!type": "fn(handle: +Handle)",
    "!doc": "<p>Remove a topic listener.</p>\n"
   },
   "when": {
    "!type": "fn(valueOrPromise: ?, callback?: fn(), errback?: fn(), progback?: fn()) -> +dojo",
    "!doc": "<p>Transparently applies callbacks to values and/or promises.</p>\n"
   },
   "windowUnloaded": {
    "!type": "fn()",
    "!doc": "<p>signal fired by impending window destruction. You may use\ndojo.addOnWIndowUnload() or dojo.connect() to this method to perform\npage/application cleanup methods. See dojo.addOnWindowUnload for more info.</p>\n"
   },
   "withDoc": {
    "!type": "fn(documentObject: +DocumentElement, callback: fn(), thisObject?: ?, cbArguments?: [])",
    "!doc": "<p>Invoke callback with documentObject as dojo/_base/window::doc.</p>\n"
   },
   "withGlobal": {
    "!type": "fn(globalObject: ?, callback: fn(), thisObject?: ?, cbArguments?: [])",
    "!doc": "<p>Invoke callback with globalObject as dojo.global and\nglobalObject.document as dojo.doc.</p>\n"
   },
   "xhr": {
    "!type": "fn(method: ?, args: ?)"
   },
   "xhrDelete": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP DELETE request to the server.</p>\n"
   },
   "xhrGet": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP GET request to the server.</p>\n"
   },
   "xhrPost": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP POST request to the server. In addition to the properties\nlisted for the dojo.__XhrArgs type, the following property is allowed:</p>\n"
   },
   "xhrPut": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP PUT request to the server. In addition to the properties\nlisted for the dojo.__XhrArgs type, the following property is allowed:</p>\n"
   }
  },
  "dojo/_base/kernel.config": {
   "addOnLoad": {
    "!doc": "<p>Adds a callback via dojo/ready. Useful when Dojo is added after\nthe page loads and djConfig.afterOnLoad is true. Supports the same\narguments as dojo/ready. When using a function reference, use\n<code>djConfig.addOnLoad = function(){};</code>. For object with function name use\n<code>djConfig.addOnLoad = [myObject, &quot;functionName&quot;];</code> and for object with\nfunction reference use\n<code>djConfig.addOnLoad = [myObject, function(){}];</code></p>\n"
   },
   "afterOnLoad": {},
   "baseUrl": {},
   "callback": {
    "!doc": "<p>Defines a callback to be used when dependencies are defined before \nthe loader has been loaded. When provided, they cause the loader to \nexecute require(deps, callback) once it has finished loading. \nShould be used with deps.</p>\n"
   },
   "debugContainerId": {},
   "debugHeight": {},
   "defaultDuration": {
    "!doc": "<p>Default duration, in milliseconds, for wipe and fade animations within dijits.\nAssigned to dijit.defaultDuration.</p>\n"
   },
   "deferredInstrumentation": {
    "!doc": "<p>Whether deferred instrumentation should be loaded or included\nin builds.</p>\n"
   },
   "deps": {
    "!doc": "<p>Defines dependencies to be used before the loader has been loaded.\nWhen provided, they cause the loader to execute require(deps, callback) \nonce it has finished loading. Should be used with callback.</p>\n"
   },
   "dojoBlankHtmlUrl": {
    "!doc": "<p>Used by some modules to configure an empty iframe. Used by dojo/io/iframe and\ndojo/back, and dijit/popup support in IE where an iframe is needed to make sure native\ncontrols do not bleed through the popups. Normally this configuration variable\ndoes not need to be set, except when using cross-domain/CDN Dojo builds.\nSave dojo/resources/blank.html to your domain and set <code>djConfig.dojoBlankHtmlUrl</code>\nto the path on your domain your copy of blank.html.</p>\n"
   },
   "extraLocale": {
    "!doc": "<p>No default value. Specifies additional locales whose\nresources should also be loaded alongside the default locale when\ncalls to <code>dojo.requireLocalization()</code> are processed.</p>\n"
   },
   "ioPublish": {
    "!doc": "<p>Set this to true to enable publishing of topics for the different phases of\nIO operations. Publishing is done via dojo/topic.publish(). See dojo/main.__IoPublish for a list\nof topics that are published.</p>\n"
   },
   "isDebug": {
    "!doc": "<p>Defaults to <code>false</code>. If set to <code>true</code>, ensures that Dojo provides\nextended debugging feedback via Firebug. If Firebug is not available\non your platform, setting <code>isDebug</code> to <code>true</code> will force Dojo to\npull in (and display) the version of Firebug Lite which is\nintegrated into the Dojo distribution, thereby always providing a\ndebugging/logging console when <code>isDebug</code> is enabled. Note that\nFirebug&#39;s <code>console.*</code> methods are ALWAYS defined by Dojo. If\n<code>isDebug</code> is false and you are on a platform without Firebug, these\nmethods will be defined as no-ops.</p>\n"
   },
   "locale": {
    "!doc": "<p>The locale to assume for loading localized resources in this page,\nspecified according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a>.\nMust be specified entirely in lowercase, e.g. <code>en-us</code> and <code>zh-cn</code>.\nSee the documentation for <code>dojo.i18n</code> and <code>dojo.requireLocalization</code>\nfor details on loading localized resources. If no locale is specified,\nDojo assumes the locale of the user agent, according to <code>navigator.userLanguage</code>\nor <code>navigator.language</code> properties.</p>\n"
   },
   "modulePaths": {
    "!doc": "<p>A map of module names to paths relative to <code>dojo.baseUrl</code>. The\nkey/value pairs correspond directly to the arguments which\n<code>dojo.registerModulePath</code> accepts. Specifying\n<code>djConfig.modulePaths = { &quot;foo&quot;: &quot;../../bar&quot; }</code> is the equivalent\nof calling <code>dojo.registerModulePath(&quot;foo&quot;, &quot;../../bar&quot;);</code>. Multiple\nmodules may be configured via <code>djConfig.modulePaths</code>.</p>\n"
   },
   "parseOnLoad": {
    "!doc": "<p>Run the parser after the page is loaded</p>\n"
   },
   "require": {
    "!doc": "<p>An array of module names to be loaded immediately after dojo.js has been included\nin a page.</p>\n"
   },
   "transparentColor": {
    "!doc": "<p>Array containing the r, g, b components used as transparent color in dojo.Color;\nif undefined, [255,255,255] (white) will be used.</p>\n"
   },
   "urchin": {
    "!doc": "<p>Used by <code>dojox.analytics.Urchin</code> as the default UA-123456-7 account\nnumber used when being created. Alternately, you can pass an acct:&quot;&quot;\nparameter to the constructor a la: new dojox.analytics.Urchin({ acct:&quot;UA-123456-7&quot; });</p>\n"
   },
   "useCustomLogger": {
    "!doc": "<p>If set to a value that evaluates to true such as a string or array and\nisDebug is true and Firebug is not available or running, then it bypasses\nthe creation of Firebug Lite allowing you to define your own console object.</p>\n"
   },
   "useDeferredInstrumentation": {
    "!doc": "<p>Whether the deferred instrumentation should be used.\n\n</p>\n<ul>\n<li><code>&quot;report-rejections&quot;</code>: report each rejection as it occurs.</li>\n<li><code>true</code> or <code>1</code> or <code>&quot;report-unhandled-rejections&quot;</code>: wait 1 second\n  in an attempt to detect unhandled rejections.</li>\n</ul>\n"
   }
  },
  "dojo/_base/kernel.global": {
   "undefined_callbacks": {},
   "$": {
    "!type": "fn()"
   },
   "GoogleSearchStoreCallback_undefined_NaN": {
    "!type": "fn(start: ?, data: ?, responseCode: ?, errorMsg: ?)"
   },
   "jQuery": {
    "!type": "fn()"
   },
   "swfIsInHTML": {
    "!type": "fn()"
   },
   "undefined_onload": {
    "!type": "fn()"
   }
  },
  "dojo/_base/kernel.dijit": {
   "_editor": {},
   "_tree": {},
   "form": {},
   "layout": {},
   "range": {
    "!doc": "<p>W3C range API</p>\n"
   },
   "registry": {},
   "tree": {},
   "byId": {
    "!type": "fn(id: ?)"
   },
   "Calendar": {
    "!type": "fn()"
   },
   "CalendarLite": {
    "!type": "fn()"
   },
   "CheckedMenuItem": {
    "!type": "fn()"
   },
   "ColorPalette": {
    "!type": "fn()"
   },
   "Declaration": {
    "!type": "fn()"
   },
   "Destroyable": {
    "!type": "fn()"
   },
   "Dialog": {
    "!type": "fn()"
   },
   "DialogUnderlay": {
    "!type": "fn()"
   },
   "DropDownMenu": {
    "!type": "fn()"
   },
   "Dye": {
    "!type": "fn()"
   },
   "Editor": {
    "!type": "fn()"
   },
   "Fieldset": {
    "!type": "fn()"
   },
   "InlineEditBox": {
    "!type": "fn()"
   },
   "Menu": {
    "!type": "fn()"
   },
   "MenuBar": {
    "!type": "fn()"
   },
   "MenuBarItem": {
    "!type": "fn()"
   },
   "MenuItem": {
    "!type": "fn()"
   },
   "MenuSeparator": {
    "!type": "fn()"
   },
   "PopupMenuBarItem": {
    "!type": "fn()"
   },
   "PopupMenuItem": {
    "!type": "fn()"
   },
   "ProgressBar": {
    "!type": "fn()"
   },
   "RadioButtonMenuItem": {
    "!type": "fn()"
   },
   "TitlePane": {
    "!type": "fn()"
   },
   "Toolbar": {
    "!type": "fn()"
   },
   "ToolbarSeparator": {
    "!type": "fn()"
   },
   "Tooltip": {
    "!type": "fn()"
   },
   "TooltipDialog": {
    "!type": "fn()"
   },
   "Tree": {
    "!type": "fn()"
   },
   "WidgetSet": {
    "!type": "fn()"
   }
  },
  "dojo/_base/kernel.dojox": {
   "analytics": {},
   "app": {},
   "atom": {},
   "av": {},
   "BidiComplex": {},
   "calc": {},
   "calendar": {},
   "charting": {},
   "collections": {},
   "color": {},
   "css3": {},
   "data": {},
   "date": {},
   "dgauges": {},
   "dnd": {},
   "drawing": {},
   "dtl": {},
   "editor": {},
   "embed": {},
   "encoding": {},
   "enhanced": {},
   "flash": {
    "!doc": "<p>Utilities to embed and communicate with the Flash player from Javascript</p>\n"
   },
   "form": {},
   "fx": {},
   "gantt": {},
   "gauges": {},
   "geo": {},
   "gesture": {},
   "gfx": {},
   "gfx3d": {},
   "grid": {},
   "help": {
    "!doc": "<p>Adds the help function to all variables.</p>\n"
   },
   "highlight": {},
   "html": {},
   "image": {},
   "io": {},
   "jq": {},
   "json": {},
   "jsonPath": {},
   "lang": {},
   "layout": {},
   "math": {},
   "mdnd": {},
   "mobile": {},
   "mvc": {},
   "openlayers": {},
   "rails": {},
   "robot": {},
   "rpc": {},
   "secure": {},
   "sketch": {},
   "sql": {},
   "storage": {},
   "string": {},
   "testing": {},
   "timing": {},
   "treemap": {},
   "uuid": {},
   "validate": {},
   "widget": {},
   "xml": {},
   "sprintf": {
    "!type": "fn(format: string, filler: +mixed...)"
   }
  },
  "dojo/_base/kernel.scopeMap": {
   "dijit": {},
   "dojo": {},
   "dojox": {}
  },
  "dojo/_base/kernel.version": {
   "flag": {},
   "major": {},
   "minor": {},
   "patch": {},
   "revision": {},
   "toString": {
    "!type": "fn() -> string"
   }
  },
  "dojo/_base/kernel._hasResource": {},
  "dojo/_base/kernel.doc": {
   "documentElement": {},
   "dojoClick": {}
  },
  "dojo/_base/kernel.mouseButtons": {
   "LEFT": {
    "!doc": "<p>Numeric value of the left mouse button for the platform.</p>\n"
   },
   "MIDDLE": {
    "!doc": "<p>Numeric value of the middle mouse button for the platform.</p>\n"
   },
   "RIGHT": {
    "!doc": "<p>Numeric value of the right mouse button for the platform.</p>\n"
   },
   "isButton": {
    "!type": "fn(e: +Event, button: number) -> bool",
    "!doc": "<p>Checks an event object for a pressed button</p>\n"
   },
   "isLeft": {
    "!type": "fn(e: +Event) -> bool",
    "!doc": "<p>Checks an event object for the pressed left button</p>\n"
   },
   "isMiddle": {
    "!type": "fn(e: +Event) -> bool",
    "!doc": "<p>Checks an event object for the pressed middle button</p>\n"
   },
   "isRight": {
    "!type": "fn(e: +Event) -> bool",
    "!doc": "<p>Checks an event object for the pressed right button</p>\n"
   }
  },
  "dojo/_base/kernel.keys": {
   "ALT": {},
   "BACKSPACE": {},
   "CAPS_LOCK": {},
   "CLEAR": {},
   "copyKey": {},
   "CTRL": {},
   "DELETE": {},
   "DOWN_ARROW": {},
   "DOWN_DPAD": {},
   "END": {},
   "ENTER": {},
   "ESCAPE": {},
   "F1": {},
   "F10": {},
   "F11": {},
   "F12": {},
   "F13": {},
   "F14": {},
   "F15": {},
   "F2": {},
   "F3": {},
   "F4": {},
   "F5": {},
   "F6": {},
   "F7": {},
   "F8": {},
   "F9": {},
   "HELP": {},
   "HOME": {},
   "INSERT": {},
   "LEFT_ARROW": {},
   "LEFT_DPAD": {},
   "LEFT_WINDOW": {},
   "META": {},
   "NUM_LOCK": {},
   "NUMPAD_0": {},
   "NUMPAD_1": {},
   "NUMPAD_2": {},
   "NUMPAD_3": {},
   "NUMPAD_4": {},
   "NUMPAD_5": {},
   "NUMPAD_6": {},
   "NUMPAD_7": {},
   "NUMPAD_8": {},
   "NUMPAD_9": {},
   "NUMPAD_DIVIDE": {},
   "NUMPAD_ENTER": {},
   "NUMPAD_MINUS": {},
   "NUMPAD_MULTIPLY": {},
   "NUMPAD_PERIOD": {},
   "NUMPAD_PLUS": {},
   "PAGE_DOWN": {},
   "PAGE_UP": {},
   "PAUSE": {},
   "RIGHT_ARROW": {},
   "RIGHT_DPAD": {},
   "RIGHT_WINDOW": {},
   "SCROLL_LOCK": {},
   "SELECT": {},
   "SHIFT": {},
   "SPACE": {},
   "TAB": {},
   "UP_ARROW": {},
   "UP_DPAD": {}
  },
  "dojo/_base/kernel._nodeDataCache": {},
  "dojo/_base/kernel.fx": {
   "easing": {
    "!doc": "<p>Collection of easing functions to use beyond the default\n<code>dojo._defaultEasing</code> function.</p>\n"
   },
   "chain": {
    "!type": "fn(animations: +dojo)",
    "!doc": "<p>Chain a list of <code>dojo/_base/fx.Animation</code>s to run in sequence\n</p>\n"
   },
   "combine": {
    "!type": "fn(animations: +dojo)",
    "!doc": "<p>Combine a list of <code>dojo/_base/fx.Animation</code>s to run in parallel\n</p>\n"
   },
   "slideTo": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Slide a node to a new top/left position\n</p>\n"
   },
   "Toggler": {
    "!type": "fn()"
   },
   "wipeIn": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Expand a node to it&#39;s natural height.\n</p>\n"
   },
   "wipeOut": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Shrink a node to nothing and hide it.\n</p>\n"
   }
  },
  "dojo/_base/kernel.date": {
   "stamp": {
    "!doc": "<p>TODOC</p>\n"
   },
   "add": {
    "!type": "fn(date: +Date, interval: string, amount: number)",
    "!doc": "<p>Add to a Date in intervals of different size, from milliseconds to years</p>\n"
   },
   "compare": {
    "!type": "fn(date1: +Date, date2?: +Date, portion?: string) -> number",
    "!doc": "<p>Compare two date objects by date, time, or both.</p>\n"
   },
   "difference": {
    "!type": "fn(date1: +Date, date2?: +Date, interval?: string)",
    "!doc": "<p>Get the difference in a specific unit of time (e.g., number of\nmonths, weeks, days, etc.) between two dates, rounded to the\nnearest integer.</p>\n"
   },
   "getDaysInMonth": {
    "!type": "fn(dateObject: +Date) -> number",
    "!doc": "<p>Returns the number of days in the month used by dateObject</p>\n"
   },
   "getTimezoneName": {
    "!type": "fn(dateObject: +Date)",
    "!doc": "<p>Get the user&#39;s time zone as provided by the browser</p>\n"
   },
   "isLeapYear": {
    "!type": "fn(dateObject: +Date) -> bool",
    "!doc": "<p>Determines if the year of the dateObject is a leap year</p>\n"
   }
  },
  "dojo/_base/kernel.html": {
   "set": {
    "!type": "fn(node: +Node, cont: string|+Node|+NodeList, params?: ?)",
    "!doc": "<p>inserts (replaces) the given content into the given node. dojo/dom-construct.place(cont, node, &quot;only&quot;)\nmay be a better choice for simple HTML insertion.</p>\n"
   }
  },
  "dojo/_base/kernel.Stateful": {
   "!type": "fn()",
   "prototype": {
    "_attrPairNames": {
     "!doc": "<p>Used across all instances a hash to cache attribute names and their getter \nand setter names.</p>\n"
    },
    "get": {
     "!type": "fn(name: string)",
     "!doc": "<p>Get a property on a Stateful instance.</p>\n"
    },
    "postscript": {
     "!type": "fn(params?: ?)"
    },
    "set": {
     "!type": "fn(name: string, value: ?) -> fn()",
     "!doc": "<p>Set a property on a Stateful instance</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dojo/_base/kernel.contentHandlers": {
   "auto": {
    "!type": "fn(xhr: ?)"
   },
   "javascript": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which evaluates the response data, expecting it to be valid JavaScript</p>\n"
   },
   "json": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which returns a JavaScript object created from the response data</p>\n"
   },
   "json-comment-filtered": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which expects comment-filtered JSON.</p>\n"
   },
   "json-comment-optional": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which checks the presence of comment-filtered JSON and\nalternates between the <code>json</code> and <code>json-comment-filtered</code> contentHandlers.</p>\n"
   },
   "text": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which simply returns the plaintext response data</p>\n"
   },
   "xml": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler returning an XML Document parsed from the response data</p>\n"
   }
  },
  "dojo/_base/kernel._contentHandlers": {
   "auto": {
    "!type": "fn(xhr: ?)"
   },
   "javascript": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which evaluates the response data, expecting it to be valid JavaScript</p>\n"
   },
   "json": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which returns a JavaScript object created from the response data</p>\n"
   },
   "json-comment-filtered": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which expects comment-filtered JSON.</p>\n"
   },
   "json-comment-optional": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which checks the presence of comment-filtered JSON and\nalternates between the <code>json</code> and <code>json-comment-filtered</code> contentHandlers.</p>\n"
   },
   "text": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which simply returns the plaintext response data</p>\n"
   },
   "xml": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler returning an XML Document parsed from the response data</p>\n"
   }
  },
  "dojo/_base/kernel.__IoArgs": {
   "!type": "fn()",
   "prototype": {
    "content": {
     "!doc": "<p>Contains properties with string values. These\nproperties will be serialized as name1=value2 and\npassed in the request.</p>\n"
    },
    "form": {
     "!doc": "<p>DOM node for a form. Used to extract the form values\nand send to the server.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>Acceptable values depend on the type of IO\ntransport (see specific IO calls for more information).</p>\n"
    },
    "ioPublish": {
     "!doc": "<p>Set this explicitly to false to prevent publishing of topics related to\nIO operations. Otherwise, if djConfig.ioPublish is set to true, topics\nwill be published via dojo/topic.publish() for different phases of an IO operation.\nSee dojo/main.__IoPublish for a list of topics that are published.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Default is false. If true, then a\n&quot;dojo.preventCache&quot; parameter is sent in the request\nwith a value that changes with each request\n(timestamp). Useful only with GET-type requests.</p>\n"
    },
    "rawBody": {
     "!doc": "<p>Sets the raw body for an HTTP request. If this is used, then the content\nproperty is ignored. This is mostly useful for HTTP methods that have\na body to their requests, like PUT or POST. This property can be used instead\nof postData and putData for dojo/_base/xhr.rawXhrPost and dojo/_base/xhr.rawXhrPut respectively.</p>\n"
    },
    "timeout": {
     "!doc": "<p>Milliseconds to wait for the response. If this time\npasses, the then error callbacks are called.</p>\n"
    },
    "url": {
     "!doc": "<p>URL to server endpoint.</p>\n"
    },
    "error": {
     "!type": "fn(response: ?, ioArgs: +dojo)",
     "!doc": "<p>This function will\nbe called when the request fails due to a network or server error, the url\nis invalid, etc. It will also be called if the load or handle callback throws an\nexception, unless djConfig.debugAtAllCosts is true.  This allows deployed applications\nto continue to run even when a logic error happens in the callback, while making\nit easier to troubleshoot while in debug mode.</p>\n"
    },
    "handle": {
     "!type": "fn(loadOrError: string, response: ?, ioArgs: +dojo)",
     "!doc": "<p>This function will\nbe called at the end of every request, whether or not an error occurs.</p>\n"
    },
    "load": {
     "!type": "fn(response: ?, ioArgs: +dojo)",
     "!doc": "<p>This function will be\ncalled on a successful HTTP response code.</p>\n"
    }
   }
  },
  "dojo/_base/kernel.__IoCallbackArgs": {
   "!type": "fn()",
   "prototype": {
    "args": {
     "!doc": "<p>the original object argument to the IO call.</p>\n"
    },
    "canDelete": {
     "!doc": "<p>For dojo/io/script calls only, indicates\nwhether the script tag that represents the\nrequest can be deleted after callbacks have\nbeen called. Used internally to know when\ncleanup can happen on JSONP-type requests.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>The final indicator on how the response will be\nhandled.</p>\n"
    },
    "id": {
     "!doc": "<p>For dojo/io/script calls only, the internal\nscript ID used for the request.</p>\n"
    },
    "json": {
     "!doc": "<p>For dojo/io/script calls only: holds the JSON\nresponse for JSONP-type requests. Used\ninternally to hold on to the JSON responses.\nYou should not need to access it directly --\nthe same object should be passed to the success\ncallbacks directly.</p>\n"
    },
    "query": {
     "!doc": "<p>For non-GET requests, the\nname1=value1&amp;name2=value2 parameters sent up in\nthe request.</p>\n"
    },
    "url": {
     "!doc": "<p>The final URL used for the call. Many times it\nwill be different than the original args.url\nvalue.</p>\n"
    },
    "xhr": {
     "!doc": "<p>For XMLHttpRequest calls only, the\nXMLHttpRequest object that was used for the\nrequest.</p>\n"
    }
   }
  },
  "dojo/_base/kernel.__IoPublish": {
   "!type": "fn()",
   "prototype": {
    "done": {
     "!doc": "<p>&quot;/dojo/io/done&quot; is sent whenever an IO request has completed,\neither by loading or by erroring. It passes the error and\nthe dojo.Deferred for the request with the topic.</p>\n"
    },
    "error": {
     "!doc": "<p>&quot;/dojo/io/error&quot; is sent whenever an IO request has errored.\nIt passes the error and the dojo.Deferred\nfor the request with the topic.</p>\n"
    },
    "load": {
     "!doc": "<p>&quot;/dojo/io/load&quot; is sent whenever an IO request has loaded\nsuccessfully. It passes the response and the dojo.Deferred\nfor the request with the topic.</p>\n"
    },
    "send": {
     "!doc": "<p>&quot;/dojo/io/send&quot; is sent whenever a new IO request is started.\nIt passes the dojo.Deferred for the request with the topic.</p>\n"
    },
    "start": {
     "!doc": "<p>&quot;/dojo/io/start&quot; is sent when there are no outstanding IO\nrequests, and a new IO request is started. No arguments\nare passed with this topic.</p>\n"
    },
    "stop": {
     "!doc": "<p>&quot;/dojo/io/stop&quot; is sent when all outstanding IO requests have\nfinished. No arguments are passed with this topic.</p>\n"
    }
   }
  },
  "dojo/_base/kernel.__XhrArgs": {
   "!type": "fn()",
   "prototype": {
    "content": {
     "!doc": "<p>Contains properties with string values. These\nproperties will be serialized as name1=value2 and\npassed in the request.</p>\n"
    },
    "contentType": {
     "!doc": "<p>&quot;application/x-www-form-urlencoded&quot; is default. Set to false to\nprevent a Content-Type header from being sent, or to a string\nto send a different Content-Type.</p>\n"
    },
    "failOk": {
     "!doc": "<p>false is default. Indicates whether a request should be\nallowed to fail (and therefore no console error message in\nthe event of a failure)</p>\n"
    },
    "form": {
     "!doc": "<p>DOM node for a form. Used to extract the form values\nand send to the server.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>Acceptable values are: text (default), json, json-comment-optional,\njson-comment-filtered, javascript, xml. See <code>dojo/_base/xhr.contentHandlers</code></p>\n"
    },
    "headers": {
     "!doc": "<p>Additional HTTP headers to send in the request.</p>\n"
    },
    "ioPublish": {
     "!doc": "<p>Set this explicitly to false to prevent publishing of topics related to\nIO operations. Otherwise, if djConfig.ioPublish is set to true, topics\nwill be published via dojo/topic.publish() for different phases of an IO operation.\nSee dojo/main.__IoPublish for a list of topics that are published.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Default is false. If true, then a\n&quot;dojo.preventCache&quot; parameter is sent in the request\nwith a value that changes with each request\n(timestamp). Useful only with GET-type requests.</p>\n"
    },
    "rawBody": {
     "!doc": "<p>Sets the raw body for an HTTP request. If this is used, then the content\nproperty is ignored. This is mostly useful for HTTP methods that have\na body to their requests, like PUT or POST. This property can be used instead\nof postData and putData for dojo/_base/xhr.rawXhrPost and dojo/_base/xhr.rawXhrPut respectively.</p>\n"
    },
    "sync": {
     "!doc": "<p>false is default. Indicates whether the request should\nbe a synchronous (blocking) request.</p>\n"
    },
    "timeout": {
     "!doc": "<p>Milliseconds to wait for the response. If this time\npasses, the then error callbacks are called.</p>\n"
    },
    "url": {
     "!doc": "<p>URL to server endpoint.</p>\n"
    },
    "error": {
     "!type": "fn(response: ?, ioArgs: +dojo)",
     "!doc": "<p>This function will\nbe called when the request fails due to a network or server error, the url\nis invalid, etc. It will also be called if the load or handle callback throws an\nexception, unless djConfig.debugAtAllCosts is true.  This allows deployed applications\nto continue to run even when a logic error happens in the callback, while making\nit easier to troubleshoot while in debug mode.</p>\n"
    },
    "handle": {
     "!type": "fn(loadOrError: string, response: ?, ioArgs: +dojo)",
     "!doc": "<p>This function will\nbe called at the end of every request, whether or not an error occurs.</p>\n"
    },
    "load": {
     "!type": "fn(response: ?, ioArgs: +dojo)",
     "!doc": "<p>This function will be\ncalled on a successful HTTP response code.</p>\n"
    }
   }
  },
  "dojo/_base/kernel.back": {
   "addToHistory": {
    "!type": "fn(args: ?)",
    "!doc": "<p>adds a state object (args) to the history list.</p>\n"
   },
   "getHash": {
    "!type": "fn()"
   },
   "goBack": {
    "!type": "fn()",
    "!doc": "<p>private method. Do not call this directly.</p>\n"
   },
   "goForward": {
    "!type": "fn()",
    "!doc": "<p>private method. Do not call this directly.</p>\n"
   },
   "init": {
    "!type": "fn()",
    "!doc": "<p>Initializes the undo stack. This must be called from a <script>\nblock that lives inside the <code>&lt;body&gt;</code> tag to prevent bugs on IE.\n\n</p>\n<p>Only call this method before the page&#39;s DOM is finished loading. Otherwise\nit will not work. Be careful with xdomain loading or djConfig.debugAtAllCosts scenarios,\nin order for this method to work, dojo/back will need to be part of a build layer.</p>\n"
   },
   "setHash": {
    "!type": "fn(h: ?)"
   },
   "setInitialState": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sets the state object and back callback for the very first page\nthat is loaded.\n\n</p>\n<p>It is recommended that you call this method as part of an event\nlistener that is registered via dojo/ready.</p>\n"
   }
  },
  "dojo/_base/kernel.cldr": {
   "monetary": {
    "!doc": "<p>TODOC</p>\n"
   },
   "supplemental": {
    "!doc": "<p>TODOC</p>\n"
   }
  },
  "dojo/_base/kernel.i18n": {
   "cache": {},
   "dynamic": {},
   "unitTests": {},
   "getL10nName": {
    "!type": "fn(moduleName: ?, bundleName: ?, locale: ?) -> string"
   },
   "getLocalization": {
    "!type": "fn(moduleName: ?, bundleName: ?, locale: ?)"
   },
   "load": {
    "!type": "fn(id: ?, require: ?, load: ?)",
    "!doc": "<p>id is in one of the following formats\n\n</p>\n<ol>\n<li><p><path>/nls/<bundle>\n =&gt; load the bundle, localized to config.locale; load all bundles localized to\n config.extraLocale (if any); return the loaded bundle localized to config.locale.</p>\n</li>\n<li><p><path>/nls/<locale>/<bundle>\n =&gt; load then return the bundle localized to <locale></p>\n</li>\n<li><p><em>preload</em><path>/nls/<module>*<JSON array of available locales>\n =&gt; for config.locale and all config.extraLocale, load all bundles found\n in the best-matching bundle rollup. A value of 1 is returned, which\n is meaningless other than to say the plugin is executing the requested\n preloads</p>\n</li>\n</ol>\n<p>In cases 1 and 2, <path> is always normalized to an absolute module id upon entry; see\nnormalize. In case 3, it <path> is assumed to be absolute; this is arranged by the builder.\n\n</p>\n<p>To load a bundle means to insert the bundle into the plugin&#39;s cache and publish the bundle\nvalue to the loader. Given <path>, <bundle>, and a particular <locale>, the cache key\n\n</p>\n<pre><code>&lt;path&gt;/nls/&lt;bundle&gt;/&lt;locale&gt;</code></pre>\n<p>will hold the value. Similarly, then plugin will publish this value to the loader by\n\n</p>\n<pre><code>define(&quot;&lt;path&gt;/nls/&lt;bundle&gt;/&lt;locale&gt;&quot;, &lt;bundle-value&gt;);</code></pre>\n<p>Given this algorithm, other machinery can provide fast load paths be preplacing\nvalues in the plugin&#39;s cache, which is public. When a load is demanded the\ncache is inspected before starting any loading. Explicitly placing values in the plugin\ncache is an advanced/experimental feature that should not be needed; use at your own risk.\n\n</p>\n<p>For the normal AMD algorithm, the root bundle is loaded first, which instructs the\nplugin what additional localized bundles are required for a particular locale. These\nadditional locales are loaded and a mix of the root and each progressively-specific\nlocale is returned. For example:\n\n</p>\n<ol>\n<li><p>The client demands &quot;dojo/i18n!some/path/nls/someBundle</p>\n</li>\n<li><p>The loader demands load(some/path/nls/someBundle)</p>\n</li>\n<li><p>This plugin require&#39;s &quot;some/path/nls/someBundle&quot;, which is the root bundle.</p>\n</li>\n<li><p>Assuming config.locale is &quot;ab-cd-ef&quot; and the root bundle indicates that localizations\nare available for &quot;ab&quot; and &quot;ab-cd-ef&quot; (note the missing &quot;ab-cd&quot;, then the plugin\nrequires &quot;some/path/nls/ab/someBundle&quot; and &quot;some/path/nls/ab-cd-ef/someBundle&quot;</p>\n</li>\n<li><p>Upon receiving all required bundles, the plugin constructs the value of the bundle\nab-cd-ef as...</p>\n<pre><code> mixin(mixin(mixin({}, require(&quot;some/path/nls/someBundle&quot;),\n     require(&quot;some/path/nls/ab/someBundle&quot;)),\n     require(&quot;some/path/nls/ab-cd-ef/someBundle&quot;));</code></pre>\n</li>\n</ol>\n<p>This value is inserted into the cache and published to the loader at the\nkey/module-id some/path/nls/someBundle/ab-cd-ef.\n\n</p>\n<p>The special preload signature (case 3) instructs the plugin to stop servicing all normal requests\n(further preload requests will be serviced) until all ongoing preloading has completed.\n\n</p>\n<p>The preload signature instructs the plugin that a special rollup module is available that contains\none or more flattened, localized bundles. The JSON array of available locales indicates which locales\nare available. Here is an example:\n\n</p>\n<pre><code>*preload*some/path/nls/someModule*[&quot;root&quot;, &quot;ab&quot;, &quot;ab-cd-ef&quot;]</code></pre>\n<p>This indicates the following rollup modules are available:\n\n</p>\n<pre><code>some/path/nls/someModule_ROOT\nsome/path/nls/someModule_ab\nsome/path/nls/someModule_ab-cd-ef</code></pre>\n<p>Each of these modules is a normal AMD module that contains one or more flattened bundles in a hash.\nFor example, assume someModule contained the bundles some/bundle/path/someBundle and\nsome/bundle/path/someOtherBundle, then some/path/nls/someModule_ab would be expressed as follows:\n\n</p>\n<pre><code>define({\n    some/bundle/path/someBundle:&lt;value of someBundle, flattened with respect to locale ab&gt;,\n    some/bundle/path/someOtherBundle:&lt;value of someOtherBundle, flattened with respect to locale ab&gt;,\n});</code></pre>\n<p>E.g., given this design, preloading for locale==&quot;ab&quot; can execute the following algorithm:\n\n</p>\n<pre><code>require([&quot;some/path/nls/someModule_ab&quot;], function(rollup){\n    for(var p in rollup){\n        var id = p + &quot;/ab&quot;,\n        cache[id] = rollup[p];\n        define(id, rollup[p]);\n    }\n});</code></pre>\n<p>Similarly, if &quot;ab-cd&quot; is requested, the algorithm can determine that &quot;ab&quot; is the best available and\nload accordingly.\n\n</p>\n<p>The builder will write such rollups for every layer if a non-empty localeList  profile property is\nprovided. Further, the builder will include the following cache entry in the cache associated with\nany layer.\n\n</p>\n<pre><code>&quot;*now&quot;:function(r){r([&#39;dojo/i18n!*preload*&lt;path&gt;/nls/&lt;module&gt;*&lt;JSON array of available locales&gt;&#39;]);}</code></pre>\n<p>The *now special cache module instructs the loader to apply the provided function to context-require\nwith respect to the particular layer being defined. This causes the plugin to hold all normal service\nrequests until all preloading is complete.\n\n</p>\n<p>Notice that this algorithm is rarely better than the standard AMD load algorithm. Consider the normal case\nwhere the target locale has a single segment and a layer depends on a single bundle:\n\n</p>\n<p>Without Preloads:\n\n</p>\n<ol>\n<li>Layer loads root bundle.</li>\n<li>bundle is demanded; plugin loads single localized bundle.</li>\n</ol>\n<p>With Preloads:\n\n</p>\n<ol>\n<li>Layer causes preloading of target bundle.</li>\n<li>bundle is demanded; service is delayed until preloading complete; bundle is returned.</li>\n</ol>\n<p>In each case a single transaction is required to load the target bundle. In cases where multiple bundles\nare required and/or the locale has multiple segments, preloads still requires a single transaction whereas\nthe normal path requires an additional transaction for each additional bundle/locale-segment. However all\nof these additional transactions can be done concurrently. Owing to this analysis, the entire preloading\nalgorithm can be discard during a build by setting the has feature dojo-preload-i18n-Api to false.</p>\n"
   },
   "normalize": {
    "!type": "fn(id: ?, toAbsMid: ?)",
    "!doc": "<p>id may be relative.\npreload has form <code>*preload*&lt;path&gt;/nls/&lt;module&gt;*&lt;flattened locales&gt;</code> and\ntherefore never looks like a relative</p>\n"
   },
   "normalizeLocale": {
    "!type": "fn(locale: ?)"
   }
  },
  "dojo/_base/kernel.colors": {
   "makeGrey": {
    "!type": "fn(g: number, a?: number)",
    "!doc": "<p>creates a greyscale color with an optional alpha</p>\n"
   }
  },
  "dojo/_base/kernel.regexp": {
   "buildGroupRE": {
    "!type": "fn(arr: ?|[], re: fn(), nonCapture?: bool)",
    "!doc": "<p>Builds a regular expression that groups subexpressions</p>\n"
   },
   "escapeString": {
    "!type": "fn(str: string, except?: string)",
    "!doc": "<p>Adds escape sequences for special characters in regular expressions</p>\n"
   },
   "group": {
    "!type": "fn(expression: string, nonCapture?: bool) -> string",
    "!doc": "<p>adds group match to expression</p>\n"
   }
  },
  "dojo/_base/kernel.string": {
   "escape": {
    "!type": "fn(str: string) -> string",
    "!doc": "<p>Efficiently escape a string for insertion into HTML (innerHTML or attributes), replacing &amp;, &lt;, &gt;, &quot;, &#39;, and / characters.</p>\n"
   },
   "pad": {
    "!type": "fn(text: string, size: +Integer, ch?: string, end?: bool) -> number",
    "!doc": "<p>Pad a string to guarantee that it is at least <code>size</code> length by\nfilling with the character <code>ch</code> at either the start or end of the\nstring. Pads at the start, by default.</p>\n"
   },
   "rep": {
    "!type": "fn(str: string, num: +Integer) -> string",
    "!doc": "<p>Efficiently replicate a string <code>n</code> times.</p>\n"
   },
   "substitute": {
    "!type": "fn(template: string, map: ?|[], transform?: fn(), thisObject?: ?)",
    "!doc": "<p>Performs parameterized substitutions on a string. Throws an\nexception if any parameter is unmatched.</p>\n"
   },
   "trim": {
    "!type": "fn(str: string) -> string",
    "!doc": "<p>Trims whitespace from both sides of the string</p>\n"
   }
  },
  "dojo/_base/kernel.number": {
   "_numberPatternRE": {},
   "format": {
    "!type": "fn(value: number, options?: ?)",
    "!doc": "<p>Format a Number as a String, using locale-specific settings</p>\n"
   },
   "parse": {
    "!type": "fn(expression: string, options?: ?) -> number",
    "!doc": "<p>Convert a properly formatted string to a primitive Number, using\nlocale-specific settings.</p>\n"
   },
   "regexp": {
    "!type": "fn(options?: ?)",
    "!doc": "<p>Builds the regular needed to parse a number</p>\n"
   },
   "round": {
    "!type": "fn(value: number, places?: number, increment?: number) -> number",
    "!doc": "<p>Rounds to the nearest value with the given number of decimal places, away from zero</p>\n"
   }
  },
  "dojo/_base/kernel.currency": {
   "format": {
    "!type": "fn(value: number, options?: +__FormatOptions)",
    "!doc": "<p>Format a Number as a currency, using locale-specific settings\n</p>\n"
   },
   "parse": {
    "!type": "fn(expression: string, options?: ?)"
   },
   "regexp": {
    "!type": "fn(options?: ?)"
   }
  },
  "dojo/_base/kernel.data": {
   "api": {},
   "util": {},
   "ItemFileReadStore": {
    "!type": "fn()"
   },
   "ItemFileWriteStore": {
    "!type": "fn()"
   },
   "ObjectStore": {
    "!type": "fn()"
   }
  },
  "dojo/_base/kernel.dnd": {
   "autoscroll": {
    "!doc": "<p>Used by dojo/dnd/Manager to scroll document or internal node when the user\ndrags near the edge of the viewport or a scrollable node</p>\n"
   },
   "move": {},
   "AutoSource": {
    "!type": "fn()"
   },
   "Avatar": {
    "!type": "fn()"
   },
   "Container": {
    "!type": "fn()"
   },
   "Manager": {
    "!type": "fn()"
   },
   "Moveable": {
    "!type": "fn()"
   },
   "Mover": {
    "!type": "fn()"
   },
   "Selector": {
    "!type": "fn()"
   },
   "Source": {
    "!type": "fn()"
   },
   "Target": {
    "!type": "fn()"
   },
   "TimedMoveable": {
    "!type": "fn()"
   }
  },
  "dojo/_base/kernel.touch": {
   "cancel": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to &#39;touchcancel&#39;|&#39;mouseleave&#39; for the given node</p>\n"
   },
   "enter": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to mouse.enter or touch equivalent for the given node</p>\n"
   },
   "leave": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to mouse.leave or touch equivalent for the given node</p>\n"
   },
   "move": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener that fires when the mouse cursor or a finger is dragged over the given node.</p>\n"
   },
   "out": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to &#39;mouseout&#39; or touch equivalent for the given node</p>\n"
   },
   "over": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to &#39;mouseover&#39; or touch equivalent for the given node</p>\n"
   },
   "press": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to &#39;touchstart&#39;|&#39;mousedown&#39; for the given node</p>\n"
   },
   "release": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to releasing the mouse button while the cursor is over the given node\n(i.e. &quot;mouseup&quot;) or for removing the finger from the screen while touching the given node.</p>\n"
   }
  },
  "dojo/_base/kernel.window": {
   "get": {
    "!type": "fn(doc: +Document)",
    "!doc": "<p>Get window object associated with document doc.</p>\n"
   },
   "getBox": {
    "!type": "fn(doc?: +Document)",
    "!doc": "<p>Returns the dimensions and scroll position of the viewable area of a browser window</p>\n"
   },
   "scrollIntoView": {
    "!type": "fn(node: +Node, pos?: ?)",
    "!doc": "<p>Scroll the passed node into view using minimal movement, if it is not already.</p>\n"
   }
  },
  "dojo/_base/kernel.gears": {
   "available": {
    "!doc": "<p>True if client is using Google Gears</p>\n"
   }
  },
  "dojo/_base/kernel.io": {
   "iframe": {},
   "script": {
    "!doc": "<p>TODOC</p>\n"
   }
  },
  "dojo/_base/kernel.rpc": {
   "JsonpService": {
    "!type": "fn()"
   },
   "JsonService": {
    "!type": "fn()"
   },
   "RpcService": {
    "!type": "fn()"
   }
  },
  "dojo/_base/kernel.store": {
   "util": {},
   "Cache": {
    "!type": "fn(masterStore: ?, cachingStore: ?, options: ?)"
   },
   "DataStore": {
    "!type": "fn()"
   },
   "JsonRest": {
    "!type": "fn()"
   },
   "Memory": {
    "!type": "fn()"
   },
   "Observable": {
    "!type": "fn(store: +dojo)",
    "!doc": "<p>The Observable store wrapper takes a store and sets an observe method on query()\nresults that can be used to monitor results for changes.\n</p>\n"
   }
  },
  "dojo/_base/kernel.tests": {},
  "dojo/has": {
   "!type": "fn(name: string|+Integer) -> bool",
   "!doc": "<p>Return the current value of the named feature.\n</p>\n",
   "cache": {},
   "add": {
    "!type": "fn(name: string|+Integer, test: fn(), now?: bool, force?: bool)",
    "!doc": "<p>Register a new feature test for some named feature.</p>\n"
   },
   "clearElement": {
    "!type": "fn(element: ?)",
    "!doc": "<p>Deletes the contents of the element passed to test functions.</p>\n"
   },
   "load": {
    "!type": "fn(id: string, parentRequire: fn(), loaded: fn())",
    "!doc": "<p>Conditional loading of AMD modules based on a has feature test value.</p>\n"
   },
   "normalize": {
    "!type": "fn(id: ?, toAbsMid: fn())",
    "!doc": "<p>Resolves id into a module id based on possibly-nested tenary expression that branches on has feature test value(s).\n</p>\n"
   }
  },
  "dojo/_base/config": {
   "addOnLoad": {
    "!doc": "<p>Adds a callback via dojo/ready. Useful when Dojo is added after\nthe page loads and djConfig.afterOnLoad is true. Supports the same\narguments as dojo/ready. When using a function reference, use\n<code>djConfig.addOnLoad = function(){};</code>. For object with function name use\n<code>djConfig.addOnLoad = [myObject, &quot;functionName&quot;];</code> and for object with\nfunction reference use\n<code>djConfig.addOnLoad = [myObject, function(){}];</code></p>\n"
   },
   "afterOnLoad": {},
   "baseUrl": {},
   "callback": {
    "!doc": "<p>Defines a callback to be used when dependencies are defined before \nthe loader has been loaded. When provided, they cause the loader to \nexecute require(deps, callback) once it has finished loading. \nShould be used with deps.</p>\n"
   },
   "debugContainerId": {},
   "debugHeight": {},
   "defaultDuration": {
    "!doc": "<p>Default duration, in milliseconds, for wipe and fade animations within dijits.\nAssigned to dijit.defaultDuration.</p>\n"
   },
   "deferredInstrumentation": {
    "!doc": "<p>Whether deferred instrumentation should be loaded or included\nin builds.</p>\n"
   },
   "deps": {
    "!doc": "<p>Defines dependencies to be used before the loader has been loaded.\nWhen provided, they cause the loader to execute require(deps, callback) \nonce it has finished loading. Should be used with callback.</p>\n"
   },
   "dojoBlankHtmlUrl": {
    "!doc": "<p>Used by some modules to configure an empty iframe. Used by dojo/io/iframe and\ndojo/back, and dijit/popup support in IE where an iframe is needed to make sure native\ncontrols do not bleed through the popups. Normally this configuration variable\ndoes not need to be set, except when using cross-domain/CDN Dojo builds.\nSave dojo/resources/blank.html to your domain and set <code>djConfig.dojoBlankHtmlUrl</code>\nto the path on your domain your copy of blank.html.</p>\n"
   },
   "extraLocale": {
    "!doc": "<p>No default value. Specifies additional locales whose\nresources should also be loaded alongside the default locale when\ncalls to <code>dojo.requireLocalization()</code> are processed.</p>\n"
   },
   "ioPublish": {
    "!doc": "<p>Set this to true to enable publishing of topics for the different phases of\nIO operations. Publishing is done via dojo/topic.publish(). See dojo/main.__IoPublish for a list\nof topics that are published.</p>\n"
   },
   "isDebug": {
    "!doc": "<p>Defaults to <code>false</code>. If set to <code>true</code>, ensures that Dojo provides\nextended debugging feedback via Firebug. If Firebug is not available\non your platform, setting <code>isDebug</code> to <code>true</code> will force Dojo to\npull in (and display) the version of Firebug Lite which is\nintegrated into the Dojo distribution, thereby always providing a\ndebugging/logging console when <code>isDebug</code> is enabled. Note that\nFirebug&#39;s <code>console.*</code> methods are ALWAYS defined by Dojo. If\n<code>isDebug</code> is false and you are on a platform without Firebug, these\nmethods will be defined as no-ops.</p>\n"
   },
   "locale": {
    "!doc": "<p>The locale to assume for loading localized resources in this page,\nspecified according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a>.\nMust be specified entirely in lowercase, e.g. <code>en-us</code> and <code>zh-cn</code>.\nSee the documentation for <code>dojo.i18n</code> and <code>dojo.requireLocalization</code>\nfor details on loading localized resources. If no locale is specified,\nDojo assumes the locale of the user agent, according to <code>navigator.userLanguage</code>\nor <code>navigator.language</code> properties.</p>\n"
   },
   "modulePaths": {
    "!doc": "<p>A map of module names to paths relative to <code>dojo.baseUrl</code>. The\nkey/value pairs correspond directly to the arguments which\n<code>dojo.registerModulePath</code> accepts. Specifying\n<code>djConfig.modulePaths = { &quot;foo&quot;: &quot;../../bar&quot; }</code> is the equivalent\nof calling <code>dojo.registerModulePath(&quot;foo&quot;, &quot;../../bar&quot;);</code>. Multiple\nmodules may be configured via <code>djConfig.modulePaths</code>.</p>\n"
   },
   "parseOnLoad": {
    "!doc": "<p>Run the parser after the page is loaded</p>\n"
   },
   "require": {
    "!doc": "<p>An array of module names to be loaded immediately after dojo.js has been included\nin a page.</p>\n"
   },
   "transparentColor": {
    "!doc": "<p>Array containing the r, g, b components used as transparent color in dojo.Color;\nif undefined, [255,255,255] (white) will be used.</p>\n"
   },
   "urchin": {
    "!doc": "<p>Used by <code>dojox.analytics.Urchin</code> as the default UA-123456-7 account\nnumber used when being created. Alternately, you can pass an acct:&quot;&quot;\nparameter to the constructor a la: new dojox.analytics.Urchin({ acct:&quot;UA-123456-7&quot; });</p>\n"
   },
   "useCustomLogger": {
    "!doc": "<p>If set to a value that evaluates to true such as a string or array and\nisDebug is true and Firebug is not available or running, then it bypasses\nthe creation of Firebug Lite allowing you to define your own console object.</p>\n"
   },
   "useDeferredInstrumentation": {
    "!doc": "<p>Whether the deferred instrumentation should be used.\n\n</p>\n<ul>\n<li><code>&quot;report-rejections&quot;</code>: report each rejection as it occurs.</li>\n<li><code>true</code> or <code>1</code> or <code>&quot;report-unhandled-rejections&quot;</code>: wait 1 second\n  in an attempt to detect unhandled rejections.</li>\n</ul>\n"
   }
  },
  "dojo/_base/config.modulePaths": {},
  "dojo/_base/lang": {
   "_extraNames": {
    "!doc": "<p>Lists property names that must be explicitly processed during for-in iteration\nin environments that have has(&quot;bug-for-in-skips-shadowed&quot;) true.</p>\n"
   },
   "clone": {
    "!type": "fn(src: +anything) -> +anything",
    "!doc": "<p>Clones objects (including DOM nodes) and all children.\nWarning: do not clone cyclic structures.</p>\n"
   },
   "delegate": {
    "!type": "fn(obj: ?, props: +Object...)",
    "!doc": "<p>Returns a new object which &quot;looks&quot; to obj for properties which it\ndoes not have a value for. Optionally takes a bag of properties to\nseed the returned object with initially.</p>\n"
   },
   "exists": {
    "!type": "fn(name: string, obj?: ?) -> bool",
    "!doc": "<p>determine if an object supports a given method</p>\n"
   },
   "extend": {
    "!type": "fn(ctor: ?, props: ?)",
    "!doc": "<p>Adds all properties and methods of props to constructor&#39;s\nprototype, making them available to all instances created with\nconstructor.</p>\n"
   },
   "getObject": {
    "!type": "fn(name: string, create?: bool, context?: ?)",
    "!doc": "<p>Get a property from a dot-separated string, such as &quot;A.B.C&quot;</p>\n"
   },
   "hitch": {
    "!type": "fn(scope: ?, method: fn()|+String...)",
    "!doc": "<p>Returns a function that will only ever execute in the given scope.\nThis allows for easy use of object member functions\nin callbacks and other places in which the &quot;this&quot; keyword may\notherwise not reference the expected scope.\nAny number of default positional arguments may be passed as parameters\nbeyond &quot;method&quot;.\nEach of these values will be used to &quot;placehold&quot; (similar to curry)\nfor the hitched function.</p>\n"
   },
   "isAlien": {
    "!type": "fn(it: ?)",
    "!doc": "<p>Returns true if it is a built-in function or some other kind of\noddball that <em>should</em> report as a function but doesn&#39;t</p>\n"
   },
   "isArray": {
    "!type": "fn(it: +anything)",
    "!doc": "<p>Return true if it is an Array.\nDoes not work on Arrays created in other windows.</p>\n"
   },
   "isArrayLike": {
    "!type": "fn(it: +anything) -> bool",
    "!doc": "<p>similar to isArray() but more permissive</p>\n"
   },
   "isFunction": {
    "!type": "fn(it: +anything) -> bool",
    "!doc": "<p>Return true if it is a Function</p>\n"
   },
   "isObject": {
    "!type": "fn(it: +anything) -> bool",
    "!doc": "<p>Returns true if it is a JavaScript object (or an Array, a Function\nor null)</p>\n"
   },
   "isString": {
    "!type": "fn(it: +anything) -> bool",
    "!doc": "<p>Return true if it is a String</p>\n"
   },
   "mixin": {
    "!type": "fn(dest: ?, sources: +Object...)",
    "!doc": "<p>Copies/adds all properties of one or more sources to dest; returns dest.</p>\n"
   },
   "partial": {
    "!type": "fn(method: fn()|string)",
    "!doc": "<p>similar to hitch() except that the scope object is left to be\nwhatever the execution context eventually becomes.</p>\n"
   },
   "replace": {
    "!type": "fn(tmpl: string, map: ?|fn(), pattern?: +RegEx) -> string",
    "!doc": "<p>Performs parameterized substitutions on a string. Throws an\nexception if any parameter is unmatched.</p>\n"
   },
   "setObject": {
    "!type": "fn(name: string, value: +anything, context?: ?)",
    "!doc": "<p>Set a property from a dot-separated string, such as &quot;A.B.C&quot;</p>\n"
   },
   "trim": {
    "!type": "fn(str: string) -> string",
    "!doc": "<p>Trims whitespace from both sides of the string</p>\n"
   }
  },
  "dojo/sniff": {
   "!type": "fn()",
   "!doc": "<p>This module sets has() flags based on the current browser.\nIt returns the has() function.</p>\n"
  },
  "dojo/Deferred": {
   "!type": "fn(canceler?: fn())",
   "!doc": "<p>Creates a new deferred. This API is preferred over\n<code>dojo/_base/Deferred</code>.</p>\n",
   "promise": {},
   "cancel": {
    "!type": "fn(reason: ?, strict?: bool)",
    "!doc": "<p>Inform the deferred it may cancel its asynchronous operation.</p>\n"
   },
   "isCanceled": {
    "!type": "fn() -> bool",
    "!doc": "<p>Checks whether the deferred has been canceled.</p>\n"
   },
   "isFulfilled": {
    "!type": "fn() -> bool",
    "!doc": "<p>Checks whether the deferred has been resolved or rejected.</p>\n"
   },
   "isRejected": {
    "!type": "fn() -> bool",
    "!doc": "<p>Checks whether the deferred has been rejected.</p>\n"
   },
   "isResolved": {
    "!type": "fn() -> bool",
    "!doc": "<p>Checks whether the deferred has been resolved.</p>\n"
   },
   "progress": {
    "!type": "fn(update: ?, strict?: bool) -> +dojo",
    "!doc": "<p>Emit a progress update on the deferred.</p>\n"
   },
   "reject": {
    "!type": "fn(error: ?, strict?: bool) -> +dojo",
    "!doc": "<p>Reject the deferred.</p>\n"
   },
   "resolve": {
    "!type": "fn(value: ?, strict?: bool) -> +dojo",
    "!doc": "<p>Resolve the deferred.</p>\n"
   },
   "then": {
    "!type": "fn(callback?: fn(), errback?: fn(), progback?: fn()) -> +dojo",
    "!doc": "<p>Add new callbacks to the deferred.</p>\n"
   },
   "prototype": {
    "toString": {
     "!type": "fn() -> string"
    }
   }
  },
  "dojo/errors/CancelError": {
   "!type": "fn()",
   "!doc": "<p>Default error if a promise is canceled without a reason.</p>\n"
  },
  "dojo/errors/create": {
   "!type": "fn(name: ?, ctor: ?, base: ?, props: ?) -> fn()"
  },
  "dojo/promise/Promise": {
   "!type": "fn()",
   "!doc": "<p>The public interface to a deferred.</p>\n",
   "prototype": {
    "always": {
     "!type": "fn(callbackOrErrback?: fn()) -> +dojo",
     "!doc": "<p>Add a callback to be invoked when the promise is resolved\nor rejected.</p>\n"
    },
    "cancel": {
     "!type": "fn(reason: ?, strict?: bool)",
     "!doc": "<p>Inform the deferred it may cancel its asynchronous operation.</p>\n"
    },
    "isCanceled": {
     "!type": "fn() -> bool",
     "!doc": "<p>Checks whether the promise has been canceled.</p>\n"
    },
    "isFulfilled": {
     "!type": "fn() -> bool",
     "!doc": "<p>Checks whether the promise has been resolved or rejected.</p>\n"
    },
    "isRejected": {
     "!type": "fn() -> bool",
     "!doc": "<p>Checks whether the promise has been rejected.</p>\n"
    },
    "isResolved": {
     "!type": "fn() -> bool",
     "!doc": "<p>Checks whether the promise has been resolved.</p>\n"
    },
    "otherwise": {
     "!type": "fn(errback?: fn()) -> +dojo",
     "!doc": "<p>Add new errbacks to the promise.</p>\n"
    },
    "then": {
     "!type": "fn(callback?: fn(), errback?: fn(), progback?: fn()) -> +dojo",
     "!doc": "<p>Add new callbacks to the promise.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string"
    },
    "trace": {
     "!type": "fn() -> +dojo",
     "!doc": "<p>Trace the promise.</p>\n"
    },
    "traceRejected": {
     "!type": "fn() -> +dojo",
     "!doc": "<p>Trace rejection of the promise.</p>\n"
    }
   }
  },
  "dojo/DeferredList": {
   "!type": "fn(list: [], fireOnOneCallback?: bool, fireOnOneErrback?: bool, consumeErrors?: bool, canceller?: fn())",
   "!doc": "<p>Deprecated, use dojo/promise/all instead.\nProvides event handling for a group of Deferred objects.</p>\n",
   "prototype": {
    "gatherResults": {
     "!type": "fn(deferredList: +dojo) -> +dojo",
     "!doc": "<p>Gathers the results of the deferreds for packaging\nas the parameters to the Deferred Lists&#39; callback</p>\n"
    }
   }
  },
  "dojo/_base/Deferred": {
   "!type": "fn(canceller?: fn())",
   "!doc": "<p>Deprecated.   This module defines the legacy dojo/_base/Deferred API.\nNew code should use dojo/Deferred instead.</p>\n",
   "prototype": {
    "fired": {},
    "addBoth": {
     "!type": "fn(callback: fn())",
     "!doc": "<p>Add handler as both successful callback and error callback for this deferred instance.</p>\n"
    },
    "addCallback": {
     "!type": "fn(callback: fn())",
     "!doc": "<p>Adds successful callback for this deferred instance.</p>\n"
    },
    "addErrback": {
     "!type": "fn(errback: fn())",
     "!doc": "<p>Adds error callback for this deferred instance.</p>\n"
    }
   },
   "promise": {},
   "addCallbacks": {
    "!type": "fn(callback?: fn(), errback?: fn())",
    "!doc": "<p>Adds callback and error callback for this deferred instance.</p>\n"
   },
   "callback": {
    "!type": "fn(value: ?)",
    "!doc": "<p>Fulfills the Deferred instance successfully with the provide value</p>\n"
   },
   "cancel": {
    "!type": "fn()",
    "!doc": "<p>Cancels the asynchronous operation</p>\n"
   },
   "errback": {
    "!type": "fn(error: ?)",
    "!doc": "<p>Fulfills the Deferred instance as an error with the provided error</p>\n"
   },
   "isCanceled": {
    "!type": "fn() -> bool",
    "!doc": "<p>Checks whether the deferred has been canceled.</p>\n"
   },
   "isFulfilled": {
    "!type": "fn() -> bool",
    "!doc": "<p>Checks whether the deferred has been resolved or rejected.</p>\n"
   },
   "isRejected": {
    "!type": "fn() -> bool",
    "!doc": "<p>Checks whether the deferred has been rejected.</p>\n"
   },
   "isResolved": {
    "!type": "fn() -> bool",
    "!doc": "<p>Checks whether the deferred has been resolved.</p>\n"
   },
   "progress": {
    "!type": "fn(update: ?)",
    "!doc": "<p>Send progress events to all listeners</p>\n"
   },
   "reject": {
    "!type": "fn(error: ?)",
    "!doc": "<p>Fulfills the Deferred instance as an error with the provided error</p>\n"
   },
   "resolve": {
    "!type": "fn(value: ?)",
    "!doc": "<p>Fulfills the Deferred instance successfully with the provide value</p>\n"
   },
   "then": {
    "!type": "fn(resolvedCallback?: fn(), errorCallback?: fn(), progressCallback?: fn())",
    "!doc": "<p>Adds a fulfilledHandler, errorHandler, and progressHandler to be called for\ncompletion of a promise. The fulfilledHandler is called when the promise\nis fulfilled. The errorHandler is called when a promise fails. The\nprogressHandler is called for progress events. All arguments are optional\nand non-function values are ignored. The progressHandler is not only an\noptional argument, but progress events are purely optional. Promise\nproviders are not required to ever create progress events.\n\n</p>\n<p>This function will return a new promise that is fulfilled when the given\nfulfilledHandler or errorHandler callback is finished. This allows promise\noperations to be chained together. The value returned from the callback\nhandler is the fulfillment value for the returned promise. If the callback\nthrows an error, the returned promise will be moved to failed state.\n</p>\n"
   },
   "when": {
    "!type": "fn(valueOrPromise: ?, callback?: fn(), errback?: fn(), progback?: fn()) -> +dojo",
    "!doc": "<p>Transparently applies callbacks to values and/or promises.</p>\n"
   }
  },
  "dojo/when": {
   "!type": "fn(valueOrPromise: ?, callback?: fn(), errback?: fn(), progback?: fn()) -> +dojo",
   "!doc": "<p>Transparently applies callbacks to values and/or promises.</p>\n"
  },
  "dojo/_base/array": {
   "clearCache": {
    "!type": "fn()"
   },
   "every": {
    "!type": "fn(arr: []|string, callback: fn()|string, thisObject?: ?) -> bool",
    "!doc": "<p>Determines whether or not every item in arr satisfies the\ncondition implemented by callback.</p>\n"
   },
   "filter": {
    "!type": "fn(arr: [], callback: fn()|string, thisObject?: ?) -> []",
    "!doc": "<p>Returns a new Array with those items from arr that match the\ncondition implemented by callback.</p>\n"
   },
   "forEach": {
    "!type": "fn(arr: []|string, callback: fn()|string, thisObject?: ?)",
    "!doc": "<p>for every item in arr, callback is invoked. Return values are ignored.\nIf you want to break out of the loop, consider using array.every() or array.some().\nforEach does not allow breaking out of the loop over the items in arr.</p>\n"
   },
   "indexOf": {
    "!type": "fn(arr: [], value: ?, fromIndex?: +Integer, findLast?: bool) -> number",
    "!doc": "<p>locates the first index of the provided value in the\npassed array. If the value is not found, -1 is returned.</p>\n"
   },
   "lastIndexOf": {
    "!type": "fn(arr: ?, value: ?, fromIndex?: +Integer) -> number",
    "!doc": "<p>locates the last index of the provided value in the passed\narray. If the value is not found, -1 is returned.</p>\n"
   },
   "map": {
    "!type": "fn(arr: []|string, callback: fn()|string, thisObject?: ?, Ctr: ?) -> []",
    "!doc": "<p>applies callback to each element of arr and returns\nan Array with the results</p>\n"
   },
   "some": {
    "!type": "fn(arr: []|string, callback: fn()|string, thisObject?: ?) -> bool",
    "!doc": "<p>Determines whether or not any item in arr satisfies the\ncondition implemented by callback.</p>\n"
   }
  },
  "dojo/Evented": {
   "!type": "fn()",
   "!doc": "<p>A class that can be used as a mixin or base class,\nto add on() and emit() methods to a class\nfor listening for events and emitting events:</p>\n",
   "prototype": {
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    }
   }
  },
  "dojo/aspect": {
   "after": {
    "!type": "fn(target: ?, methodName: string, advice: fn(), receiveArguments?: bool)",
    "!doc": "<p>The &quot;after&quot; export of the aspect module is a function that can be used to attach\n&quot;after&quot; advice to a method. This function will be executed after the original method\nis executed. By default the function will be called with a single argument, the return\nvalue of the original method, or the the return value of the last executed advice (if a previous one exists).\nThe fourth (optional) argument can be set to true to so the function receives the original\narguments (from when the original method was called) rather than the return value.\nIf there are multiple &quot;after&quot; advisors, they are executed in the order they were registered.</p>\n"
   },
   "around": {
    "!type": "fn(target: ?, methodName: string, advice: fn())",
    "!doc": "<p>The &quot;around&quot; export of the aspect module is a function that can be used to attach\n&quot;around&quot; advice to a method. The advisor function is immediately executed when\nthe around() is called, is passed a single argument that is a function that can be\ncalled to continue execution of the original method (or the next around advisor).\nThe advisor function should return a function, and this function will be called whenever\nthe method is called. It will be called with the arguments used to call the method.\nWhatever this function returns will be returned as the result of the method call (unless after advise changes it).</p>\n"
   },
   "before": {
    "!type": "fn(target: ?, methodName: string, advice: fn())",
    "!doc": "<p>The &quot;before&quot; export of the aspect module is a function that can be used to attach\n&quot;before&quot; advice to a method. This function will be executed before the original method\nis executed. This function will be called with the arguments used to call the method.\nThis function may optionally return an array as the new arguments to use to call\nthe original method (or the previous, next-to-execute before advice, if one exists).\nIf the before method doesn&#39;t return anything (returns undefined) the original arguments\nwill be preserved.\nIf there are multiple &quot;before&quot; advisors, they are executed in the reverse order they were registered.</p>\n"
   }
  },
  "dojo/on": {
   "!type": "fn(target: +Element|?, type: string|fn(), listener: fn(), dontFix: ?)",
   "!doc": "<p>A function that provides core event listening functionality. With this function\nyou can provide a target, event type, and listener to be notified of\nfuture matching events that are fired.</p>\n",
   "emit": {
    "!type": "fn(target: ?, type: ?, event: ?)"
   },
   "matches": {
    "!type": "fn(node: +Node, selector: string, context: +Node, children: bool, matchesTarget?: ?|+dojo) -> +Node",
    "!doc": "<p>Check if a node match the current selector within the constraint of a context</p>\n"
   },
   "once": {
    "!type": "fn(target: ?, type: ?, listener: ?, dontFix: ?)",
    "!doc": "<p>This function acts the same as on(), but will only call the listener once. The \nlistener will be called for the first\nevent that takes place and then listener will automatically be removed.</p>\n"
   },
   "parse": {
    "!type": "fn(target: ?, type: ?, listener: ?, addListener: ?, dontFix: ?, matchesTarget: ?)"
   },
   "pausable": {
    "!type": "fn(target: ?, type: ?, listener: ?, dontFix: ?)",
    "!doc": "<p>This function acts the same as on(), but with pausable functionality. The\nreturned signal object has pause() and resume() functions. Calling the\npause() method will cause the listener to not be called for future events. Calling the\nresume() method will cause the listener to again be called for future events.</p>\n"
   },
   "selector": {
    "!type": "fn(selector: ?, eventType: ?, children: ?) -> fn()",
    "!doc": "<p>Creates a new extension event with event delegation. This is based on\nthe provided event type (can be extension event) that\nonly calls the listener when the CSS selector matches the target of the event.\n\n</p>\n<p>The application must require() an appropriate level of dojo/query to handle the selector.</p>\n"
   }
  },
  "dojo/NodeList-data": {
   "!type": "fn()",
   "!doc": "<p>Adds data() and removeData() methods to NodeList, and returns NodeList constructor.</p>\n"
  },
  "dojo/query": {
   "!type": "fn(selector: string, context?: string|+Node)",
   "!doc": "<p>This modules provides DOM querying functionality. The module export is a function\nthat can be used to query for DOM nodes by CSS selector and returns a NodeList\nrepresenting the matching nodes.</p>\n",
   "load": {
    "!type": "fn(id: ?, parentRequire: ?, loaded: ?)",
    "!doc": "<p>can be used as AMD plugin to conditionally load new query engine</p>\n"
   },
   "NodeList": {
    "!type": "fn(array: ?) -> []",
    "!doc": "<p>Array-like object which adds syntactic\nsugar for chaining, common iteration operations, animation, and\nnode manipulation. NodeLists are most often returned as the\nresult of dojo/query() calls.</p>\n"
   }
  },
  "dojo/dom": {
   "byId": {
    "!type": "fn(id: string|+Node, doc?: +Document) -> +Element",
    "!doc": "<p>Returns DOM node with matching <code>id</code> attribute or falsy value (ex: null or undefined)\nif not found.  If <code>id</code> is a DomNode, this function is a no-op.\n</p>\n"
   },
   "isDescendant": {
    "!type": "fn(node: +Node|string, ancestor: +Node|string) -> bool",
    "!doc": "<p>Returns true if node is a descendant of ancestor</p>\n"
   },
   "setSelectable": {
    "!type": "fn(node: string|+Node, selectable: bool)"
   }
  },
  "dojo/_base/window": {
   "doc": {
    "!doc": "<p>Alias for the current document. &#39;doc&#39; can be modified\nfor temporary context shifting. See also withDoc().</p>\n"
   },
   "global": {
    "!doc": "<p>Alias for the current window. &#39;global&#39; can be modified\nfor temporary context shifting. See also withGlobal().</p>\n"
   },
   "body": {
    "!type": "fn(doc?: +Document)",
    "!doc": "<p>Return the body element of the specified document or of dojo/_base/window::doc.</p>\n"
   },
   "setContext": {
    "!type": "fn(globalObject: ?, globalDocument: +DocumentElement)",
    "!doc": "<p>changes the behavior of many core Dojo functions that deal with\nnamespace and DOM lookup, changing them to work in a new global\ncontext (e.g., an iframe). The varibles dojo.global and dojo.doc\nare modified as a result of calling this function and the result of\n<code>dojo.body()</code> likewise differs.</p>\n"
   },
   "withDoc": {
    "!type": "fn(documentObject: +DocumentElement, callback: fn(), thisObject?: ?, cbArguments?: [])",
    "!doc": "<p>Invoke callback with documentObject as dojo/_base/window::doc.</p>\n"
   },
   "withGlobal": {
    "!type": "fn(globalObject: ?, callback: fn(), thisObject?: ?, cbArguments?: [])",
    "!doc": "<p>Invoke callback with globalObject as dojo.global and\nglobalObject.document as dojo.doc.</p>\n"
   }
  },
  "dojo/_base/window.global": {
   "undefined_callbacks": {},
   "$": {
    "!type": "fn()"
   },
   "GoogleSearchStoreCallback_undefined_NaN": {
    "!type": "fn(start: ?, data: ?, responseCode: ?, errorMsg: ?)"
   },
   "jQuery": {
    "!type": "fn()"
   },
   "swfIsInHTML": {
    "!type": "fn()"
   },
   "undefined_onload": {
    "!type": "fn()"
   }
  },
  "dojo/_base/window.doc": {
   "documentElement": {},
   "dojoClick": {}
  },
  "dojo/selector/_loader": {
   "load": {
    "!type": "fn(id: ?, parentRequire: ?, loaded: ?, config: ?)"
   }
  },
  "dojo/dom-attr": {
   "get": {
    "!type": "fn(node: +Node|string, name: string)",
    "!doc": "<p>Gets an attribute on an HTML element.</p>\n"
   },
   "getNodeProp": {
    "!type": "fn(node: +Node|string, name: string)",
    "!doc": "<p>Returns an effective value of a property or an attribute.</p>\n"
   },
   "has": {
    "!type": "fn(node: +Node|string, name: string) -> bool",
    "!doc": "<p>Returns true if the requested attribute is specified on the\ngiven element, and false otherwise.</p>\n"
   },
   "remove": {
    "!type": "fn(node: +Node|string, name: string)",
    "!doc": "<p>Removes an attribute from an HTML element.</p>\n"
   },
   "set": {
    "!type": "fn(node: +Node|string, name: string|?, value?: string)",
    "!doc": "<p>Sets an attribute on an HTML element.</p>\n"
   }
  },
  "dojo/dom-style": {
   "get": {
    "!type": "fn(node: +Node|string, name: string|?)"
   },
   "getComputedStyle": {
    "!type": "fn(node: +Node)",
    "!doc": "<p>Returns a &quot;computed style&quot; object.\n</p>\n"
   },
   "set": {
    "!type": "fn(node: +Node|string, name: string|?, value?: string)"
   },
   "toPixelValue": {
    "!type": "fn(node: +Node, value: string) -> number",
    "!doc": "<p>converts style value to pixels on IE or return a numeric value.</p>\n"
   }
  },
  "dojo/dom-prop": {
   "names": {},
   "get": {
    "!type": "fn(node: +Node|string, name: string)",
    "!doc": "<p>Gets a property on an HTML element.</p>\n"
   },
   "set": {
    "!type": "fn(node: +Node|string, name: string|?, value?: string)",
    "!doc": "<p>Sets a property on an HTML element.</p>\n"
   }
  },
  "dojo/dom-prop.names": {
   "class": {},
   "colspan": {},
   "for": {},
   "frameborder": {},
   "readonly": {},
   "rowspan": {},
   "tabindex": {},
   "textcontent": {},
   "valuetype": {}
  },
  "dojo/dom-construct": {
   "create": {
    "!type": "fn(tag: +Node|string, attrs: ?, refNode?: +Node|string, pos?: string)",
    "!doc": "<p>Create an element, allowing for optional attribute decoration\nand placement.</p>\n"
   },
   "destroy": {
    "!type": "fn(node: +Node|string)",
    "!doc": "<p>Removes a node from its parent, clobbering it and all of its\nchildren.\n</p>\n"
   },
   "empty": {
    "!type": "fn(node: +Node|string)",
    "!doc": "<p>safely removes all children of the node.</p>\n"
   },
   "place": {
    "!type": "fn(node: +Node|+DocumentFragment|string, refNode: +Node|string, position?: string|number) -> +Node",
    "!doc": "<p>Attempt to insert node into the DOM, choosing from various positioning options.\nReturns the first argument resolved to a DOM node.</p>\n"
   },
   "toDom": {
    "!type": "fn(frag: string, doc?: +DocumentNode)",
    "!doc": "<p>instantiates an HTML fragment returning the corresponding DOM.</p>\n"
   }
  },
  "dojo/_base/connect": {
   "connect": {
    "!type": "fn(obj?: ?, event: string, context: ?|?, method: string|fn(), dontFix?: bool)",
    "!doc": "<p><code>dojo.connect</code> is a deprecated event handling and delegation method in\nDojo. It allows one function to &quot;listen in&quot; on the execution of\nany other, triggering the second whenever the first is called. Many\nlisteners may be attached to a function, and source functions may\nbe either regular function calls or DOM events.\n</p>\n"
   },
   "connectPublisher": {
    "!type": "fn(topic: string, obj?: ?, event: string)",
    "!doc": "<p>Ensure that every time obj.event() is called, a message is published\non the topic. Returns a handle which can be passed to\ndojo.disconnect() to disable subsequent automatic publication on\nthe topic.</p>\n"
   },
   "disconnect": {
    "!type": "fn(handle: +Handle)",
    "!doc": "<p>Remove a link created by dojo.connect.</p>\n"
   },
   "isCopyKey": {
    "!type": "fn(e: +Event)",
    "!doc": "<p>Checks an event for the copy key (meta on Mac, and ctrl anywhere else)</p>\n"
   },
   "publish": {
    "!type": "fn(topic: string, args?: [])",
    "!doc": "<p>Invoke all listener method subscribed to topic.</p>\n"
   },
   "subscribe": {
    "!type": "fn(topic: string, context?: ?, method: string|fn())",
    "!doc": "<p>Attach a listener to a named topic. The listener function is invoked whenever the\nnamed topic is published (see: dojo.publish).\nReturns a handle which is needed to unsubscribe this listener.</p>\n"
   },
   "unsubscribe": {
    "!type": "fn(handle: +Handle)",
    "!doc": "<p>Remove a topic listener.</p>\n"
   }
  },
  "dojo/topic": {
   "publish": {
    "!type": "fn(topic: string, event: ?)",
    "!doc": "<p>Publishes a message to a topic on the pub/sub hub. All arguments after\nthe first will be passed to the subscribers, so any number of arguments\ncan be provided (not just event).</p>\n"
   },
   "subscribe": {
    "!type": "fn(topic: string, listener: fn())",
    "!doc": "<p>Subscribes to a topic on the pub/sub hub</p>\n"
   }
  },
  "dojo/_base/event": {
   "fix": {
    "!type": "fn(evt: +Event, sender: +Node) -> +Event",
    "!doc": "<p>normalizes properties on the event object including event\nbubbling methods, keystroke normalization, and x/y positions</p>\n"
   },
   "stop": {
    "!type": "fn(evt: +Event)",
    "!doc": "<p>prevents propagation and clobbers the default action of the\npassed event</p>\n"
   }
  },
  "dojo/dom-geometry": {
   "boxModel": {},
   "docScroll": {
    "!type": "fn(doc?: +Document)",
    "!doc": "<p>Returns an object with {node, x, y} with corresponding offsets.</p>\n"
   },
   "fixIeBiDiScrollLeft": {
    "!type": "fn(scrollLeft: number, doc?: +Document) -> number",
    "!doc": "<p>In RTL direction, scrollLeft should be a negative value, but IE\nreturns a positive one. All codes using documentElement.scrollLeft\nmust call this function to fix this error, otherwise the position\nwill offset to right when there is a horizontal scrollbar.</p>\n"
   },
   "getBorderExtents": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>returns an object with properties useful for noting the border\ndimensions.</p>\n"
   },
   "getContentBox": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>Returns an object that encodes the width, height, left and top\npositions of the node&#39;s content box, irrespective of the\ncurrent box model.</p>\n"
   },
   "getIeDocumentElementOffset": {
    "!type": "fn(doc?: +Document)",
    "!doc": "<p>returns the offset in x and y from the document body to the\nvisual edge of the page for IE</p>\n"
   },
   "getMarginBox": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>returns an object that encodes the width, height, left and top\npositions of the node&#39;s margin box.</p>\n"
   },
   "getMarginExtents": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>returns object with properties useful for box fitting with\nregards to box margins (i.e., the outer-box).\n\n</p>\n<ul>\n<li>l/t = marginLeft, marginTop, respectively</li>\n<li>w = total width, margin inclusive</li>\n<li>h = total height, margin inclusive</li>\n</ul>\n<p>The w/h are used for calculating boxes.\nNormally application code will not need to invoke this\ndirectly, and will use the ...box... functions instead.</p>\n"
   },
   "getMarginSize": {
    "!type": "fn(node: +Node|string, computedStyle?: ?)",
    "!doc": "<p>returns an object that encodes the width and height of\nthe node&#39;s margin box</p>\n"
   },
   "getPadBorderExtents": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>Returns object with properties useful for box fitting with\nregards to padding.</p>\n"
   },
   "getPadExtents": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>Returns object with special values specifically useful for node\nfitting.</p>\n"
   },
   "isBodyLtr": {
    "!type": "fn(doc?: +Document) -> bool",
    "!doc": "<p>Returns true if the current language is left-to-right, and false otherwise.</p>\n"
   },
   "normalizeEvent": {
    "!type": "fn(event: ?)",
    "!doc": "<p>Normalizes the geometry of a DOM event, normalizing the pageX, pageY,\noffsetX, offsetY, layerX, and layerX properties</p>\n"
   },
   "position": {
    "!type": "fn(node: +Node|string, includeScroll?: bool)",
    "!doc": "<p>Gets the position and size of the passed element relative to\nthe viewport (if includeScroll==false), or relative to the\ndocument root (if includeScroll==true).\n</p>\n"
   },
   "setContentSize": {
    "!type": "fn(node: +Node, box: ?, computedStyle?: ?)",
    "!doc": "<p>Sets the size of the node&#39;s contents, irrespective of margins,\npadding, or borders.</p>\n"
   },
   "setMarginBox": {
    "!type": "fn(node: +Node, box: ?, computedStyle?: ?)",
    "!doc": "<p>sets the size of the node&#39;s margin box and placement\n(left/top), irrespective of box model. Think of it as a\npassthrough to setBox that handles box-model vagaries for\nyou.</p>\n"
   }
  },
  "dojo/mouse": {
   "enter": {
    "!doc": "<p>This is an extension event for the mouseenter that IE provides, emulating the\nbehavior on other browsers.</p>\n"
   },
   "leave": {
    "!doc": "<p>This is an extension event for the mouseleave that IE provides, emulating the\nbehavior on other browsers.</p>\n"
   },
   "isLeft": {
    "!type": "fn()",
    "!doc": "<p>Test an event object (from a mousedown event) to see if the left button was pressed.</p>\n"
   },
   "isMiddle": {
    "!type": "fn()",
    "!doc": "<p>Test an event object (from a mousedown event) to see if the middle button was pressed.</p>\n"
   },
   "isRight": {
    "!type": "fn()",
    "!doc": "<p>Test an event object (from a mousedown event) to see if the right button was pressed.</p>\n"
   },
   "wheel": {
    "!type": "fn(node: ?, listener: ?)",
    "!doc": "<p>This is an extension event for the mousewheel that non-Mozilla browsers provide,\nemulating the behavior on Mozilla based browsers.</p>\n"
   }
  },
  "dojo/_base/sniff": {},
  "dojo/keys": {
   "ALT": {},
   "BACKSPACE": {},
   "CAPS_LOCK": {},
   "CLEAR": {},
   "copyKey": {},
   "CTRL": {},
   "DELETE": {},
   "DOWN_ARROW": {},
   "DOWN_DPAD": {},
   "END": {},
   "ENTER": {},
   "ESCAPE": {},
   "F1": {},
   "F10": {},
   "F11": {},
   "F12": {},
   "F13": {},
   "F14": {},
   "F15": {},
   "F2": {},
   "F3": {},
   "F4": {},
   "F5": {},
   "F6": {},
   "F7": {},
   "F8": {},
   "F9": {},
   "HELP": {},
   "HOME": {},
   "INSERT": {},
   "LEFT_ARROW": {},
   "LEFT_DPAD": {},
   "LEFT_WINDOW": {},
   "META": {},
   "NUM_LOCK": {},
   "NUMPAD_0": {},
   "NUMPAD_1": {},
   "NUMPAD_2": {},
   "NUMPAD_3": {},
   "NUMPAD_4": {},
   "NUMPAD_5": {},
   "NUMPAD_6": {},
   "NUMPAD_7": {},
   "NUMPAD_8": {},
   "NUMPAD_9": {},
   "NUMPAD_DIVIDE": {},
   "NUMPAD_ENTER": {},
   "NUMPAD_MINUS": {},
   "NUMPAD_MULTIPLY": {},
   "NUMPAD_PERIOD": {},
   "NUMPAD_PLUS": {},
   "PAGE_DOWN": {},
   "PAGE_UP": {},
   "PAUSE": {},
   "RIGHT_ARROW": {},
   "RIGHT_DPAD": {},
   "RIGHT_WINDOW": {},
   "SCROLL_LOCK": {},
   "SELECT": {},
   "SHIFT": {},
   "SPACE": {},
   "TAB": {},
   "UP_ARROW": {},
   "UP_DPAD": {}
  },
  "dojo/NodeList-dom": {
   "!type": "fn()",
   "!doc": "<p>Adds DOM related methods to NodeList, and returns NodeList constructor.</p>\n"
  },
  "dojo/dom-class": {
   "add": {
    "!type": "fn(node: string|+Node, classStr: string|[])",
    "!doc": "<p>Adds the specified classes to the end of the class list on the\npassed node. Will not re-apply duplicate classes.\n</p>\n"
   },
   "contains": {
    "!type": "fn(node: string|+Node, classStr: string) -> bool",
    "!doc": "<p>Returns whether or not the specified classes are a portion of the\nclass list currently applied to the node.</p>\n"
   },
   "remove": {
    "!type": "fn(node: string|+Node, classStr?: string|[])",
    "!doc": "<p>Removes the specified classes from node. No <code>contains()</code>\ncheck is required.\n</p>\n"
   },
   "replace": {
    "!type": "fn(node: string|+Node, addClassStr: string|[], removeClassStr?: string|[])",
    "!doc": "<p>Replaces one or more classes on a node if not present.\nOperates more quickly than calling dojo.removeClass and dojo.addClass\n</p>\n"
   },
   "toggle": {
    "!type": "fn(node: string|+Node, classStr: string|[], condition?: bool) -> bool",
    "!doc": "<p>Adds a class to node if not present, or removes if present.\nPass a boolean condition if you want to explicitly add or remove.\nReturns the condition that was specified directly or indirectly.\n</p>\n"
   }
  },
  "dojo/NodeList-fx": {
   "!type": "fn()",
   "!doc": "<p>Adds dojo.fx animation support to dojo.query() by extending the NodeList class\nwith additional FX functions.  NodeList is the array-like object used to hold query results.</p>\n",
   "prototype": {
    "fadeTo": {
     "!type": "fn(args: ?)",
     "!doc": "<p>fade all elements of the node list to a specified opacity</p>\n"
    },
    "highlight": {
     "!type": "fn(args: ?)",
     "!doc": "<p>highlight all elements of the node list.\nReturns an instance of dojo.Animation</p>\n"
    },
    "sizeTo": {
     "!type": "fn(args: ?)",
     "!doc": "<p>size all elements of this NodeList. Returns an instance of dojo.Animation</p>\n"
    },
    "slideBy": {
     "!type": "fn(args: ?)",
     "!doc": "<p>slide all elements of this NodeList. Returns an instance of dojo.Animation</p>\n"
    },
    "wipeTo": {
     "!type": "fn(args: ?)",
     "!doc": "<p>Wipe all elements of the NodeList to a specified width: or height:</p>\n"
    }
   }
  },
  "dojo/_base/fx": {
   "anim": {
    "!type": "fn(node: +Node|string, properties: ?, duration?: +Integer, easing?: fn(), onEnd?: fn(), delay?: +Integer)",
    "!doc": "<p>A simpler interface to <code>animateProperty()</code>, also returns\nan instance of <code>Animation</code> but begins the animation\nimmediately, unlike nearly every other Dojo animation API.</p>\n"
   },
   "animateProperty": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Returns an animation that will transition the properties of\nnode defined in <code>args</code> depending how they are defined in\n<code>args.properties</code>\n</p>\n"
   },
   "Animation": {
    "!type": "fn(args: ?)",
    "!doc": "<p>A generic animation class that fires callbacks into its handlers\nobject at various states.</p>\n"
   },
   "fadeIn": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Returns an animation that will fade node defined in &#39;args&#39; from\nits current opacity to fully opaque.</p>\n"
   },
   "fadeOut": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Returns an animation that will fade node defined in &#39;args&#39;\nfrom its current opacity to fully transparent.</p>\n"
   }
  },
  "dojo/_base/declare": {
   "!type": "fn(className?: string, superclass: fn()|fn(), props: ?) -> +dojo",
   "!doc": "<p>Create a feature-rich constructor from compact notation.</p>\n",
   "__DeclareCreatedObject": {
    "!doc": "<p>dojo/_base/declare() returns a constructor <code>C</code>.   <code>new C()</code> returns an Object with the following\nmethods, in addition to the methods and properties specified via the arguments passed to declare().</p>\n"
   },
   "safeMixin": {
    "!type": "fn(target: ?, source: ?)",
    "!doc": "<p>Mix in properties skipping a constructor and decorating functions\nlike it is done by declare().</p>\n"
   }
  },
  "dojo/_base/declare.__DeclareCreatedObject": {
   "createSubclass": {
    "!type": "fn(mixins: fn(), props?: ?) -> +dojo",
    "!doc": "<p>Create a subclass of the declared class from a list of base classes.</p>\n"
   },
   "extend": {
    "!type": "fn(source: ?)",
    "!doc": "<p>Adds all properties and methods of source to constructor&#39;s\nprototype, making them available to all instances created with\nconstructor. This method is specific to constructors created with\ndeclare().</p>\n"
   },
   "getInherited": {
    "!type": "fn(name?: string, args: +Arguments)",
    "!doc": "<p>Returns a super method.</p>\n"
   },
   "inherited": {
    "!type": "fn(name?: string, args: +Arguments, newArgs?: ?)",
    "!doc": "<p>Calls a super method.</p>\n"
   },
   "isInstanceOf": {
    "!type": "fn(cls: fn())",
    "!doc": "<p>Checks the inheritance chain to see if it is inherited from this\nclass.</p>\n"
   }
  },
  "dojo/_base/Color": {
   "!type": "fn(color: []|string|?)",
   "!doc": "<p>Takes a named string, hex string, array of rgb or rgba values,\nan object with r, g, b, and a properties, or another <code>Color</code> object\nand creates a new Color instance to work from.\n</p>\n",
   "prototype": {
    "a": {},
    "b": {},
    "g": {},
    "r": {},
    "sanitize": {
     "!type": "fn()",
     "!doc": "<p>makes sure that the object has correct attributes</p>\n"
    },
    "setColor": {
     "!type": "fn(color: []|string|?) -> fn()",
     "!doc": "<p>Takes a named string, hex string, array of rgb or rgba values,\nan object with r, g, b, and a properties, or another <code>Color</code> object\nand sets this color instance to that value.\n</p>\n"
    },
    "toCmy": {
     "!type": "fn()",
     "!doc": "<p>Convert this Color to a CMY definition.</p>\n"
    },
    "toCmyk": {
     "!type": "fn()",
     "!doc": "<p>Convert this Color to a CMYK definition.</p>\n"
    },
    "toCss": {
     "!type": "fn(includeAlpha?: bool) -> string",
     "!doc": "<p>Returns a css color string in rgb(a) representation</p>\n"
    },
    "toHex": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a CSS color string in hexadecimal representation</p>\n"
    },
    "toHsl": {
     "!type": "fn()",
     "!doc": "<p>Convert this Color to an HSL definition.</p>\n"
    },
    "toHsv": {
     "!type": "fn()",
     "!doc": "<p>Convert this Color to an HSV definition.</p>\n"
    },
    "toRgb": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns 3 component array of rgb values</p>\n"
    },
    "toRgba": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns a 4 component array of rgba values from the color\nrepresented by this object.</p>\n"
    },
    "toString": {
     "!type": "fn()",
     "!doc": "<p>Returns a visual representation of the color</p>\n"
    }
   },
   "named": {
    "!doc": "<p>Dictionary list of all CSS named colors, by name. Values are 3-item arrays with corresponding RG and B values.</p>\n"
   },
   "blendColors": {
    "!type": "fn(start: +dojo, end: +dojo, weight: number, obj?: +dojo)",
    "!doc": "<p>Blend colors end and start with weight from 0 to 1, 0.5 being a 50/50 blend,\ncan reuse a previously allocated Color object for the result</p>\n"
   },
   "fromArray": {
    "!type": "fn(a: [], obj?: +dojo)",
    "!doc": "<p>Builds a <code>Color</code> from a 3 or 4 element array, mapping each\nelement in sequence to the rgb(a) values of the color.</p>\n"
   },
   "fromHex": {
    "!type": "fn(color: string, obj?: +dojo)",
    "!doc": "<p>Converts a hex string with a &#39;#&#39; prefix to a color object.\nSupports 12-bit #rgb shorthand. Optionally accepts a\n<code>Color</code> object to update with the parsed value.\n</p>\n"
   },
   "fromRgb": {
    "!type": "fn(color: string, obj?: +dojo)",
    "!doc": "<p>get rgb(a) array from css-style color declarations</p>\n"
   },
   "fromString": {
    "!type": "fn(str: string, obj?: +dojo)",
    "!doc": "<p>Parses <code>str</code> for a color value. Accepts hex, rgb, and rgba\nstyle color values.</p>\n"
   },
   "makeGrey": {
    "!type": "fn(g: number, a?: number)",
    "!doc": "<p>creates a greyscale color with an optional alpha</p>\n"
   }
  },
  "dojo/_base/Color.named": {
   "aliceblue": {},
   "antiquewhite": {},
   "aqua": {},
   "aquamarine": {},
   "azure": {},
   "beige": {},
   "bisque": {},
   "black": {},
   "blanchedalmond": {},
   "blue": {},
   "blueviolet": {},
   "brown": {},
   "burlywood": {},
   "cadetblue": {},
   "chartreuse": {},
   "chocolate": {},
   "coral": {},
   "cornflowerblue": {},
   "cornsilk": {},
   "crimson": {},
   "cyan": {},
   "darkblue": {},
   "darkcyan": {},
   "darkgoldenrod": {},
   "darkgray": {},
   "darkgreen": {},
   "darkgrey": {},
   "darkkhaki": {},
   "darkmagenta": {},
   "darkolivegreen": {},
   "darkorange": {},
   "darkorchid": {},
   "darkred": {},
   "darksalmon": {},
   "darkseagreen": {},
   "darkslateblue": {},
   "darkslategray": {},
   "darkslategrey": {},
   "darkturquoise": {},
   "darkviolet": {},
   "deeppink": {},
   "deepskyblue": {},
   "dimgray": {},
   "dimgrey": {},
   "dodgerblue": {},
   "firebrick": {},
   "floralwhite": {},
   "forestgreen": {},
   "fuchsia": {},
   "gainsboro": {},
   "ghostwhite": {},
   "gold": {},
   "goldenrod": {},
   "gray": {},
   "green": {},
   "greenyellow": {},
   "grey": {},
   "honeydew": {},
   "hotpink": {},
   "indianred": {},
   "indigo": {},
   "ivory": {},
   "khaki": {},
   "lavender": {},
   "lavenderblush": {},
   "lawngreen": {},
   "lemonchiffon": {},
   "lightblue": {},
   "lightcoral": {},
   "lightcyan": {},
   "lightgoldenrodyellow": {},
   "lightgray": {},
   "lightgreen": {},
   "lightgrey": {},
   "lightpink": {},
   "lightsalmon": {},
   "lightseagreen": {},
   "lightskyblue": {},
   "lightslategray": {},
   "lightslategrey": {},
   "lightsteelblue": {},
   "lightyellow": {},
   "lime": {},
   "limegreen": {},
   "linen": {},
   "magenta": {},
   "maroon": {},
   "mediumaquamarine": {},
   "mediumblue": {},
   "mediumorchid": {},
   "mediumpurple": {},
   "mediumseagreen": {},
   "mediumslateblue": {},
   "mediumspringgreen": {},
   "mediumturquoise": {},
   "mediumvioletred": {},
   "midnightblue": {},
   "mintcream": {},
   "mistyrose": {},
   "moccasin": {},
   "navajowhite": {},
   "navy": {},
   "oldlace": {},
   "olive": {},
   "olivedrab": {},
   "orange": {},
   "orangered": {},
   "orchid": {},
   "palegoldenrod": {},
   "palegreen": {},
   "paleturquoise": {},
   "palevioletred": {},
   "papayawhip": {},
   "peachpuff": {},
   "peru": {},
   "pink": {},
   "plum": {},
   "powderblue": {},
   "purple": {},
   "red": {},
   "rosybrown": {},
   "royalblue": {},
   "saddlebrown": {},
   "salmon": {},
   "sandybrown": {},
   "seagreen": {},
   "seashell": {},
   "sienna": {},
   "silver": {},
   "skyblue": {},
   "slateblue": {},
   "slategray": {},
   "slategrey": {},
   "snow": {},
   "springgreen": {},
   "steelblue": {},
   "tan": {},
   "teal": {},
   "thistle": {},
   "tomato": {},
   "transparent": {},
   "turquoise": {},
   "violet": {},
   "wheat": {},
   "white": {},
   "whitesmoke": {},
   "yellow": {},
   "yellowgreen": {}
  },
  "dojo/fx": {
   "easing": {
    "!doc": "<p>Collection of easing functions to use beyond the default\n<code>dojo._defaultEasing</code> function.</p>\n"
   },
   "chain": {
    "!type": "fn(animations: +dojo)",
    "!doc": "<p>Chain a list of <code>dojo/_base/fx.Animation</code>s to run in sequence\n</p>\n"
   },
   "combine": {
    "!type": "fn(animations: +dojo)",
    "!doc": "<p>Combine a list of <code>dojo/_base/fx.Animation</code>s to run in parallel\n</p>\n"
   },
   "slideTo": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Slide a node to a new top/left position\n</p>\n"
   },
   "Toggler": {
    "!type": "fn()"
   },
   "wipeIn": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Expand a node to it&#39;s natural height.\n</p>\n"
   },
   "wipeOut": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Shrink a node to nothing and hide it.\n</p>\n"
   }
  },
  "dojo/fx.Toggler": {
   "!type": "fn()",
   "prototype": {
    "_hideAnim": {},
    "_hideArgs": {},
    "_isHiding": {},
    "_isShowing": {},
    "_showAnim": {},
    "_showArgs": {},
    "hideDuration": {
     "!doc": "<p>Time in milliseconds to run the hide Animation</p>\n"
    },
    "node": {
     "!doc": "<p>the node to target for the showing and hiding animations</p>\n"
    },
    "showDuration": {
     "!doc": "<p>Time in milliseconds to run the show Animation</p>\n"
    },
    "hide": {
     "!type": "fn(delay?: +Integer)",
     "!doc": "<p>Toggle the node to hidden</p>\n"
    },
    "hideFunc": {
     "!type": "fn(args: ?)",
     "!doc": "<p>The function that returns the <code>dojo.Animation</code> to hide the node</p>\n"
    },
    "show": {
     "!type": "fn(delay?: +Integer)",
     "!doc": "<p>Toggle the node to showing</p>\n"
    },
    "showFunc": {
     "!type": "fn(args: ?)",
     "!doc": "<p>The function that returns the <code>dojo.Animation</code> to show the node</p>\n"
    }
   }
  },
  "dojo/fx.easing": {
   "backIn": {
    "!type": "fn(n?: +Decimal) -> number",
    "!doc": "<p>An easing function that starts away from the target,\nand quickly accelerates towards the end value.\n\n</p>\n<p>Use caution when the easing will cause values to become\nnegative as some properties cannot be set to negative values.</p>\n"
   },
   "backInOut": {
    "!type": "fn(n?: +Decimal) -> number",
    "!doc": "<p>An easing function combining the effects of <code>backIn</code> and <code>backOut</code></p>\n"
   },
   "backOut": {
    "!type": "fn(n?: +Decimal) -> number",
    "!doc": "<p>An easing function that pops past the range briefly, and slowly comes back.</p>\n"
   },
   "bounceIn": {
    "!type": "fn(n?: +Decimal) -> number",
    "!doc": "<p>An easing function that &#39;bounces&#39; near the beginning of an Animation</p>\n"
   },
   "bounceInOut": {
    "!type": "fn(n?: +Decimal) -> number",
    "!doc": "<p>An easing function that &#39;bounces&#39; at the beginning and end of the Animation</p>\n"
   },
   "bounceOut": {
    "!type": "fn(n?: +Decimal) -> number",
    "!doc": "<p>An easing function that &#39;bounces&#39; near the end of an Animation</p>\n"
   },
   "circIn": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "circInOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "circOut": {
    "!type": "fn(n?: +Decimal)"
   },
   "cubicIn": {
    "!type": "fn(n?: +Decimal)"
   },
   "cubicInOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "cubicOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "elasticIn": {
    "!type": "fn(n?: +Decimal) -> +Decimal",
    "!doc": "<p>An easing function the elastically snaps from the start value</p>\n"
   },
   "elasticInOut": {
    "!type": "fn(n?: +Decimal) -> number",
    "!doc": "<p>An easing function that elasticly snaps around the value, near\nthe beginning and end of the Animation.</p>\n"
   },
   "elasticOut": {
    "!type": "fn(n?: +Decimal) -> +Decimal",
    "!doc": "<p>An easing function that elasticly snaps around the target value,\nnear the end of the Animation</p>\n"
   },
   "expoIn": {
    "!type": "fn(n?: +Decimal)"
   },
   "expoInOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "expoOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "linear": {
    "!type": "fn(n?: +Decimal) -> +Decimal",
    "!doc": "<p>A linear easing function</p>\n"
   },
   "quadIn": {
    "!type": "fn(n?: +Decimal)"
   },
   "quadInOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "quadOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "quartIn": {
    "!type": "fn(n?: +Decimal)"
   },
   "quartInOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "quartOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "quintIn": {
    "!type": "fn(n?: +Decimal)"
   },
   "quintInOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "quintOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "sineIn": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "sineInOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "sineOut": {
    "!type": "fn(n?: +Decimal)"
   }
  },
  "dojo/ready": {
   "!type": "fn(priority?: +Integer, context: ?, callback?: fn())",
   "!doc": "<p>Add a function to execute on DOM content loaded and all requested modules have arrived and been evaluated.\nIn most cases, the <code>domReady</code> plug-in should suffice and this method should not be needed.\n\n</p>\n<p>When called in a non-browser environment, just checks that all requested modules have arrived and been\nevaluated.</p>\n"
  },
  "dojo/NodeList-html": {
   "!type": "fn()",
   "!doc": "<p>Adds a chainable html method to dojo/query() / NodeList instances for setting/replacing node content</p>\n"
  },
  "dojo/html": {
   "set": {
    "!type": "fn(node: +Node, cont: string|+Node|+NodeList, params?: ?)",
    "!doc": "<p>inserts (replaces) the given content into the given node. dojo/dom-construct.place(cont, node, &quot;only&quot;)\nmay be a better choice for simple HTML insertion.</p>\n"
   }
  },
  "dojo/html._ContentSetter": {
   "!type": "fn()",
   "prototype": {
    "cleanContent": {
     "!doc": "<p>Should the content be treated as a full html document,\nand the real content stripped of <html>, <body> wrapper before injection</p>\n"
    },
    "content": {
     "!doc": "<p>The content to be placed in the node. Can be an HTML string, a node reference, or a enumerable list of nodes</p>\n"
    },
    "extractContent": {
     "!doc": "<p>Should the content be treated as a full html document,\nand the real content stripped of <code>&lt;html&gt; &lt;body&gt;</code> wrapper before injection</p>\n"
    },
    "id": {
     "!doc": "<p>Usually only used internally, and auto-generated with each instance</p>\n"
    },
    "node": {
     "!doc": "<p>An node which will be the parent element that we set content into</p>\n"
    },
    "parseContent": {
     "!doc": "<p>Should the node by passed to the parser after the new content is set</p>\n"
    },
    "parserScope": {
     "!doc": "<p>Flag passed to parser.  Root for attribute names to search for.   If scopeName is dojo,\nwill search for data-dojo-type (or dojoType).  For backwards compatibility\nreasons defaults to dojo._scopeName (which is &quot;dojo&quot; except when\nmulti-version support is used, when it will be something like dojo16, dojo20, etc.)</p>\n"
    },
    "startup": {
     "!doc": "<p>Start the child widgets after parsing them.   Only obeyed if parseContent is true.</p>\n"
    },
    "empty": {
     "!type": "fn()",
     "!doc": "<p>cleanly empty out existing content</p>\n"
    },
    "set": {
     "!type": "fn(cont?: string|+Node|+NodeList, params?: ?)",
     "!doc": "<p>front-end to the set-content sequence</p>\n"
    },
    "setContent": {
     "!type": "fn()",
     "!doc": "<p>sets the content on the node</p>\n"
    },
    "tearDown": {
     "!type": "fn()",
     "!doc": "<p>manually reset the Setter instance if its being re-used for example for another set()</p>\n"
    }
   }
  },
  "dojo/parser": {
   "_attrName": {},
   "_query": {},
   "construct": {
    "!type": "fn(ctor: fn(), node: +Node, mixin?: ?, options?: ?, scripts?: [+DomNode], inherited?: ?)",
    "!doc": "<p>Calls new ctor(params, node), where params is the hash of parameters specified on the node,\nexcluding data-dojo-type and data-dojo-mixins.   Does not call startup().</p>\n"
   },
   "instantiate": {
    "!type": "fn(nodes: [], mixin?: ?, options?: ?)",
    "!doc": "<p>Takes array of nodes, and turns them into class instances and\npotentially calls a startup method to allow them to connect with\nany children.</p>\n"
   },
   "parse": {
    "!type": "fn(rootNode?: +Node, options?: ?) -> +Mixed",
    "!doc": "<p>Scan the DOM for class instances, and instantiate them.</p>\n"
   },
   "scan": {
    "!type": "fn(root?: +Node, options: ?) -> +Promise",
    "!doc": "<p>Scan a DOM tree and return an array of objects representing the DOMNodes\nthat need to be turned into widgets.</p>\n"
   }
  },
  "dojo/_base/url": {
   "!type": "fn()",
   "authority": {},
   "fragment": {},
   "host": {},
   "password": {},
   "path": {},
   "port": {},
   "query": {},
   "scheme": {},
   "uri": {},
   "user": {},
   "prototype": {
    "toString": {
     "!type": "fn()"
    }
   }
  },
  "dojo/_base/url.scheme": {},
  "dojo/_base/url.authority": {},
  "dojo/_base/url.query": {},
  "dojo/_base/url.fragment": {},
  "dojo/_base/url.user": {},
  "dojo/_base/url.password": {},
  "dojo/_base/url.port": {},
  "dojo/promise/all": {
   "!type": "fn(objectOrArray?: ?|[]) -> +dojo",
   "!doc": "<p>Takes multiple promises and returns a new promise that is fulfilled\nwhen all promises have been resolved or one has been rejected.</p>\n"
  },
  "dojo/date/stamp": {
   "_isoRegExp": {},
   "fromISOString": {
    "!type": "fn(formattedString: string, defaultTime?: number)",
    "!doc": "<p>Returns a Date object given a string formatted according to a subset of the ISO-8601 standard.\n</p>\n"
   },
   "toISOString": {
    "!type": "fn(dateObject: +Date, options?: ?)",
    "!doc": "<p>Format a Date object as a string according a subset of the ISO-8601 standard\n</p>\n"
   }
  },
  "dojo/NodeList-manipulate": {
   "!type": "fn()",
   "!doc": "<p>Adds chainable methods to dojo.query() / NodeList instances for manipulating HTML\nand DOM nodes and their properties.</p>\n"
  },
  "dojo/NodeList-traverse": {
   "!type": "fn()",
   "!doc": "<p>Adds chainable methods to dojo/query() / NodeList instances for traversing the DOM</p>\n"
  },
  "dojo/NodeList": {
   "!type": "fn(array: ?) -> []",
   "!doc": "<p>Array-like object which adds syntactic\nsugar for chaining, common iteration operations, animation, and\nnode manipulation. NodeLists are most often returned as the\nresult of dojo/query() calls.</p>\n",
   "_nodeDataCache": {},
   "events": {},
   "prototype": {
    "addClass": {
     "!type": "fn(className: string|[])",
     "!doc": "<p>adds the specified class to every node in the list</p>\n"
    },
    "addClassFx": {
     "!type": "fn(cssClass: ?, args: ?)",
     "!doc": "<p>Animate the effects of adding a class to all nodes in this list.\nsee <code>dojox.fx.addClass</code></p>\n"
    },
    "addContent": {
     "!type": "fn(content: string|+Node|?|+dojo, position?: string|+Integer) -> fn()",
     "!doc": "<p>add a node, NodeList or some HTML as a string to every item in the\nlist.  Returns the original list.</p>\n"
    },
    "adopt": {
     "!type": "fn(queryOrListOrNode: string|[]|+Node, position?: string)",
     "!doc": "<p>places any/all elements in queryOrListOrNode at a\nposition relative to the first element in this list.\nReturns a dojo/NodeList of the adopted elements.</p>\n"
    },
    "after": {
     "!type": "fn(content: string|+Element|+NodeList)",
     "!doc": "<p>Places the content after every node in the NodeList.</p>\n"
    },
    "andSelf": {
     "!type": "fn()",
     "!doc": "<p>Adds the nodes from the previous dojo/NodeList to the current dojo/NodeList.</p>\n"
    },
    "anim": {
     "!type": "fn(properties: ?, duration?: +Integer, easing?: fn(), onEnd?: fn(), delay?: +Integer)",
     "!doc": "<p>Animate one or more CSS properties for all nodes in this list.\nThe returned animation object will already be playing when it\nis returned. See the docs for <code>dojo.anim</code> for full details.</p>\n"
    },
    "animateProperty": {
     "!type": "fn(args?: ?) -> +dojo/_base/fx.Animation",
     "!doc": "<p>Animate all elements of this NodeList across the properties specified.\nsyntax identical to <code>dojo.animateProperty</code>\n</p>\n"
    },
    "append": {
     "!type": "fn(content: string|+Node|+NodeList)",
     "!doc": "<p>appends the content to every node in the NodeList.</p>\n"
    },
    "appendTo": {
     "!type": "fn(query: string)",
     "!doc": "<p>appends nodes in this NodeList to the nodes matched by\nthe query passed to appendTo.</p>\n"
    },
    "at": {
     "!type": "fn(index: +Integer...)",
     "!doc": "<p>Returns a new NodeList comprised of items in this NodeList\nat the given index or indices.\n</p>\n"
    },
    "attr": {
     "!type": "fn(property: string, value?: string)",
     "!doc": "<p>gets or sets the DOM attribute for every element in the\nNodeList. See also <code>dojo/dom-attr</code></p>\n"
    },
    "before": {
     "!type": "fn(content: string|+Node|+NodeList)",
     "!doc": "<p>Places the content before every node in the NodeList.</p>\n"
    },
    "children": {
     "!type": "fn(query?: string)",
     "!doc": "<p>Returns all immediate child elements for nodes in this dojo/NodeList.\nOptionally takes a query to filter the child elements.</p>\n"
    },
    "clone": {
     "!type": "fn()",
     "!doc": "<p>Clones all the nodes in this NodeList and returns them as a new NodeList.</p>\n"
    },
    "closest": {
     "!type": "fn(query: string, root?: string|+Node)",
     "!doc": "<p>Returns closest parent that matches query, including current node in this\ndojo/NodeList if it matches the query.</p>\n"
    },
    "concat": {
     "!type": "fn(item?: ?)",
     "!doc": "<p>Returns a new NodeList comprised of items in this NodeList\nas well as items passed in as parameters</p>\n"
    },
    "connect": {
     "!type": "fn(methodName: string, objOrFunc: ?|fn()|string, funcName?: string)",
     "!doc": "<p>Attach event handlers to every item of the NodeList. Uses dojo.connect()\nso event properties are normalized.\n\n</p>\n<p>Application must manually require() &quot;dojo/_base/connect&quot; before using this method.</p>\n"
    },
    "coords": {
     "!type": "fn()",
     "!doc": "<p>Deprecated: Use position() for border-box x/y/w/h\nor marginBox() for margin-box w/h/l/t.\nReturns the box objects of all elements in a node list as\nan Array (<em>not</em> a NodeList). Acts like <code>domGeom.coords</code>, though assumes\nthe node passed is each node in this list.</p>\n"
    },
    "data": {
     "!type": "fn(key?: ?|string, value?: +Anything) -> +Object",
     "!doc": "<p>stash or get some arbitrary data on/from these nodes.\n</p>\n"
    },
    "delegate": {
     "!type": "fn(selector: string, eventName: string, fn: fn())",
     "!doc": "<p>Monitor nodes in this NodeList for [bubbled] events on nodes that match selector.\nCalls fn(evt) for those events, where (inside of fn()), this == the node\nthat matches the selector.</p>\n"
    },
    "dtl": {
     "!type": "fn(template: +dojox|string, context: +dojox|?) -> fn()",
     "!doc": "<p>Renders the specified template in each of the NodeList entries.</p>\n"
    },
    "empty": {
     "!type": "fn()",
     "!doc": "<p>clears all content from each node in the list. Effectively\nequivalent to removing all child nodes from every item in\nthe list.</p>\n"
    },
    "end": {
     "!type": "fn()",
     "!doc": "<p>Ends use of the current <code>NodeList</code> by returning the previous NodeList\nthat generated the current NodeList.</p>\n"
    },
    "even": {
     "!type": "fn()",
     "!doc": "<p>Returns the even nodes in this dojo/NodeList as a dojo/NodeList.</p>\n"
    },
    "every": {
     "!type": "fn(callback: fn(), thisObject?: ?)",
     "!doc": "<p>see <code>dojo/_base/array.every()</code> and the <a href=\"https://developer.mozilla.org/en/JavaScript/Reference/Global_Objects/Array/every\">Array.every\ndocs</a>.\nTakes the same structure of arguments and returns as\ndojo/_base/array.every() with the caveat that the passed array is\nimplicitly this NodeList</p>\n"
    },
    "fadeIn": {
     "!type": "fn(args?: ?) -> +dojo/_base/fx.Animation",
     "!doc": "<p>fade in all elements of this NodeList via <code>dojo.fadeIn</code>\n</p>\n"
    },
    "fadeOut": {
     "!type": "fn(args?: ?) -> +dojo/_base/fx.Animation",
     "!doc": "<p>fade out all elements of this NodeList via <code>dojo.fadeOut</code>\n</p>\n"
    },
    "filter": {
     "!type": "fn(filter: string|fn())",
     "!doc": "<p>&quot;masks&quot; the built-in javascript filter() method (supported\nin Dojo via <code>dojo.filter</code>) to support passing a simple\nstring filter in addition to supporting filtering function\nobjects.</p>\n"
    },
    "first": {
     "!type": "fn()",
     "!doc": "<p>Returns the first node in this dojo/NodeList as a dojo/NodeList.</p>\n"
    },
    "forEach": {
     "!type": "fn(callback: ?, thisObj: ?) -> fn()",
     "!doc": "<p>see <code>dojo/_base/array.forEach()</code>. The primary difference is that the acted-on\narray is implicitly this NodeList. If you want the option to break out\nof the forEach loop, use every() or some() instead.</p>\n"
    },
    "html": {
     "!type": "fn(value?: string|+Node|+NodeList)",
     "!doc": "<p>allows setting the innerHTML of each node in the NodeList,\nif there is a value passed in, otherwise, reads the innerHTML value of the first node.</p>\n"
    },
    "indexOf": {
     "!type": "fn(value: ?, fromIndex?: +Integer)",
     "!doc": "<p>see <code>dojo/_base/array.indexOf()</code>. The primary difference is that the acted-on\narray is implicitly this NodeList</p>\n"
    },
    "innerHTML": {
     "!type": "fn(value?: string|+Node|+NodeList)",
     "!doc": "<p>allows setting the innerHTML of each node in the NodeList,\nif there is a value passed in, otherwise, reads the innerHTML value of the first node.</p>\n"
    },
    "insertAfter": {
     "!type": "fn(query: string)",
     "!doc": "<p>The nodes in this NodeList will be placed after the nodes\nmatched by the query passed to insertAfter.</p>\n"
    },
    "insertBefore": {
     "!type": "fn(query: string)",
     "!doc": "<p>The nodes in this NodeList will be placed after the nodes\nmatched by the query passed to insertAfter.</p>\n"
    },
    "instantiate": {
     "!type": "fn(declaredClass: string|?, properties?: ?)",
     "!doc": "<p>Create a new instance of a specified class, using the\nspecified properties and each node in the NodeList as a\nsrcNodeRef.</p>\n"
    },
    "last": {
     "!type": "fn()",
     "!doc": "<p>Returns the last node in this dojo/NodeList as a dojo/NodeList.</p>\n"
    },
    "lastIndexOf": {
     "!type": "fn(value: ?, fromIndex?: +Integer)",
     "!doc": "<p>see <code>dojo/_base/array.lastIndexOf()</code>. The primary difference is that the\nacted-on array is implicitly this NodeList</p>\n"
    },
    "map": {
     "!type": "fn(func: fn(), obj?: fn())",
     "!doc": "<p>see <code>dojo/_base/array.map()</code>. The primary difference is that the acted-on\narray is implicitly this NodeList and the return is a\nNodeList (a subclass of Array)</p>\n"
    },
    "marginBox": {
     "!type": "fn()",
     "!doc": "<p>Returns margin-box size of nodes</p>\n"
    },
    "next": {
     "!type": "fn(query?: string)",
     "!doc": "<p>Returns the next element for nodes in this dojo/NodeList.\nOptionally takes a query to filter the next elements.</p>\n"
    },
    "nextAll": {
     "!type": "fn(query?: string)",
     "!doc": "<p>Returns all sibling elements that come after the nodes in this dojo/NodeList.\nOptionally takes a query to filter the sibling elements.</p>\n"
    },
    "odd": {
     "!type": "fn()",
     "!doc": "<p>Returns the odd nodes in this dojo/NodeList as a dojo/NodeList.</p>\n"
    },
    "on": {
     "!type": "fn(eventName: ?, listener: ?)",
     "!doc": "<p>Listen for events on the nodes in the NodeList. Basic usage is:\n</p>\n"
    },
    "orphan": {
     "!type": "fn(filter?: string)",
     "!doc": "<p>removes elements in this list that match the filter\nfrom their parents and returns them as a new NodeList.</p>\n"
    },
    "parent": {
     "!type": "fn(query?: string)",
     "!doc": "<p>Returns immediate parent elements for nodes in this dojo/NodeList.\nOptionally takes a query to filter the parent elements.</p>\n"
    },
    "parents": {
     "!type": "fn(query?: string)",
     "!doc": "<p>Returns all parent elements for nodes in this dojo/NodeList.\nOptionally takes a query to filter the child elements.</p>\n"
    },
    "place": {
     "!type": "fn(queryOrNode: string|+Node, position: string)",
     "!doc": "<p>places elements of this node list relative to the first element matched\nby queryOrNode. Returns the original NodeList. See: <code>dojo/dom-construct.place</code></p>\n"
    },
    "position": {
     "!type": "fn()",
     "!doc": "<p>Returns border-box objects (x/y/w/h) of all elements in a node list\nas an Array (<em>not</em> a NodeList). Acts like <code>dojo/dom-geometry-position</code>, though\nassumes the node passed is each node in this list.</p>\n"
    },
    "prepend": {
     "!type": "fn(content: string|+Node|+NodeList)",
     "!doc": "<p>prepends the content to every node in the NodeList.</p>\n"
    },
    "prependTo": {
     "!type": "fn(query: string)",
     "!doc": "<p>prepends nodes in this NodeList to the nodes matched by\nthe query passed to prependTo.</p>\n"
    },
    "prev": {
     "!type": "fn(query?: string)",
     "!doc": "<p>Returns the previous element for nodes in this dojo/NodeList.\nOptionally takes a query to filter the previous elements.</p>\n"
    },
    "prevAll": {
     "!type": "fn(query?: string)",
     "!doc": "<p>Returns all sibling elements that come before the nodes in this dojo/NodeList.\nOptionally takes a query to filter the sibling elements.</p>\n"
    },
    "query": {
     "!type": "fn(queryStr: string) -> fn()",
     "!doc": "<p>Returns a new list whose members match the passed query,\nassuming elements of the current NodeList as the root for\neach search.</p>\n"
    },
    "remove": {
     "!type": "fn(filter?: string)",
     "!doc": "<p>removes elements in this list that match the filter\nfrom their parents and returns them as a new NodeList.</p>\n"
    },
    "removeAttr": {
     "!type": "fn(name: string)",
     "!doc": "<p>Removes an attribute from each node in the list.</p>\n"
    },
    "removeClass": {
     "!type": "fn(className?: string|[])",
     "!doc": "<p>removes the specified class from every node in the list</p>\n"
    },
    "removeClassFx": {
     "!type": "fn(cssClass: ?, args: ?)",
     "!doc": "<p>Animate the effect of removing a class to all nodes in this list.\nsee <code>dojox.fx.removeClass</code></p>\n"
    },
    "removeData": {
     "!type": "fn(key?: string)",
     "!doc": "<p>Remove the data associated with these nodes.</p>\n"
    },
    "replaceAll": {
     "!type": "fn(query: string) -> fn()",
     "!doc": "<p>replaces nodes matched by the query passed to replaceAll with the nodes\nin this NodeList.</p>\n"
    },
    "replaceClass": {
     "!type": "fn(addClassStr: string|[], removeClassStr?: string|[])",
     "!doc": "<p>Replaces one or more classes on a node if not present.\nOperates more quickly than calling <code>removeClass()</code> and <code>addClass()</code></p>\n"
    },
    "replaceWith": {
     "!type": "fn(content: string|+Node|+NodeList) -> fn()",
     "!doc": "<p>Replaces each node in ths NodeList with the content passed to replaceWith.</p>\n"
    },
    "siblings": {
     "!type": "fn(query?: string)",
     "!doc": "<p>Returns all sibling elements for nodes in this dojo/NodeList.\nOptionally takes a query to filter the sibling elements.</p>\n"
    },
    "slice": {
     "!type": "fn(begin: +Integer, end?: +Integer)",
     "!doc": "<p>Returns a new NodeList, maintaining this one in place</p>\n"
    },
    "slideTo": {
     "!type": "fn(args?: ?) -> +dojo/_base/fx.Animation",
     "!doc": "<p>slide all elements of the node list to the specified place via <code>dojo/fx.slideTo()</code>\n</p>\n"
    },
    "some": {
     "!type": "fn(callback: fn(), thisObject?: ?)",
     "!doc": "<p>Takes the same structure of arguments and returns as\n<code>dojo/_base/array.some()</code> with the caveat that the passed array is\nimplicitly this NodeList.  See <code>dojo/_base/array.some()</code> and Mozilla&#39;s\n<a href=\"https://developer.mozilla.org/en/JavaScript/Reference/Global_Objects/Array/some\">Array.some\ndocumentation</a>.</p>\n"
    },
    "splice": {
     "!type": "fn(index: +Integer, howmany?: +Integer, item?: +Object...)",
     "!doc": "<p>Returns a new NodeList, manipulating this NodeList based on\nthe arguments passed, potentially splicing in new elements\nat an offset, optionally deleting elements</p>\n"
    },
    "style": {
     "!type": "fn(property: string, value?: string)",
     "!doc": "<p>gets or sets the CSS property for every element in the NodeList</p>\n"
    },
    "text": {
     "!type": "fn(value: string) -> fn()",
     "!doc": "<p>allows setting the text value of each node in the NodeList,\nif there is a value passed in, otherwise, returns the text value for all the\nnodes in the NodeList in one string.</p>\n"
    },
    "toggleClass": {
     "!type": "fn(className: string, condition?: bool)",
     "!doc": "<p>Adds a class to node if not present, or removes if present.\nPass a boolean condition if you want to explicitly add or remove.</p>\n"
    },
    "toggleClassFx": {
     "!type": "fn(cssClass: ?, force: ?, args: ?)",
     "!doc": "<p>Animate the effect of adding or removing a class to all nodes in this list.\nsee <code>dojox.fx.toggleClass</code></p>\n"
    },
    "toString": {
     "!type": "fn()"
    },
    "val": {
     "!type": "fn(value: string|[]) -> fn()",
     "!doc": "<p>If a value is passed, allows seting the value property of form elements in this\nNodeList, or properly selecting/checking the right value for radio/checkbox/select\nelements. If no value is passed, the value of the first node in this NodeList\nis returned.</p>\n"
    },
    "wipeIn": {
     "!type": "fn(args?: ?) -> +dojo/_base/fx.Animation",
     "!doc": "<p>wipe in all elements of this NodeList via <code>dojo/fx.wipeIn()</code>\n</p>\n"
    },
    "wipeOut": {
     "!type": "fn(args?: ?) -> +dojo/_base/fx.Animation",
     "!doc": "<p>wipe out all elements of this NodeList via <code>dojo/fx.wipeOut()</code>\n</p>\n"
    },
    "wrap": {
     "!type": "fn(html: string|+Node) -> fn()",
     "!doc": "<p>Wrap each node in the NodeList with html passed to wrap.</p>\n"
    },
    "wrapAll": {
     "!type": "fn(html: string|+Node) -> fn()",
     "!doc": "<p>Insert html where the first node in this NodeList lives, then place all\nnodes in this NodeList as the child of the html.</p>\n"
    },
    "wrapInner": {
     "!type": "fn(html: string|+Node) -> fn()",
     "!doc": "<p>For each node in the NodeList, wrap all its children with the passed in html.</p>\n"
    }
   }
  },
  "dojo/NodeList._nodeDataCache": {},
  "dojo/Stateful": {
   "!type": "fn()",
   "prototype": {
    "_attrPairNames": {
     "!doc": "<p>Used across all instances a hash to cache attribute names and their getter \nand setter names.</p>\n"
    },
    "get": {
     "!type": "fn(name: string)",
     "!doc": "<p>Get a property on a Stateful instance.</p>\n"
    },
    "postscript": {
     "!type": "fn(params?: ?)"
    },
    "set": {
     "!type": "fn(name: string, value: ?) -> fn()",
     "!doc": "<p>Set a property on a Stateful instance</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dojo/_base/NodeList": {},
  "dojo/_base/html": {},
  "dojo/_base/browser": {},
  "dojo/_base/unload": {
   "addOnUnload": {
    "!type": "fn(obj?: ?|fn(), functionName?: string|fn())",
    "!doc": "<p>Registers a function to be triggered when the page unloads.\nDeprecated, use on(window, &quot;beforeunload&quot;, lang.hitch(obj, functionName)) instead.</p>\n"
   },
   "addOnWindowUnload": {
    "!type": "fn(obj?: ?|fn(), functionName?: string|fn())",
    "!doc": "<p>Registers a function to be triggered when window.onunload fires.\nDeprecated, use on(window, &quot;unload&quot;, lang.hitch(obj, functionName)) instead.</p>\n"
   }
  },
  "dojo/_base/xhr": {
   "!type": "fn(method: string, args: ?, hasBody?: bool)",
   "!doc": "<p>Deprecated.   Use dojo/request instead.</p>\n",
   "contentHandlers": {
    "!doc": "<p>A map of available XHR transport handle types. Name matches the\n<code>handleAs</code> attribute passed to XHR calls.</p>\n"
   },
   "del": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP DELETE request to the server.</p>\n"
   },
   "fieldToObject": {
    "!type": "fn(inputNode: +Node|string)",
    "!doc": "<p>Serialize a form field to a JavaScript object.</p>\n"
   },
   "formToJson": {
    "!type": "fn(formNode: +Node|string, prettyPrint?: bool) -> string",
    "!doc": "<p>Create a serialized JSON string from a form node or string\nID identifying the form to serialize</p>\n"
   },
   "formToObject": {
    "!type": "fn(formNode: +Node|string)",
    "!doc": "<p>Serialize a form node to a JavaScript object.</p>\n"
   },
   "formToQuery": {
    "!type": "fn(formNode: +Node|string) -> string",
    "!doc": "<p>Returns a URL-encoded string representing the form passed as either a\nnode or string ID identifying the form to serialize</p>\n"
   },
   "get": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP GET request to the server.</p>\n"
   },
   "objectToQuery": {
    "!type": "fn(map: ?)",
    "!doc": "<p>takes a name/value mapping object and returns a string representing\na URL-encoded version of that object.</p>\n"
   },
   "post": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP POST request to the server. In addition to the properties\nlisted for the dojo.__XhrArgs type, the following property is allowed:</p>\n"
   },
   "put": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP PUT request to the server. In addition to the properties\nlisted for the dojo.__XhrArgs type, the following property is allowed:</p>\n"
   },
   "queryToObject": {
    "!type": "fn(str: string)",
    "!doc": "<p>Create an object representing a de-serialized query section of a\nURL. Query keys with multiple values are returned in an array.\n</p>\n"
   }
  },
  "dojo/_base/xhr.contentHandlers": {
   "auto": {
    "!type": "fn(xhr: ?)"
   },
   "javascript": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which evaluates the response data, expecting it to be valid JavaScript</p>\n"
   },
   "json": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which returns a JavaScript object created from the response data</p>\n"
   },
   "json-comment-filtered": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which expects comment-filtered JSON.</p>\n"
   },
   "json-comment-optional": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which checks the presence of comment-filtered JSON and\nalternates between the <code>json</code> and <code>json-comment-filtered</code> contentHandlers.</p>\n"
   },
   "text": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which simply returns the plaintext response data</p>\n"
   },
   "xml": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler returning an XML Document parsed from the response data</p>\n"
   }
  },
  "dojo/io-query": {
   "objectToQuery": {
    "!type": "fn(map: ?)",
    "!doc": "<p>takes a name/value mapping object and returns a string representing\na URL-encoded version of that object.</p>\n"
   },
   "queryToObject": {
    "!type": "fn(str: string)",
    "!doc": "<p>Create an object representing a de-serialized query section of a\nURL. Query keys with multiple values are returned in an array.\n</p>\n"
   }
  },
  "dojo/dom-form": {
   "fieldToObject": {
    "!type": "fn(inputNode: +Node|string)",
    "!doc": "<p>Serialize a form field to a JavaScript object.</p>\n"
   },
   "toJson": {
    "!type": "fn(formNode: +Node|string, prettyPrint?: bool) -> string",
    "!doc": "<p>Create a serialized JSON string from a form node or string\nID identifying the form to serialize</p>\n"
   },
   "toObject": {
    "!type": "fn(formNode: +Node|string)",
    "!doc": "<p>Serialize a form node to a JavaScript object.</p>\n"
   },
   "toQuery": {
    "!type": "fn(formNode: +Node|string) -> string",
    "!doc": "<p>Returns a URL-encoded string representing the form passed as either a\nnode or string ID identifying the form to serialize</p>\n"
   }
  },
  "dojo/json": {
   "parse": {
    "!type": "fn(str: ?, strict: ?)",
    "!doc": "<p>Parses a <a href=\"http://json.org\">JSON</a> string to return a JavaScript object.</p>\n"
   },
   "stringify": {
    "!type": "fn(value: ?, replacer: ?, spacer: ?)",
    "!doc": "<p>Returns a <a href=\"http://json.org\">JSON</a> serialization of an object.</p>\n"
   }
  },
  "dojo/_base/json": {},
  "dojo/request/watch": {
   "!type": "fn(dfd: +Deferred)",
   "!doc": "<p>Watches the io request represented by dfd to see if it completes.</p>\n",
   "prototype": {
    "ioCheck": {
     "!doc": "<p>Function used to check if basic IO call worked. Gets the dfd\nobject as its only argument.</p>\n"
    },
    "resHandle": {
     "!doc": "<p>Function used to process response. Gets the dfd\nobject as its only argument.</p>\n"
    },
    "validCheck": {
     "!doc": "<p>Function used to check if the IO request is still valid. Gets the dfd\nobject as its only argument.</p>\n"
    }
   },
   "cancelAll": {
    "!type": "fn()",
    "!doc": "<p>Cancels all pending IO requests, regardless of IO type</p>\n"
   }
  },
  "dojo/request/util": {
   "addCommonMethods": {
    "!type": "fn(provider: ?, methods: ?)"
   },
   "checkStatus": {
    "!type": "fn(stat: ?) -> bool"
   },
   "deepCopy": {
    "!type": "fn(target: ?, source: ?)"
   },
   "deepCreate": {
    "!type": "fn(source: ?, properties: ?)"
   },
   "deferred": {
    "!type": "fn(response: ?, cancel: ?, isValid: ?, isReady: ?, handleResponse: ?, last: ?)"
   },
   "notify": {
    "!type": "fn(type?: string, listener?: fn())",
    "!doc": "<p>Register a listener to be notified when an event\nin dojo/request happens.</p>\n"
   },
   "parseArgs": {
    "!type": "fn(url: ?, options: ?, skipData: ?)"
   }
  },
  "dojo/errors/RequestError": {
   "!type": "fn()",
   "!doc": "<p>TODOC</p>\n"
  },
  "dojo/errors/RequestTimeoutError": {
   "!type": "fn()",
   "!doc": "<p>TODOC</p>\n"
  },
  "dojo/request/xhr": {
   "!type": "fn(url: string, options?: +dojo) -> +dojo",
   "!doc": "<p>Sends a request using XMLHttpRequest with the given URL and options.</p>\n",
   "del": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP DELETE request using XMLHttpRequest with the given URL and options.</p>\n"
   },
   "get": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP GET request using XMLHttpRequest with the given URL and options.</p>\n"
   },
   "post": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP POST request using XMLHttpRequest with the given URL and options.</p>\n"
   },
   "put": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP PUT request using XMLHttpRequest with the given URL and options.</p>\n"
   }
  },
  "dojo/request/xhr.__BaseOptions": {
   "!type": "fn()",
   "prototype": {
    "data": {
     "!doc": "<p>Data to transfer. This is ignored for GET and DELETE\nrequests.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>How to handle the response from the server.  Default is\n&#39;text&#39;.  Other values are &#39;json&#39;, &#39;javascript&#39;, and &#39;xml&#39;.</p>\n"
    },
    "headers": {
     "!doc": "<p>Headers to use for the request.</p>\n"
    },
    "password": {
     "!doc": "<p>Password to use during the request.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Whether to append a cache-busting parameter to the URL.</p>\n"
    },
    "query": {
     "!doc": "<p>Query parameters to append to the URL.</p>\n"
    },
    "sync": {
     "!doc": "<p>Whether to make a synchronous request or not. Default\nis <code>false</code> (asynchronous).</p>\n"
    },
    "timeout": {
     "!doc": "<p>Milliseconds to wait for the response.  If this time\npasses, the then the promise is rejected.</p>\n"
    },
    "user": {
     "!doc": "<p>Username to use during the request.</p>\n"
    },
    "withCredentials": {
     "!doc": "<p>For cross-site requests, whether to send credentials\nor not.</p>\n"
    }
   }
  },
  "dojo/request/xhr.__MethodOptions": {
   "!type": "fn()",
   "prototype": {
    "method": {
     "!doc": "<p>The HTTP method to use to make the request. Must be\nuppercase. Default is <code>&quot;GET&quot;</code>.</p>\n"
    }
   }
  },
  "dojo/request/xhr.__Options": {
   "!type": "fn()",
   "prototype": {
    "data": {
     "!doc": "<p>Data to transfer. This is ignored for GET and DELETE\nrequests.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>How to handle the response from the server.  Default is\n&#39;text&#39;.  Other values are &#39;json&#39;, &#39;javascript&#39;, and &#39;xml&#39;.</p>\n"
    },
    "headers": {
     "!doc": "<p>Headers to use for the request.</p>\n"
    },
    "method": {
     "!doc": "<p>The HTTP method to use to make the request. Must be\nuppercase. Default is <code>&quot;GET&quot;</code>.</p>\n"
    },
    "password": {
     "!doc": "<p>Password to use during the request.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Whether to append a cache-busting parameter to the URL.</p>\n"
    },
    "query": {
     "!doc": "<p>Query parameters to append to the URL.</p>\n"
    },
    "sync": {
     "!doc": "<p>Whether to make a synchronous request or not. Default\nis <code>false</code> (asynchronous).</p>\n"
    },
    "timeout": {
     "!doc": "<p>Milliseconds to wait for the response.  If this time\npasses, the then the promise is rejected.</p>\n"
    },
    "user": {
     "!doc": "<p>Username to use during the request.</p>\n"
    },
    "withCredentials": {
     "!doc": "<p>For cross-site requests, whether to send credentials\nor not.</p>\n"
    }
   }
  },
  "dojo/request/handlers": {
   "!type": "fn(response: ?)",
   "register": {
    "!type": "fn(name: ?, handler: ?)"
   }
  },
  "dojo/request": {
   "!type": "fn(url: string, options?: +dojo) -> +dojo",
   "!doc": "<p>Send a request using the default transport for the current platform.</p>\n",
   "del": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP DELETE request using the default transport for the current platform.</p>\n"
   },
   "get": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP GET request using the default transport for the current platform.</p>\n"
   },
   "post": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP POST request using the default transport for the current platform.</p>\n"
   },
   "put": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP POST request using the default transport for the current platform.</p>\n"
   }
  },
  "dojo/request.__Promise": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/promise/Promise.prototype",
    "response": {
     "!doc": "<p>A promise resolving to an object representing\nthe response from the server.</p>\n"
    },
    "always": {
     "!type": "fn(callbackOrErrback?: fn()) -> +dojo",
     "!doc": "<p>Add a callback to be invoked when the promise is resolved\nor rejected.</p>\n"
    },
    "cancel": {
     "!type": "fn(reason: ?, strict?: bool)",
     "!doc": "<p>Inform the deferred it may cancel its asynchronous operation.</p>\n"
    },
    "isCanceled": {
     "!type": "fn() -> bool",
     "!doc": "<p>Checks whether the promise has been canceled.</p>\n"
    },
    "isFulfilled": {
     "!type": "fn() -> bool",
     "!doc": "<p>Checks whether the promise has been resolved or rejected.</p>\n"
    },
    "isRejected": {
     "!type": "fn() -> bool",
     "!doc": "<p>Checks whether the promise has been rejected.</p>\n"
    },
    "isResolved": {
     "!type": "fn() -> bool",
     "!doc": "<p>Checks whether the promise has been resolved.</p>\n"
    },
    "otherwise": {
     "!type": "fn(errback?: fn()) -> +dojo",
     "!doc": "<p>Add new errbacks to the promise.</p>\n"
    },
    "then": {
     "!type": "fn(callback?: fn(), errback?: fn(), progback?: fn()) -> +dojo",
     "!doc": "<p>Add new callbacks to the promise.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string"
    },
    "trace": {
     "!type": "fn() -> fn()"
    },
    "traceRejected": {
     "!type": "fn() -> fn()"
    }
   }
  },
  "dojo/request.__BaseOptions": {
   "!type": "fn()",
   "prototype": {
    "data": {
     "!doc": "<p>Data to transfer.  This is ignored for GET and DELETE\nrequests.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>How to handle the response from the server.  Default is\n&#39;text&#39;.  Other values are &#39;json&#39;, &#39;javascript&#39;, and &#39;xml&#39;.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Whether to append a cache-busting parameter to the URL.</p>\n"
    },
    "query": {
     "!doc": "<p>Query parameters to append to the URL.</p>\n"
    },
    "timeout": {
     "!doc": "<p>Milliseconds to wait for the response.  If this time\npasses, the then the promise is rejected.</p>\n"
    }
   }
  },
  "dojo/request.__MethodOptions": {
   "!type": "fn()",
   "prototype": {
    "method": {
     "!doc": "<p>The HTTP method to use to make the request.  Must be\nuppercase.</p>\n"
    }
   }
  },
  "dojo/request.__Options": {
   "!type": "fn()",
   "prototype": {
    "data": {
     "!doc": "<p>Data to transfer.  This is ignored for GET and DELETE\nrequests.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>How to handle the response from the server.  Default is\n&#39;text&#39;.  Other values are &#39;json&#39;, &#39;javascript&#39;, and &#39;xml&#39;.</p>\n"
    },
    "method": {
     "!doc": "<p>The HTTP method to use to make the request.  Must be\nuppercase.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Whether to append a cache-busting parameter to the URL.</p>\n"
    },
    "query": {
     "!doc": "<p>Query parameters to append to the URL.</p>\n"
    },
    "timeout": {
     "!doc": "<p>Milliseconds to wait for the response.  If this time\npasses, the then the promise is rejected.</p>\n"
    }
   }
  },
  "dojo/_base/loader": {},
  "dojo/_base/query": {},
  "dojo/_firebug/firebug": {},
  "dojo/back": {
   "addToHistory": {
    "!type": "fn(args: ?)",
    "!doc": "<p>adds a state object (args) to the history list.</p>\n"
   },
   "getHash": {
    "!type": "fn()"
   },
   "goBack": {
    "!type": "fn()",
    "!doc": "<p>private method. Do not call this directly.</p>\n"
   },
   "goForward": {
    "!type": "fn()",
    "!doc": "<p>private method. Do not call this directly.</p>\n"
   },
   "init": {
    "!type": "fn()",
    "!doc": "<p>Initializes the undo stack. This must be called from a <script>\nblock that lives inside the <code>&lt;body&gt;</code> tag to prevent bugs on IE.\n\n</p>\n<p>Only call this method before the page&#39;s DOM is finished loading. Otherwise\nit will not work. Be careful with xdomain loading or djConfig.debugAtAllCosts scenarios,\nin order for this method to work, dojo/back will need to be part of a build layer.</p>\n"
   },
   "setHash": {
    "!type": "fn(h: ?)"
   },
   "setInitialState": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sets the state object and back callback for the very first page\nthat is loaded.\n\n</p>\n<p>It is recommended that you call this method as part of an event\nlistener that is registered via dojo/ready.</p>\n"
   }
  },
  "dojo/behavior": {
   "_behaviors": {},
   "add": {
    "!type": "fn(behaviorObj: ?)",
    "!doc": "<p>Add the specified behavior to the list of behaviors, ignoring existing\nmatches.</p>\n"
   },
   "apply": {
    "!type": "fn()",
    "!doc": "<p>Applies all currently registered behaviors to the document.\n</p>\n"
   }
  },
  "dojo/domReady": {
   "!type": "fn(callback: ?)",
   "!doc": "<p>Plugin to delay require()/define() callback from firing until the DOM has finished loading.</p>\n",
   "_Q": {},
   "load": {
    "!type": "fn(id: ?, req: ?, load: ?)"
   }
  },
  "dojo/cache": {
   "!type": "fn(module: string|?, url: string, value?: string|?)",
   "!doc": "<p>A getter and setter for storing the string content associated with the\nmodule and url arguments.</p>\n"
  },
  "dojo/text": {
   "dynamic": {},
   "load": {
    "!type": "fn(id: string, require: fn(), load: fn())"
   },
   "normalize": {
    "!type": "fn(id: ?, toAbsMid: ?) -> string"
   }
  },
  "dojo/cldr/monetary": {
   "getData": {
    "!type": "fn(code: string)",
    "!doc": "<p>A mapping of currency code to currency-specific formatting information. Returns a unique object with properties: places, round.</p>\n"
   }
  },
  "dojo/cldr/supplemental": {
   "getFirstDayOfWeek": {
    "!type": "fn(locale?: string) -> number",
    "!doc": "<p>Returns a zero-based index for first day of the week</p>\n"
   },
   "getWeekend": {
    "!type": "fn(locale?: string)",
    "!doc": "<p>Returns a hash containing the start and end days of the weekend</p>\n"
   }
  },
  "dojo/i18n": {
   "cache": {},
   "dynamic": {},
   "unitTests": {},
   "getL10nName": {
    "!type": "fn(moduleName: ?, bundleName: ?, locale: ?) -> string"
   },
   "getLocalization": {
    "!type": "fn(moduleName: ?, bundleName: ?, locale: ?)"
   },
   "load": {
    "!type": "fn(id: ?, require: ?, load: ?)",
    "!doc": "<p>id is in one of the following formats\n\n</p>\n<ol>\n<li><p><path>/nls/<bundle>\n =&gt; load the bundle, localized to config.locale; load all bundles localized to\n config.extraLocale (if any); return the loaded bundle localized to config.locale.</p>\n</li>\n<li><p><path>/nls/<locale>/<bundle>\n =&gt; load then return the bundle localized to <locale></p>\n</li>\n<li><p><em>preload</em><path>/nls/<module>*<JSON array of available locales>\n =&gt; for config.locale and all config.extraLocale, load all bundles found\n in the best-matching bundle rollup. A value of 1 is returned, which\n is meaningless other than to say the plugin is executing the requested\n preloads</p>\n</li>\n</ol>\n<p>In cases 1 and 2, <path> is always normalized to an absolute module id upon entry; see\nnormalize. In case 3, it <path> is assumed to be absolute; this is arranged by the builder.\n\n</p>\n<p>To load a bundle means to insert the bundle into the plugin&#39;s cache and publish the bundle\nvalue to the loader. Given <path>, <bundle>, and a particular <locale>, the cache key\n\n</p>\n<pre><code>&lt;path&gt;/nls/&lt;bundle&gt;/&lt;locale&gt;</code></pre>\n<p>will hold the value. Similarly, then plugin will publish this value to the loader by\n\n</p>\n<pre><code>define(&quot;&lt;path&gt;/nls/&lt;bundle&gt;/&lt;locale&gt;&quot;, &lt;bundle-value&gt;);</code></pre>\n<p>Given this algorithm, other machinery can provide fast load paths be preplacing\nvalues in the plugin&#39;s cache, which is public. When a load is demanded the\ncache is inspected before starting any loading. Explicitly placing values in the plugin\ncache is an advanced/experimental feature that should not be needed; use at your own risk.\n\n</p>\n<p>For the normal AMD algorithm, the root bundle is loaded first, which instructs the\nplugin what additional localized bundles are required for a particular locale. These\nadditional locales are loaded and a mix of the root and each progressively-specific\nlocale is returned. For example:\n\n</p>\n<ol>\n<li><p>The client demands &quot;dojo/i18n!some/path/nls/someBundle</p>\n</li>\n<li><p>The loader demands load(some/path/nls/someBundle)</p>\n</li>\n<li><p>This plugin require&#39;s &quot;some/path/nls/someBundle&quot;, which is the root bundle.</p>\n</li>\n<li><p>Assuming config.locale is &quot;ab-cd-ef&quot; and the root bundle indicates that localizations\nare available for &quot;ab&quot; and &quot;ab-cd-ef&quot; (note the missing &quot;ab-cd&quot;, then the plugin\nrequires &quot;some/path/nls/ab/someBundle&quot; and &quot;some/path/nls/ab-cd-ef/someBundle&quot;</p>\n</li>\n<li><p>Upon receiving all required bundles, the plugin constructs the value of the bundle\nab-cd-ef as...</p>\n<pre><code> mixin(mixin(mixin({}, require(&quot;some/path/nls/someBundle&quot;),\n     require(&quot;some/path/nls/ab/someBundle&quot;)),\n     require(&quot;some/path/nls/ab-cd-ef/someBundle&quot;));</code></pre>\n</li>\n</ol>\n<p>This value is inserted into the cache and published to the loader at the\nkey/module-id some/path/nls/someBundle/ab-cd-ef.\n\n</p>\n<p>The special preload signature (case 3) instructs the plugin to stop servicing all normal requests\n(further preload requests will be serviced) until all ongoing preloading has completed.\n\n</p>\n<p>The preload signature instructs the plugin that a special rollup module is available that contains\none or more flattened, localized bundles. The JSON array of available locales indicates which locales\nare available. Here is an example:\n\n</p>\n<pre><code>*preload*some/path/nls/someModule*[&quot;root&quot;, &quot;ab&quot;, &quot;ab-cd-ef&quot;]</code></pre>\n<p>This indicates the following rollup modules are available:\n\n</p>\n<pre><code>some/path/nls/someModule_ROOT\nsome/path/nls/someModule_ab\nsome/path/nls/someModule_ab-cd-ef</code></pre>\n<p>Each of these modules is a normal AMD module that contains one or more flattened bundles in a hash.\nFor example, assume someModule contained the bundles some/bundle/path/someBundle and\nsome/bundle/path/someOtherBundle, then some/path/nls/someModule_ab would be expressed as follows:\n\n</p>\n<pre><code>define({\n    some/bundle/path/someBundle:&lt;value of someBundle, flattened with respect to locale ab&gt;,\n    some/bundle/path/someOtherBundle:&lt;value of someOtherBundle, flattened with respect to locale ab&gt;,\n});</code></pre>\n<p>E.g., given this design, preloading for locale==&quot;ab&quot; can execute the following algorithm:\n\n</p>\n<pre><code>require([&quot;some/path/nls/someModule_ab&quot;], function(rollup){\n    for(var p in rollup){\n        var id = p + &quot;/ab&quot;,\n        cache[id] = rollup[p];\n        define(id, rollup[p]);\n    }\n});</code></pre>\n<p>Similarly, if &quot;ab-cd&quot; is requested, the algorithm can determine that &quot;ab&quot; is the best available and\nload accordingly.\n\n</p>\n<p>The builder will write such rollups for every layer if a non-empty localeList  profile property is\nprovided. Further, the builder will include the following cache entry in the cache associated with\nany layer.\n\n</p>\n<pre><code>&quot;*now&quot;:function(r){r([&#39;dojo/i18n!*preload*&lt;path&gt;/nls/&lt;module&gt;*&lt;JSON array of available locales&gt;&#39;]);}</code></pre>\n<p>The *now special cache module instructs the loader to apply the provided function to context-require\nwith respect to the particular layer being defined. This causes the plugin to hold all normal service\nrequests until all preloading is complete.\n\n</p>\n<p>Notice that this algorithm is rarely better than the standard AMD load algorithm. Consider the normal case\nwhere the target locale has a single segment and a layer depends on a single bundle:\n\n</p>\n<p>Without Preloads:\n\n</p>\n<ol>\n<li>Layer loads root bundle.</li>\n<li>bundle is demanded; plugin loads single localized bundle.</li>\n</ol>\n<p>With Preloads:\n\n</p>\n<ol>\n<li>Layer causes preloading of target bundle.</li>\n<li>bundle is demanded; service is delayed until preloading complete; bundle is returned.</li>\n</ol>\n<p>In each case a single transaction is required to load the target bundle. In cases where multiple bundles\nare required and/or the locale has multiple segments, preloads still requires a single transaction whereas\nthe normal path requires an additional transaction for each additional bundle/locale-segment. However all\nof these additional transactions can be done concurrently. Owing to this analysis, the entire preloading\nalgorithm can be discard during a build by setting the has feature dojo-preload-i18n-Api to false.</p>\n"
   },
   "normalize": {
    "!type": "fn(id: ?, toAbsMid: ?)",
    "!doc": "<p>id may be relative.\npreload has form <code>*preload*&lt;path&gt;/nls/&lt;module&gt;*&lt;flattened locales&gt;</code> and\ntherefore never looks like a relative</p>\n"
   },
   "normalizeLocale": {
    "!type": "fn(locale: ?)"
   }
  },
  "dojo/i18n.cache": {},
  "dojo/colors": {
   "ThreeD": {}
  },
  "dojo/cookie": {
   "!type": "fn(name: string, value?: string, props?: ?)",
   "!doc": "<p>Get or set a cookie.</p>\n",
   "isSupported": {
    "!type": "fn() -> bool",
    "!doc": "<p>Use to determine if the current browser supports cookies or not.\n\n</p>\n<p>Returns true if user allows cookies.\nReturns false if user doesn&#39;t allow cookies.</p>\n"
   }
  },
  "dojo/regexp": {
   "buildGroupRE": {
    "!type": "fn(arr: ?|[], re: fn(), nonCapture?: bool)",
    "!doc": "<p>Builds a regular expression that groups subexpressions</p>\n"
   },
   "escapeString": {
    "!type": "fn(str: string, except?: string)",
    "!doc": "<p>Adds escape sequences for special characters in regular expressions</p>\n"
   },
   "group": {
    "!type": "fn(expression: string, nonCapture?: bool) -> string",
    "!doc": "<p>adds group match to expression</p>\n"
   }
  },
  "dojo/currency": {
   "format": {
    "!type": "fn(value: number, options?: +__FormatOptions)",
    "!doc": "<p>Format a Number as a currency, using locale-specific settings\n</p>\n"
   },
   "parse": {
    "!type": "fn(expression: string, options?: ?)"
   },
   "regexp": {
    "!type": "fn(options?: ?)"
   }
  },
  "dojo/currency.__FormatOptions": {
   "!type": "fn()",
   "prototype": {
    "currency": {
     "!doc": "<p>an <a href=\"http://en.wikipedia.org/wiki/ISO_4217\">ISO4217</a> currency code, a three letter sequence like &quot;USD&quot;.\nFor use with dojo.currency only.</p>\n"
    },
    "fractional": {
     "!doc": "<p>If false, show no decimal places, overriding places and pattern settings.</p>\n"
    },
    "locale": {
     "!doc": "<p>override the locale used to determine formatting rules</p>\n"
    },
    "pattern": {
     "!doc": "<p>override <a href=\"http://www.unicode.org/reports/tr35/#Number_Format_Patterns\">formatting pattern</a>\nwith this string.  Default value is based on locale.  Overriding this property will defeat\nlocalization.  Literal characters in patterns are not supported.</p>\n"
    },
    "places": {
     "!doc": "<p>number of decimal places to show.  Default is defined based on which currency is used.</p>\n"
    },
    "round": {
     "!doc": "<p>5 rounds to nearest .5; 0 rounds to nearest whole (default). -1\nmeans do not round.</p>\n"
    },
    "symbol": {
     "!doc": "<p>localized currency symbol. The default will be looked up in table of supported currencies in <code>dojo.cldr</code>\nA <a href=\"http://en.wikipedia.org/wiki/ISO_4217\">ISO4217</a> currency code will be used if not found.</p>\n"
    },
    "type": {
     "!doc": "<p>Should not be set.  Value is assumed to be &quot;currency&quot;.</p>\n"
    }
   }
  },
  "dojo/currency.__ParseOptions": {
   "!type": "fn()",
   "prototype": {
    "currency": {
     "!doc": "<p>an <a href=\"http://en.wikipedia.org/wiki/ISO_4217\">ISO4217</a> currency code, a three letter sequence like &quot;USD&quot;.\nFor use with dojo.currency only.</p>\n"
    },
    "fractional": {
     "!doc": "<p>Whether to include the fractional portion, where the number of decimal places are implied by the currency\nor explicit &#39;places&#39; parameter.  The value [true,false] makes the fractional portion optional.\nBy default for currencies, it the fractional portion is optional.</p>\n"
    },
    "locale": {
     "!doc": "<p>override the locale used to determine formatting rules</p>\n"
    },
    "pattern": {
     "!doc": "<p>override <a href=\"http://www.unicode.org/reports/tr35/#Number_Format_Patterns\">formatting pattern</a>\nwith this string.  Default value is based on locale.  Overriding this property will defeat\nlocalization.  Literal characters in patterns are not supported.</p>\n"
    },
    "places": {
     "!doc": "<p>fixed number of decimal places to accept.  The default is determined based on which currency is used.</p>\n"
    },
    "strict": {
     "!doc": "<p>strict parsing, false by default.  Strict parsing requires input as produced by the format() method.\nNon-strict is more permissive, e.g. flexible on white space, omitting thousands separators</p>\n"
    },
    "symbol": {
     "!doc": "<p>localized currency symbol. The default will be looked up in table of supported currencies in <code>dojo.cldr</code>\nA <a href=\"http://en.wikipedia.org/wiki/ISO_4217\">ISO4217</a> currency code will be used if not found.</p>\n"
    },
    "type": {
     "!doc": "<p>Should not be set.  Value is assumed to be currency.</p>\n"
    }
   }
  },
  "dojo/number": {
   "_numberPatternRE": {},
   "format": {
    "!type": "fn(value: number, options?: ?)",
    "!doc": "<p>Format a Number as a String, using locale-specific settings</p>\n"
   },
   "parse": {
    "!type": "fn(expression: string, options?: ?) -> number",
    "!doc": "<p>Convert a properly formatted string to a primitive Number, using\nlocale-specific settings.</p>\n"
   },
   "regexp": {
    "!type": "fn(options?: ?)",
    "!doc": "<p>Builds the regular needed to parse a number</p>\n"
   },
   "round": {
    "!type": "fn(value: number, places?: number, increment?: number) -> number",
    "!doc": "<p>Rounds to the nearest value with the given number of decimal places, away from zero</p>\n"
   }
  },
  "dojo/number.__FormatOptions": {
   "!type": "fn()",
   "prototype": {
    "fractional": {
     "!doc": "<p>If false, show no decimal places, overriding places and pattern settings.</p>\n"
    },
    "locale": {
     "!doc": "<p>override the locale used to determine formatting rules</p>\n"
    },
    "pattern": {
     "!doc": "<p>override <a href=\"http://www.unicode.org/reports/tr35/#Number_Format_Patterns\">formatting pattern</a>\nwith this string.  Default value is based on locale.  Overriding this property will defeat\nlocalization.  Literal characters in patterns are not supported.</p>\n"
    },
    "places": {
     "!doc": "<p>fixed number of decimal places to show.  This overrides any\ninformation in the provided pattern.</p>\n"
    },
    "round": {
     "!doc": "<p>5 rounds to nearest .5; 0 rounds to nearest whole (default). -1\nmeans do not round.</p>\n"
    },
    "type": {
     "!doc": "<p>choose a format type based on the locale from the following:\ndecimal, scientific (not yet supported), percent, currency. decimal by default.</p>\n"
    }
   }
  },
  "dojo/number.__FormatAbsoluteOptions": {
   "!type": "fn()",
   "prototype": {
    "decimal": {
     "!doc": "<p>the decimal separator</p>\n"
    },
    "group": {
     "!doc": "<p>the group separator</p>\n"
    },
    "places": {
     "!doc": "<p>number of decimal places.  the range &quot;n,m&quot; will format to m places.</p>\n"
    },
    "round": {
     "!doc": "<p>5 rounds to nearest .5; 0 rounds to nearest whole (default). -1\nmeans don&#39;t round.</p>\n"
    }
   }
  },
  "dojo/number.__RegexpOptions": {
   "!type": "fn()",
   "prototype": {
    "locale": {
     "!doc": "<p>override the locale used to determine formatting rules</p>\n"
    },
    "pattern": {
     "!doc": "<p>override <a href=\"http://www.unicode.org/reports/tr35/#Number_Format_Patterns\">formatting pattern</a>\nwith this string.  Default value is based on locale.  Overriding this property will defeat\nlocalization.</p>\n"
    },
    "places": {
     "!doc": "<p>number of decimal places to accept: Infinity, a positive number, or\na range &quot;n,m&quot;.  Defined by pattern or Infinity if pattern not provided.</p>\n"
    },
    "strict": {
     "!doc": "<p>strict parsing, false by default.  Strict parsing requires input as produced by the format() method.\nNon-strict is more permissive, e.g. flexible on white space, omitting thousands separators</p>\n"
    },
    "type": {
     "!doc": "<p>choose a format type based on the locale from the following:\ndecimal, scientific (not yet supported), percent, currency. decimal by default.</p>\n"
    }
   }
  },
  "dojo/number.__ParseOptions": {
   "!type": "fn()",
   "prototype": {
    "fractional": {
     "!doc": "<p>Whether to include the fractional portion, where the number of decimal places are implied by pattern\nor explicit &#39;places&#39; parameter.  The value [true,false] makes the fractional portion optional.</p>\n"
    },
    "locale": {
     "!doc": "<p>override the locale used to determine formatting rules</p>\n"
    },
    "pattern": {
     "!doc": "<p>override <a href=\"http://www.unicode.org/reports/tr35/#Number_Format_Patterns\">formatting pattern</a>\nwith this string.  Default value is based on locale.  Overriding this property will defeat\nlocalization.  Literal characters in patterns are not supported.</p>\n"
    },
    "strict": {
     "!doc": "<p>strict parsing, false by default.  Strict parsing requires input as produced by the format() method.\nNon-strict is more permissive, e.g. flexible on white space, omitting thousands separators</p>\n"
    },
    "type": {
     "!doc": "<p>choose a format type based on the locale from the following:\ndecimal, scientific (not yet supported), percent, currency. decimal by default.</p>\n"
    }
   }
  },
  "dojo/number.__RealNumberRegexpFlags": {
   "!type": "fn()",
   "prototype": {
    "decimal": {
     "!doc": "<p>A string for the character used as the decimal point.  Default\nis &quot;.&quot;.</p>\n"
    },
    "eSigned": {
     "!doc": "<p>The leading plus-or-minus sign on the exponent.  Can be true,\nfalse, or [true, false].  Default is [true, false], (i.e. will\nmatch if it is signed or unsigned).  flags in regexp.integer can be\napplied.</p>\n"
    },
    "exponent": {
     "!doc": "<p>Express in exponential notation.  Can be true, false, or [true,\nfalse]. Default is [true, false], (i.e. will match if the\nexponential part is present are not).</p>\n"
    },
    "fractional": {
     "!doc": "<p>Whether decimal places are used.  Can be true, false, or [true,\nfalse].  Default is [true, false] which means optional.</p>\n"
    },
    "places": {
     "!doc": "<p>The integer number of decimal places or a range given as &quot;n,m&quot;.  If\nnot given, the decimal part is optional and the number of places is\nunlimited.</p>\n"
    }
   }
  },
  "dojo/number.__IntegerRegexpFlags": {
   "!type": "fn()",
   "prototype": {
    "groupSize": {
     "!doc": "<p>group size between separators</p>\n"
    },
    "groupSize2": {
     "!doc": "<p>second grouping, where separators 2..n have a different interval than the first separator (for India)</p>\n"
    },
    "separator": {
     "!doc": "<p>The character used as the thousands separator. Default is no\nseparator. For more than one symbol use an array, e.g. <code>[&quot;,&quot;, &quot;&quot;]</code>,\nmakes &#39;,&#39; optional.</p>\n"
    },
    "signed": {
     "!doc": "<p>The leading plus-or-minus sign. Can be true, false, or <code>[true,false]</code>.\nDefault is <code>[true, false]</code>, (i.e. will match if it is signed\nor unsigned).</p>\n"
    }
   }
  },
  "dojo/string": {
   "escape": {
    "!type": "fn(str: string) -> string",
    "!doc": "<p>Efficiently escape a string for insertion into HTML (innerHTML or attributes), replacing &amp;, &lt;, &gt;, &quot;, &#39;, and / characters.</p>\n"
   },
   "pad": {
    "!type": "fn(text: string, size: +Integer, ch?: string, end?: bool) -> number",
    "!doc": "<p>Pad a string to guarantee that it is at least <code>size</code> length by\nfilling with the character <code>ch</code> at either the start or end of the\nstring. Pads at the start, by default.</p>\n"
   },
   "rep": {
    "!type": "fn(str: string, num: +Integer) -> string",
    "!doc": "<p>Efficiently replicate a string <code>n</code> times.</p>\n"
   },
   "substitute": {
    "!type": "fn(template: string, map: ?|[], transform?: fn(), thisObject?: ?)",
    "!doc": "<p>Performs parameterized substitutions on a string. Throws an\nexception if any parameter is unmatched.</p>\n"
   },
   "trim": {
    "!type": "fn(str: string) -> string",
    "!doc": "<p>Trims whitespace from both sides of the string</p>\n"
   }
  },
  "dojo/data/ItemFileReadStore": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/Evented.prototype",
    "_ccUrl": {},
    "clearOnClose": {
     "!doc": "<p>Parameter to allow users to specify if a close call should force a reload or not.\nBy default, it retains the old behavior of not clearing if close is called.  But\nif set true, the store will be reset to default state.  Note that by doing this,\nall item handles will become invalid and a new fetch must be issued.</p>\n"
    },
    "data": {},
    "failOk": {
     "!doc": "<p>Parameter for specifying that it is OK for the xhrGet call to fail silently.</p>\n"
    },
    "hierarchical": {
     "!doc": "<p>Parameter to indicate to process data from the url as hierarchical\n(data items can contain other data items in js form).  Default is true\nfor backwards compatibility.  False means only root items are processed\nas items, all child objects outside of type-mapped objects and those in\nspecific reference format, are left straight JS data objects.</p>\n"
    },
    "typeMap": {},
    "url": {},
    "urlPreventCache": {
     "!doc": "<p>Parameter to allow specifying if preventCache should be passed to the xhrGet call or not when loading data from a url.\nNote this does not mean the store calls the server on each fetch, only that the data load has preventCache set as an option.\nAdded for tracker: #6072</p>\n"
    },
    "close": {
     "!type": "fn(request?: +dojo|?)",
     "!doc": "<p>See dojo/data/api/Read.close()</p>\n"
    },
    "containsValue": {
     "!type": "fn(item: +dojo, attribute: ?, value: +anything)",
     "!doc": "<p>See dojo/data/api/Read.containsValue()</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "errorHandler": {
     "!type": "fn(errorData: ?, requestObject: ?)",
     "!doc": "<p>The error handler when there is an error fetching items.  This function should not be called\ndirectly and is used by simpleFetch.fetch().</p>\n"
    },
    "fetch": {
     "!type": "fn(request?: ?)",
     "!doc": "<p>The simpleFetch mixin is designed to serve as a set of function(s) that can\nbe mixed into other datastore implementations to accelerate their development.</p>\n"
    },
    "fetchHandler": {
     "!type": "fn(items: [], requestObject: ?)",
     "!doc": "<p>The handler when items are successfully fetched.  This function should not be called directly\nand is used by simpleFetch.fetch().</p>\n"
    },
    "fetchItemByIdentity": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>See dojo/data/api/Identity.fetchItemByIdentity()</p>\n"
    },
    "filter": {
     "!type": "fn(requestArgs: ?, arrayOfItems: [+item], findCallback: fn())",
     "!doc": "<p>This method handles the basic filtering needs for ItemFile* based stores.</p>\n"
    },
    "getAttributes": {
     "!type": "fn(item: +dojo) -> []",
     "!doc": "<p>See dojo/data/api/Read.getAttributes()</p>\n"
    },
    "getFeatures": {
     "!type": "fn()",
     "!doc": "<p>See dojo/data/api/Read.getFeatures()</p>\n"
    },
    "getIdentity": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>See dojo/data/api/Identity.getIdentity()</p>\n"
    },
    "getIdentityAttributes": {
     "!type": "fn(item: +dojo) -> []",
     "!doc": "<p>See dojo/data/api/Identity.getIdentityAttributes()</p>\n"
    },
    "getLabel": {
     "!type": "fn(item: +dojo) -> number",
     "!doc": "<p>See dojo/data/api/Read.getLabel()</p>\n"
    },
    "getLabelAttributes": {
     "!type": "fn(item: +dojo) -> []",
     "!doc": "<p>See dojo/data/api/Read.getLabelAttributes()</p>\n"
    },
    "getValue": {
     "!type": "fn(item: +dojo, attribute: ?, defaultValue?: +value) -> +value",
     "!doc": "<p>See dojo/data/api/Read.getValue()</p>\n"
    },
    "getValues": {
     "!type": "fn(item: +dojo, attribute: ?)",
     "!doc": "<p>See dojo/data/api/Read.getValues()</p>\n"
    },
    "hasAttribute": {
     "!type": "fn(item: +dojo, attribute: ?) -> bool",
     "!doc": "<p>See dojo/data/api/Read.hasAttribute()</p>\n"
    },
    "isItem": {
     "!type": "fn(something: +anything) -> bool",
     "!doc": "<p>See dojo/data/api/Read.isItem()</p>\n"
    },
    "isItemLoaded": {
     "!type": "fn(something: +anything)",
     "!doc": "<p>See dojo/data/api/Read.isItemLoaded()</p>\n"
    },
    "loadItem": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>See dojo/data/api/Read.loadItem()</p>\n"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    }
   }
  },
  "dojo/data/util/filter": {
   "patternToRegExp": {
    "!type": "fn(pattern: string, ignoreCase?: bool)",
    "!doc": "<p>Helper function to convert a simple pattern to a regular expression for matching.</p>\n"
   }
  },
  "dojo/data/util/simpleFetch": {
   "errorHandler": {
    "!type": "fn(errorData: ?, requestObject: ?)",
    "!doc": "<p>The error handler when there is an error fetching items.  This function should not be called\ndirectly and is used by simpleFetch.fetch().</p>\n"
   },
   "fetch": {
    "!type": "fn(request?: ?)",
    "!doc": "<p>The simpleFetch mixin is designed to serve as a set of function(s) that can\nbe mixed into other datastore implementations to accelerate their development.</p>\n"
   },
   "fetchHandler": {
    "!type": "fn(items: [], requestObject: ?)",
    "!doc": "<p>The handler when items are successfully fetched.  This function should not be called directly\nand is used by simpleFetch.fetch().</p>\n"
   }
  },
  "dojo/data/util/sorter": {
   "basicComparator": {
    "!type": "fn(a: +anything, b: +anything) -> number",
    "!doc": "<p>Basic comparison function that compares if an item is greater or less than another item</p>\n"
   },
   "createSortFunction": {
    "!type": "fn(sortSpec: [+attributes], store: +dojo)",
    "!doc": "<p>Helper function to generate the sorting function based off the list of sort attributes.</p>\n"
   }
  },
  "dojo/data/ItemFileWriteStore": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/data/ItemFileReadStore.prototype",
    "_ccUrl": {},
    "clearOnClose": {
     "!doc": "<p>Parameter to allow users to specify if a close call should force a reload or not.\nBy default, it retains the old behavior of not clearing if close is called.  But\nif set true, the store will be reset to default state.  Note that by doing this,\nall item handles will become invalid and a new fetch must be issued.</p>\n"
    },
    "data": {},
    "failOk": {
     "!doc": "<p>Parameter for specifying that it is OK for the xhrGet call to fail silently.</p>\n"
    },
    "hierarchical": {
     "!doc": "<p>Parameter to indicate to process data from the url as hierarchical\n(data items can contain other data items in js form).  Default is true\nfor backwards compatibility.  False means only root items are processed\nas items, all child objects outside of type-mapped objects and those in\nspecific reference format, are left straight JS data objects.</p>\n"
    },
    "referenceIntegrity": {},
    "typeMap": {},
    "url": {},
    "urlPreventCache": {
     "!doc": "<p>Parameter to allow specifying if preventCache should be passed to the xhrGet call or not when loading data from a url.\nNote this does not mean the store calls the server on each fetch, only that the data load has preventCache set as an option.\nAdded for tracker: #6072</p>\n"
    },
    "close": {
     "!type": "fn(request?: ?)",
     "!doc": "<p>Over-ride of base close function of ItemFileReadStore to add in check for store state.</p>\n"
    },
    "containsValue": {
     "!type": "fn(item: +dojo, attribute: ?, value: +anything)",
     "!doc": "<p>See dojo/data/api/Read.containsValue()</p>\n"
    },
    "deleteItem": {
     "!type": "fn(item: +dojo) -> bool",
     "!doc": "<p>See dojo/data/api/Write.deleteItem()</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "errorHandler": {
     "!type": "fn(errorData: ?, requestObject: ?)",
     "!doc": "<p>The error handler when there is an error fetching items.  This function should not be called\ndirectly and is used by simpleFetch.fetch().</p>\n"
    },
    "fetch": {
     "!type": "fn(request?: ?)",
     "!doc": "<p>The simpleFetch mixin is designed to serve as a set of function(s) that can\nbe mixed into other datastore implementations to accelerate their development.</p>\n"
    },
    "fetchHandler": {
     "!type": "fn(items: [], requestObject: ?)",
     "!doc": "<p>The handler when items are successfully fetched.  This function should not be called directly\nand is used by simpleFetch.fetch().</p>\n"
    },
    "fetchItemByIdentity": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>See dojo/data/api/Identity.fetchItemByIdentity()</p>\n"
    },
    "filter": {
     "!type": "fn(requestArgs: ?, arrayOfItems: [+item], findCallback: fn())",
     "!doc": "<p>This method handles the basic filtering needs for ItemFile* based stores.</p>\n"
    },
    "getAttributes": {
     "!type": "fn(item: +dojo) -> []",
     "!doc": "<p>See dojo/data/api/Read.getAttributes()</p>\n"
    },
    "getFeatures": {
     "!type": "fn()",
     "!doc": "<p>See dojo/data/api/Read.getFeatures()</p>\n"
    },
    "getIdentity": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>See dojo/data/api/Identity.getIdentity()</p>\n"
    },
    "getIdentityAttributes": {
     "!type": "fn(item: +dojo) -> []",
     "!doc": "<p>See dojo/data/api/Identity.getIdentityAttributes()</p>\n"
    },
    "getLabel": {
     "!type": "fn(item: +dojo) -> number",
     "!doc": "<p>See dojo/data/api/Read.getLabel()</p>\n"
    },
    "getLabelAttributes": {
     "!type": "fn(item: +dojo) -> []",
     "!doc": "<p>See dojo/data/api/Read.getLabelAttributes()</p>\n"
    },
    "getValue": {
     "!type": "fn(item: +dojo, attribute: ?, defaultValue?: +value) -> +value",
     "!doc": "<p>See dojo/data/api/Read.getValue()</p>\n"
    },
    "getValues": {
     "!type": "fn(item: +dojo, attribute: ?)",
     "!doc": "<p>See dojo/data/api/Read.getValues()</p>\n"
    },
    "hasAttribute": {
     "!type": "fn(item: +dojo, attribute: ?) -> bool",
     "!doc": "<p>See dojo/data/api/Read.hasAttribute()</p>\n"
    },
    "isDirty": {
     "!type": "fn(item?: +item) -> bool",
     "!doc": "<p>See dojo/data/api/Write.isDirty()</p>\n"
    },
    "isItem": {
     "!type": "fn(something: +anything) -> bool",
     "!doc": "<p>See dojo/data/api/Read.isItem()</p>\n"
    },
    "isItemLoaded": {
     "!type": "fn(something: +anything)",
     "!doc": "<p>See dojo/data/api/Read.isItemLoaded()</p>\n"
    },
    "loadItem": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>See dojo/data/api/Read.loadItem()</p>\n"
    },
    "newItem": {
     "!type": "fn(keywordArgs?: ?, parentInfo?: ?)",
     "!doc": "<p>See dojo/data/api/Write.newItem()</p>\n"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    },
    "revert": {
     "!type": "fn() -> bool",
     "!doc": "<p>See dojo/data/api/Write.revert()</p>\n"
    },
    "save": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>See dojo/data/api/Write.save()</p>\n"
    },
    "setValue": {
     "!type": "fn(item: +dojo, attribute: ?, value: ?)",
     "!doc": "<p>See dojo/data/api/Write.set()</p>\n"
    },
    "setValues": {
     "!type": "fn(item: +dojo, attribute: ?, values: [])",
     "!doc": "<p>See dojo/data/api/Write.setValues()</p>\n"
    },
    "unsetAttribute": {
     "!type": "fn(item: +dojo, attribute: ?)",
     "!doc": "<p>See dojo/data/api/Write.unsetAttribute()</p>\n"
    }
   }
  },
  "dojo/data/ObjectStore": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/Evented.prototype",
    "labelProperty": {},
    "objectStore": {},
    "changing": {
     "!type": "fn(object: ?, _deleting: bool)",
     "!doc": "<p>adds an object to the list of dirty objects.  This object\ncontains a reference to the object itself as well as a\ncloned and trimmed version of old object for use with\nrevert.</p>\n"
    },
    "close": {
     "!type": "fn(request: ?)",
     "!doc": "<p>See dojo/data/api/Read.close()</p>\n"
    },
    "containsValue": {
     "!type": "fn(item: ?, attribute: string, value: +Anything) -> bool",
     "!doc": "<p>Checks to see if &#39;item&#39; has &#39;value&#39; at &#39;attribute&#39;</p>\n"
    },
    "deleteItem": {
     "!type": "fn(item: ?)",
     "!doc": "<p>deletes item and any references to that item from the store.</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "fetch": {
     "!type": "fn(args: ?)",
     "!doc": "<p>See dojo/data/api/Read.fetch()</p>\n"
    },
    "fetchItemByIdentity": {
     "!type": "fn(args: ?)",
     "!doc": "<p>fetch an item by its identity, by looking in our index of what we have loaded</p>\n"
    },
    "getAttributes": {
     "!type": "fn(item: ?) -> []",
     "!doc": "<p>Gets the available attributes of an item&#39;s &#39;property&#39; and returns\nit as an array.</p>\n"
    },
    "getFeatures": {
     "!type": "fn()",
     "!doc": "<p>return the store feature set</p>\n"
    },
    "getIdentity": {
     "!type": "fn(item: ?)",
     "!doc": "<p>returns the identity of the given item\nSee dojo/data/api/Read.getIdentity()</p>\n"
    },
    "getIdentityAttributes": {
     "!type": "fn(item: ?) -> []",
     "!doc": "<p>returns the attributes which are used to make up the\nidentity of an item.    Basically returns this.objectStore.idProperty\nSee dojo/data/api/Read.getIdentityAttributes()</p>\n"
    },
    "getLabel": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>See dojo/data/api/Read.getLabel()</p>\n"
    },
    "getLabelAttributes": {
     "!type": "fn(item: +dojo) -> []",
     "!doc": "<p>See dojo/data/api/Read.getLabelAttributes()</p>\n"
    },
    "getValue": {
     "!type": "fn(item: ?, property: string, defaultValue?: +value) -> +value",
     "!doc": "<p>Gets the value of an item&#39;s &#39;property&#39;</p>\n"
    },
    "getValues": {
     "!type": "fn(item: ?, property: string) -> []",
     "!doc": "<p>Gets the value of an item&#39;s &#39;property&#39; and returns\nit. If this value is an array it is just returned,\nif not, the value is added to an array and that is returned.</p>\n"
    },
    "hasAttribute": {
     "!type": "fn(item: ?, attribute: string) -> bool",
     "!doc": "<p>Checks to see if item has attribute</p>\n"
    },
    "isDirty": {
     "!type": "fn(item: ?) -> bool",
     "!doc": "<p>returns true if the item is marked as dirty or true if there are any dirty items</p>\n"
    },
    "isItem": {
     "!type": "fn(item: ?) -> bool",
     "!doc": "<p>Checks to see if the argument is an item</p>\n"
    },
    "isItemLoaded": {
     "!type": "fn(item: ?)",
     "!doc": "<p>Checks to see if the item is loaded.</p>\n"
    },
    "loadItem": {
     "!type": "fn(args: ?)",
     "!doc": "<p>Loads an item and calls the callback handler. Note, that this will call the callback\nhandler even if the item is loaded. Consequently, you can use loadItem to ensure\nthat an item is loaded is situations when the item may or may not be loaded yet.\nIf you access a value directly through property access, you can use this to load\na lazy value as well (doesn&#39;t need to be an item).</p>\n"
    },
    "newItem": {
     "!type": "fn(data: ?, parentInfo: ?)",
     "!doc": "<p>adds a new item to the store at the specified point.\nTakes two parameters, data, and options.</p>\n"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    },
    "revert": {
     "!type": "fn()",
     "!doc": "<p>returns any modified data to its original state prior to a save();</p>\n"
    },
    "save": {
     "!type": "fn(kwArgs: ?)",
     "!doc": "<p>Saves the dirty data using object store provider. See dojo/data/api/Write for API.</p>\n"
    },
    "setValue": {
     "!type": "fn(item: ?, attribute: ?, value: ?)",
     "!doc": "<p>sets &#39;attribute&#39; on &#39;item&#39; to &#39;value&#39;\nSee dojo/data/api/Write.setValue()</p>\n"
    },
    "setValues": {
     "!type": "fn(item: ?, attribute: ?, values: ?)",
     "!doc": "<p>sets &#39;attribute&#39; on &#39;item&#39; to &#39;value&#39; value\nmust be an array.\nSee dojo/data/api/Write.setValues()</p>\n"
    },
    "unsetAttribute": {
     "!type": "fn(item: ?, attribute: ?)",
     "!doc": "<p>unsets &#39;attribute&#39; on &#39;item&#39;\nSee dojo/data/api/Write.unsetAttribute()</p>\n"
    }
   }
  },
  "dojo/data/api/Identity": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/data/api/Read.prototype",
    "close": {
     "!type": "fn(request?: +dojo|?)",
     "!doc": "<p>The close() method is intended for instructing the store to &#39;close&#39; out\nany information associated with a particular request.</p>\n"
    },
    "containsValue": {
     "!type": "fn(item: +dojo, attribute: ?, value: +anything)",
     "!doc": "<p>Returns true if the given <em>value</em> is one of the values that getValues()\nwould return.</p>\n"
    },
    "fetch": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>Given a query and set of defined options, such as a start and count of items to return,\nthis method executes the query and makes the results available as data items.\nThe format and expectations of stores is that they operate in a generally asynchronous\nmanner, therefore callbacks are always used to return items located by the fetch parameters.</p>\n"
    },
    "fetchItemByIdentity": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>Given the identity of an item, this method returns the item that has\nthat identity through the onItem callback.  Conforming implementations\nshould return null if there is no item with the given identity.\nImplementations of fetchItemByIdentity() may sometimes return an item\nfrom a local cache and may sometimes fetch an item from a remote server,</p>\n"
    },
    "getAttributes": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Returns an array with all the attributes that this item has.  This\nmethod will always return an array; if the item has no attributes\nat all, getAttributes() will return an empty array: [].</p>\n"
    },
    "getFeatures": {
     "!type": "fn()",
     "!doc": "<p>See dojo/data/api/Read.getFeatures()</p>\n"
    },
    "getIdentity": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Returns a unique identifier for an item.  The return value will be\neither a string or something that has a toString() method (such as,\nfor example, a dojox/uuid object).</p>\n"
    },
    "getIdentityAttributes": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Returns an array of attribute names that are used to generate the identity.\nFor most stores, this is a single attribute, but for some complex stores\nsuch as RDB backed stores that use compound (multi-attribute) identifiers\nit can be more than one.  If the identity is not composed of attributes\non the item, it will return null.  This function is intended to identify\nthe attributes that comprise the identity so that so that during a render\nof all attributes, the UI can hide the the identity information if it\nchooses.</p>\n"
    },
    "getLabel": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Method to inspect the item and return a user-readable &#39;label&#39; for the item\nthat provides a general/adequate description of what the item is.</p>\n"
    },
    "getLabelAttributes": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Method to inspect the item and return an array of what attributes of the item were used\nto generate its label, if any.</p>\n"
    },
    "getValue": {
     "!type": "fn(item: +dojo, attribute: ?, defaultValue?: +value)",
     "!doc": "<p>Returns a single attribute value.\nReturns defaultValue if and only if <em>item</em> does not have a value for <em>attribute</em>.\nReturns null if and only if null was explicitly set as the attribute value.\nReturns undefined if and only if the item does not have a value for the\ngiven attribute (which is the same as saying the item does not have the attribute).</p>\n"
    },
    "getValues": {
     "!type": "fn(item: +dojo, attribute: ?)",
     "!doc": "<p>This getValues() method works just like the getValue() method, but getValues()\nalways returns an array rather than a single attribute value.  The array\nmay be empty, may contain a single attribute value, or may contain\nmany attribute values.\nIf the item does not have a value for the given attribute, then getValues()\nwill return an empty array: [].  (So, if store.hasAttribute(item, attribute)\nhas a return of false, then store.getValues(item, attribute) will return [].)</p>\n"
    },
    "hasAttribute": {
     "!type": "fn(item: +dojo, attribute: ?)",
     "!doc": "<p>Returns true if the given <em>item</em> has a value for the given <em>attribute</em>.</p>\n"
    },
    "isItem": {
     "!type": "fn(something: +anything)",
     "!doc": "<p>Returns true if <em>something</em> is an item and came from the store instance.\nReturns false if <em>something</em> is a literal, an item from another store instance,\nor is any object other than an item.</p>\n"
    },
    "isItemLoaded": {
     "!type": "fn(something: +anything)",
     "!doc": "<p>Returns false if isItem(something) is false.  Returns false if\nif isItem(something) is true but the the item is not yet loaded\nin local memory (for example, if the item has not yet been read\nfrom the server).</p>\n"
    },
    "loadItem": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>Given an item, this method loads the item so that a subsequent call\nto store.isItemLoaded(item) will return true.  If a call to\nisItemLoaded() returns true before loadItem() is even called,\nthen loadItem() need not do any work at all and will not even invoke\nthe callback handlers.  So, before invoking this method, check that\nthe item has not already been loaded.</p>\n"
    }
   }
  },
  "dojo/data/api/Read": {
   "!type": "fn()",
   "prototype": {
    "close": {
     "!type": "fn(request?: +dojo|?)",
     "!doc": "<p>The close() method is intended for instructing the store to &#39;close&#39; out\nany information associated with a particular request.</p>\n"
    },
    "containsValue": {
     "!type": "fn(item: +dojo, attribute: ?, value: +anything)",
     "!doc": "<p>Returns true if the given <em>value</em> is one of the values that getValues()\nwould return.</p>\n"
    },
    "fetch": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>Given a query and set of defined options, such as a start and count of items to return,\nthis method executes the query and makes the results available as data items.\nThe format and expectations of stores is that they operate in a generally asynchronous\nmanner, therefore callbacks are always used to return items located by the fetch parameters.</p>\n"
    },
    "getAttributes": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Returns an array with all the attributes that this item has.  This\nmethod will always return an array; if the item has no attributes\nat all, getAttributes() will return an empty array: [].</p>\n"
    },
    "getFeatures": {
     "!type": "fn()",
     "!doc": "<p>The getFeatures() method returns an simple keyword values object\nthat specifies what interface features the datastore implements.\nA simple CsvStore may be read-only, and the only feature it\nimplements will be the &#39;dojo/data/api/Read&#39; interface, so the\ngetFeatures() method will return an object like this one:\n{&#39;dojo.data.api.Read&#39;: true}.\nA more sophisticated datastore might implement a variety of\ninterface features, like &#39;dojo.data.api.Read&#39;, &#39;dojo/data/api/Write&#39;,\n&#39;dojo.data.api.Identity&#39;, and &#39;dojo/data/api/Attribution&#39;.</p>\n"
    },
    "getLabel": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Method to inspect the item and return a user-readable &#39;label&#39; for the item\nthat provides a general/adequate description of what the item is.</p>\n"
    },
    "getLabelAttributes": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Method to inspect the item and return an array of what attributes of the item were used\nto generate its label, if any.</p>\n"
    },
    "getValue": {
     "!type": "fn(item: +dojo, attribute: ?, defaultValue?: +value)",
     "!doc": "<p>Returns a single attribute value.\nReturns defaultValue if and only if <em>item</em> does not have a value for <em>attribute</em>.\nReturns null if and only if null was explicitly set as the attribute value.\nReturns undefined if and only if the item does not have a value for the\ngiven attribute (which is the same as saying the item does not have the attribute).</p>\n"
    },
    "getValues": {
     "!type": "fn(item: +dojo, attribute: ?)",
     "!doc": "<p>This getValues() method works just like the getValue() method, but getValues()\nalways returns an array rather than a single attribute value.  The array\nmay be empty, may contain a single attribute value, or may contain\nmany attribute values.\nIf the item does not have a value for the given attribute, then getValues()\nwill return an empty array: [].  (So, if store.hasAttribute(item, attribute)\nhas a return of false, then store.getValues(item, attribute) will return [].)</p>\n"
    },
    "hasAttribute": {
     "!type": "fn(item: +dojo, attribute: ?)",
     "!doc": "<p>Returns true if the given <em>item</em> has a value for the given <em>attribute</em>.</p>\n"
    },
    "isItem": {
     "!type": "fn(something: +anything)",
     "!doc": "<p>Returns true if <em>something</em> is an item and came from the store instance.\nReturns false if <em>something</em> is a literal, an item from another store instance,\nor is any object other than an item.</p>\n"
    },
    "isItemLoaded": {
     "!type": "fn(something: +anything)",
     "!doc": "<p>Returns false if isItem(something) is false.  Returns false if\nif isItem(something) is true but the the item is not yet loaded\nin local memory (for example, if the item has not yet been read\nfrom the server).</p>\n"
    },
    "loadItem": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>Given an item, this method loads the item so that a subsequent call\nto store.isItemLoaded(item) will return true.  If a call to\nisItemLoaded() returns true before loadItem() is even called,\nthen loadItem() need not do any work at all and will not even invoke\nthe callback handlers.  So, before invoking this method, check that\nthe item has not already been loaded.</p>\n"
    }
   }
  },
  "dojo/data/api/Item": {
   "!type": "fn()",
   "prototype": {}
  },
  "dojo/data/api/Notification": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/data/api/Read.prototype",
    "close": {
     "!type": "fn(request?: +dojo|?)",
     "!doc": "<p>The close() method is intended for instructing the store to &#39;close&#39; out\nany information associated with a particular request.</p>\n"
    },
    "containsValue": {
     "!type": "fn(item: +dojo, attribute: ?, value: +anything)",
     "!doc": "<p>Returns true if the given <em>value</em> is one of the values that getValues()\nwould return.</p>\n"
    },
    "fetch": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>Given a query and set of defined options, such as a start and count of items to return,\nthis method executes the query and makes the results available as data items.\nThe format and expectations of stores is that they operate in a generally asynchronous\nmanner, therefore callbacks are always used to return items located by the fetch parameters.</p>\n"
    },
    "getAttributes": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Returns an array with all the attributes that this item has.  This\nmethod will always return an array; if the item has no attributes\nat all, getAttributes() will return an empty array: [].</p>\n"
    },
    "getFeatures": {
     "!type": "fn()",
     "!doc": "<p>See dojo/data/api/Read.getFeatures()</p>\n"
    },
    "getLabel": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Method to inspect the item and return a user-readable &#39;label&#39; for the item\nthat provides a general/adequate description of what the item is.</p>\n"
    },
    "getLabelAttributes": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Method to inspect the item and return an array of what attributes of the item were used\nto generate its label, if any.</p>\n"
    },
    "getValue": {
     "!type": "fn(item: +dojo, attribute: ?, defaultValue?: +value)",
     "!doc": "<p>Returns a single attribute value.\nReturns defaultValue if and only if <em>item</em> does not have a value for <em>attribute</em>.\nReturns null if and only if null was explicitly set as the attribute value.\nReturns undefined if and only if the item does not have a value for the\ngiven attribute (which is the same as saying the item does not have the attribute).</p>\n"
    },
    "getValues": {
     "!type": "fn(item: +dojo, attribute: ?)",
     "!doc": "<p>This getValues() method works just like the getValue() method, but getValues()\nalways returns an array rather than a single attribute value.  The array\nmay be empty, may contain a single attribute value, or may contain\nmany attribute values.\nIf the item does not have a value for the given attribute, then getValues()\nwill return an empty array: [].  (So, if store.hasAttribute(item, attribute)\nhas a return of false, then store.getValues(item, attribute) will return [].)</p>\n"
    },
    "hasAttribute": {
     "!type": "fn(item: +dojo, attribute: ?)",
     "!doc": "<p>Returns true if the given <em>item</em> has a value for the given <em>attribute</em>.</p>\n"
    },
    "isItem": {
     "!type": "fn(something: +anything)",
     "!doc": "<p>Returns true if <em>something</em> is an item and came from the store instance.\nReturns false if <em>something</em> is a literal, an item from another store instance,\nor is any object other than an item.</p>\n"
    },
    "isItemLoaded": {
     "!type": "fn(something: +anything)",
     "!doc": "<p>Returns false if isItem(something) is false.  Returns false if\nif isItem(something) is true but the the item is not yet loaded\nin local memory (for example, if the item has not yet been read\nfrom the server).</p>\n"
    },
    "loadItem": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>Given an item, this method loads the item so that a subsequent call\nto store.isItemLoaded(item) will return true.  If a call to\nisItemLoaded() returns true before loadItem() is even called,\nthen loadItem() need not do any work at all and will not even invoke\nthe callback handlers.  So, before invoking this method, check that\nthe item has not already been loaded.</p>\n"
    }
   }
  },
  "dojo/data/api/Request": {
   "!type": "fn()",
   "prototype": {
    "abort": {
     "!type": "fn()",
     "!doc": "<p>This function is a hook point for stores to provide as a way for\na fetch to be halted mid-processing.</p>\n"
    }
   }
  },
  "dojo/data/api/Write": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/data/api/Read.prototype",
    "close": {
     "!type": "fn(request?: +dojo|?)",
     "!doc": "<p>The close() method is intended for instructing the store to &#39;close&#39; out\nany information associated with a particular request.</p>\n"
    },
    "containsValue": {
     "!type": "fn(item: +dojo, attribute: ?, value: +anything)",
     "!doc": "<p>Returns true if the given <em>value</em> is one of the values that getValues()\nwould return.</p>\n"
    },
    "deleteItem": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Deletes an item from the store.</p>\n"
    },
    "fetch": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>Given a query and set of defined options, such as a start and count of items to return,\nthis method executes the query and makes the results available as data items.\nThe format and expectations of stores is that they operate in a generally asynchronous\nmanner, therefore callbacks are always used to return items located by the fetch parameters.</p>\n"
    },
    "getAttributes": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Returns an array with all the attributes that this item has.  This\nmethod will always return an array; if the item has no attributes\nat all, getAttributes() will return an empty array: [].</p>\n"
    },
    "getFeatures": {
     "!type": "fn()",
     "!doc": "<p>See dojo/data/api/Read.getFeatures()</p>\n"
    },
    "getLabel": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Method to inspect the item and return a user-readable &#39;label&#39; for the item\nthat provides a general/adequate description of what the item is.</p>\n"
    },
    "getLabelAttributes": {
     "!type": "fn(item: +dojo)",
     "!doc": "<p>Method to inspect the item and return an array of what attributes of the item were used\nto generate its label, if any.</p>\n"
    },
    "getValue": {
     "!type": "fn(item: +dojo, attribute: ?, defaultValue?: +value)",
     "!doc": "<p>Returns a single attribute value.\nReturns defaultValue if and only if <em>item</em> does not have a value for <em>attribute</em>.\nReturns null if and only if null was explicitly set as the attribute value.\nReturns undefined if and only if the item does not have a value for the\ngiven attribute (which is the same as saying the item does not have the attribute).</p>\n"
    },
    "getValues": {
     "!type": "fn(item: +dojo, attribute: ?)",
     "!doc": "<p>This getValues() method works just like the getValue() method, but getValues()\nalways returns an array rather than a single attribute value.  The array\nmay be empty, may contain a single attribute value, or may contain\nmany attribute values.\nIf the item does not have a value for the given attribute, then getValues()\nwill return an empty array: [].  (So, if store.hasAttribute(item, attribute)\nhas a return of false, then store.getValues(item, attribute) will return [].)</p>\n"
    },
    "hasAttribute": {
     "!type": "fn(item: +dojo, attribute: ?)",
     "!doc": "<p>Returns true if the given <em>item</em> has a value for the given <em>attribute</em>.</p>\n"
    },
    "isDirty": {
     "!type": "fn(item?: +item)",
     "!doc": "<p>Given an item, isDirty() returns true if the item has been modified\nsince the last save().  If isDirty() is called with no <em>item</em> argument,\nthen this function returns true if any item has been modified since\nthe last save().</p>\n"
    },
    "isItem": {
     "!type": "fn(something: +anything)",
     "!doc": "<p>Returns true if <em>something</em> is an item and came from the store instance.\nReturns false if <em>something</em> is a literal, an item from another store instance,\nor is any object other than an item.</p>\n"
    },
    "isItemLoaded": {
     "!type": "fn(something: +anything)",
     "!doc": "<p>Returns false if isItem(something) is false.  Returns false if\nif isItem(something) is true but the the item is not yet loaded\nin local memory (for example, if the item has not yet been read\nfrom the server).</p>\n"
    },
    "loadItem": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>Given an item, this method loads the item so that a subsequent call\nto store.isItemLoaded(item) will return true.  If a call to\nisItemLoaded() returns true before loadItem() is even called,\nthen loadItem() need not do any work at all and will not even invoke\nthe callback handlers.  So, before invoking this method, check that\nthe item has not already been loaded.</p>\n"
    },
    "newItem": {
     "!type": "fn(keywordArgs?: ?, parentInfo?: ?)",
     "!doc": "<p>Returns a newly created item.  Sets the attributes of the new\nitem based on the <em>keywordArgs</em> provided.  In general, the attribute\nnames in the keywords become the attributes in the new item and as for\nthe attribute values in keywordArgs, they become the values of the attributes\nin the new item.  In addition, for stores that support hierarchical item\ncreation, an optional second parameter is accepted that defines what item is the parent\nof the new item and what attribute of that item should the new item be assigned to.\nIn general, this will assume that the attribute targeted is multi-valued and a new item\nis appended onto the list of values for that attribute.</p>\n"
    },
    "revert": {
     "!type": "fn()",
     "!doc": "<p>Discards any unsaved changes.</p>\n"
    },
    "save": {
     "!type": "fn(keywordArgs: ?)",
     "!doc": "<p>Saves to the server all the changes that have been made locally.\nThe save operation may take some time and is generally performed\nin an asynchronous fashion.  The outcome of the save action is\nis passed into the set of supported callbacks for the save.</p>\n"
    },
    "setValue": {
     "!type": "fn(item: +dojo, attribute: string, value: ?)",
     "!doc": "<p>Sets the value of an attribute on an item.\nReplaces any previous value or values.</p>\n"
    },
    "setValues": {
     "!type": "fn(item: +dojo, attribute: string, values: [])",
     "!doc": "<p>Adds each value in the <em>values</em> array as a value of the given\nattribute on the given item.\nReplaces any previous value or values.\nCalling store.setValues(x, y, []) (with <em>values</em> as an empty array) has\nthe same effect as calling store.unsetAttribute(x, y).</p>\n"
    },
    "unsetAttribute": {
     "!type": "fn(item: +dojo, attribute: string)",
     "!doc": "<p>Deletes all the values of an attribute on an item.</p>\n"
    }
   }
  },
  "dojo/date/locale": {
   "addCustomFormats": {
    "!type": "fn(packageName: string, bundleName: string)",
    "!doc": "<p>Add a reference to a bundle containing localized custom formats to be\nused by date/time formatting and parsing routines.\n</p>\n"
   },
   "format": {
    "!type": "fn(dateObject: +Date, options?: ?)",
    "!doc": "<p>Format a Date object as a String, using locale-specific settings.\n</p>\n"
   },
   "getNames": {
    "!type": "fn(item: string, type: string, context?: string, locale?: string)",
    "!doc": "<p>Used to get localized strings from dojo.cldr for day or month names.\n</p>\n"
   },
   "isWeekend": {
    "!type": "fn(dateObject?: +Date, locale?: string) -> bool",
    "!doc": "<p>Determines if the date falls on a weekend, according to local custom.</p>\n"
   },
   "parse": {
    "!type": "fn(value: string, options?: ?)",
    "!doc": "<p>Convert a properly formatted string to a primitive Date object,\nusing locale-specific settings.\n</p>\n"
   },
   "regexp": {
    "!type": "fn(options?: ?)",
    "!doc": "<p>Builds the regular needed to parse a localized date</p>\n"
   }
  },
  "dojo/date/locale.__FormatOptions": {
   "!type": "fn()",
   "prototype": {
    "am": {
     "!doc": "<p>override strings for am in times</p>\n"
    },
    "datePattern": {
     "!doc": "<p>override pattern with this string</p>\n"
    },
    "formatLength": {
     "!doc": "<p>choice of long, short, medium or full (plus any custom additions).  Defaults to &#39;short&#39;</p>\n"
    },
    "fullYear": {
     "!doc": "<p>(format only) use 4 digit years whenever 2 digit years are called for</p>\n"
    },
    "locale": {
     "!doc": "<p>override the locale used to determine formatting rules</p>\n"
    },
    "pm": {
     "!doc": "<p>override strings for pm in times</p>\n"
    },
    "selector": {
     "!doc": "<p>choice of &#39;time&#39;,&#39;date&#39; (default: date and time)</p>\n"
    },
    "strict": {
     "!doc": "<p>(parse only) strict parsing, off by default</p>\n"
    },
    "timePattern": {
     "!doc": "<p>override pattern with this string</p>\n"
    }
   }
  },
  "dojo/date": {
   "add": {
    "!type": "fn(date: +Date, interval: string, amount: number)",
    "!doc": "<p>Add to a Date in intervals of different size, from milliseconds to years</p>\n"
   },
   "compare": {
    "!type": "fn(date1: +Date, date2?: +Date, portion?: string) -> number",
    "!doc": "<p>Compare two date objects by date, time, or both.</p>\n"
   },
   "difference": {
    "!type": "fn(date1: +Date, date2?: +Date, interval?: string)",
    "!doc": "<p>Get the difference in a specific unit of time (e.g., number of\nmonths, weeks, days, etc.) between two dates, rounded to the\nnearest integer.</p>\n"
   },
   "getDaysInMonth": {
    "!type": "fn(dateObject: +Date) -> number",
    "!doc": "<p>Returns the number of days in the month used by dateObject</p>\n"
   },
   "getTimezoneName": {
    "!type": "fn(dateObject: +Date)",
    "!doc": "<p>Get the user&#39;s time zone as provided by the browser</p>\n"
   },
   "isLeapYear": {
    "!type": "fn(dateObject: +Date) -> bool",
    "!doc": "<p>Determines if the year of the dateObject is a leap year</p>\n"
   }
  },
  "dojo/debounce": {
   "!type": "fn(cb: fn(), wait: +Integer)",
   "!doc": "<p>Create a function that will only execute after <code>wait</code> milliseconds</p>\n"
  },
  "dojo/dnd/AutoSource": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/dnd/Source.prototype",
    "accept": {},
    "allowNested": {
     "!doc": "<p>Indicates whether to allow dnd item nodes to be nested within other elements.\nBy default this is false, indicating that only direct children of the container can\nbe draggable dnd item nodes</p>\n"
    },
    "autoSync": {},
    "copyOnly": {},
    "current": {
     "!doc": "<p>The DOM node the mouse is currently hovered over</p>\n"
    },
    "delay": {},
    "generateText": {},
    "horizontal": {},
    "isSource": {},
    "map": {
     "!doc": "<p>Map from an item&#39;s id (which is also the DOMNode&#39;s id) to\nthe dojo/dnd/Container.Item itself.</p>\n"
    },
    "selection": {
     "!doc": "<p>The set of id&#39;s that are currently selected, such that this.selection[id] == 1\nif the node w/that id is selected.  Can iterate over selected node&#39;s id&#39;s like:\n\n</p>\n<pre><code>for(var id in this.selection)</code></pre>\n"
    },
    "selfAccept": {},
    "selfCopy": {},
    "singular": {},
    "skipForm": {},
    "withHandles": {},
    "checkAcceptance": {
     "!type": "fn(source: ?, nodes: []) -> bool",
     "!doc": "<p>checks if the target can accept nodes from this source</p>\n"
    },
    "clearItems": {
     "!type": "fn()",
     "!doc": "<p>removes all data items from the map</p>\n"
    },
    "copyState": {
     "!type": "fn(keyPressed: bool, self?: bool) -> bool",
     "!doc": "<p>Returns true if we need to copy items, false to move.\nIt is separated to be overwritten dynamically, if needed.</p>\n"
    },
    "creator": {
     "!type": "fn()",
     "!doc": "<p>creator function, dummy at the moment</p>\n"
    },
    "deleteSelectedNodes": {
     "!type": "fn() -> fn()",
     "!doc": "<p>deletes all selected items</p>\n"
    },
    "delItem": {
     "!type": "fn(key: string)",
     "!doc": "<p>removes a data item from the map by its key (id)</p>\n"
    },
    "destroy": {
     "!type": "fn()",
     "!doc": "<p>prepares the object to be garbage-collected</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "forInItems": {
     "!type": "fn(f: fn(), o?: ?) -> string",
     "!doc": "<p>iterates over a data map skipping members that\nare present in the empty object (IE and/or 3rd-party libraries).</p>\n"
    },
    "forInSelectedItems": {
     "!type": "fn(f: fn(), o?: ?)",
     "!doc": "<p>iterates over selected items;\nsee <code>dojo/dnd/Container.forInItems()</code> for details</p>\n"
    },
    "getAllNodes": {
     "!type": "fn()",
     "!doc": "<p>returns a list (an array) of all valid child nodes</p>\n"
    },
    "getItem": {
     "!type": "fn(key: string)",
     "!doc": "<p>returns a data item by its key (id)</p>\n"
    },
    "getSelectedNodes": {
     "!type": "fn()",
     "!doc": "<p>returns a list (an array) of selected nodes</p>\n"
    },
    "insertNodes": {
     "!type": "fn(addSelected: bool, data: [], before: bool, anchor: +Node) -> fn()",
     "!doc": "<p>inserts new data items (see <code>dojo/dnd/Container.insertNodes()</code> method for details)</p>\n"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, Ctor: ?)"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    },
    "selectAll": {
     "!type": "fn()",
     "!doc": "<p>selects all items</p>\n"
    },
    "selectNone": {
     "!type": "fn()",
     "!doc": "<p>unselects all items</p>\n"
    },
    "setItem": {
     "!type": "fn(key: string, data: +Container.Item)",
     "!doc": "<p>associates a data item with its key (id)</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>collects valid child items and populate the map</p>\n"
    },
    "sync": {
     "!type": "fn() -> fn()",
     "!doc": "<p>sync up the node list with the data map</p>\n"
    }
   }
  },
  "dojo/dnd/Source": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/dnd/Selector.prototype",
    "accept": {},
    "allowNested": {
     "!doc": "<p>Indicates whether to allow dnd item nodes to be nested within other elements.\nBy default this is false, indicating that only direct children of the container can\nbe draggable dnd item nodes</p>\n"
    },
    "autoSync": {},
    "copyOnly": {},
    "current": {
     "!doc": "<p>The DOM node the mouse is currently hovered over</p>\n"
    },
    "delay": {},
    "generateText": {},
    "horizontal": {},
    "isSource": {},
    "map": {
     "!doc": "<p>Map from an item&#39;s id (which is also the DOMNode&#39;s id) to\nthe dojo/dnd/Container.Item itself.</p>\n"
    },
    "selection": {
     "!doc": "<p>The set of id&#39;s that are currently selected, such that this.selection[id] == 1\nif the node w/that id is selected.  Can iterate over selected node&#39;s id&#39;s like:\n\n</p>\n<pre><code>for(var id in this.selection)</code></pre>\n"
    },
    "selfAccept": {},
    "selfCopy": {},
    "singular": {},
    "skipForm": {},
    "withHandles": {},
    "checkAcceptance": {
     "!type": "fn(source: ?, nodes: []) -> bool",
     "!doc": "<p>checks if the target can accept nodes from this source</p>\n"
    },
    "clearItems": {
     "!type": "fn()",
     "!doc": "<p>removes all data items from the map</p>\n"
    },
    "copyState": {
     "!type": "fn(keyPressed: bool, self?: bool) -> bool",
     "!doc": "<p>Returns true if we need to copy items, false to move.\nIt is separated to be overwritten dynamically, if needed.</p>\n"
    },
    "creator": {
     "!type": "fn()",
     "!doc": "<p>creator function, dummy at the moment</p>\n"
    },
    "deleteSelectedNodes": {
     "!type": "fn() -> fn()",
     "!doc": "<p>deletes all selected items</p>\n"
    },
    "delItem": {
     "!type": "fn(key: string)",
     "!doc": "<p>removes a data item from the map by its key (id)</p>\n"
    },
    "destroy": {
     "!type": "fn()",
     "!doc": "<p>prepares the object to be garbage-collected</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "forInItems": {
     "!type": "fn(f: fn(), o?: ?) -> string",
     "!doc": "<p>iterates over a data map skipping members that\nare present in the empty object (IE and/or 3rd-party libraries).</p>\n"
    },
    "forInSelectedItems": {
     "!type": "fn(f: fn(), o?: ?)",
     "!doc": "<p>iterates over selected items;\nsee <code>dojo/dnd/Container.forInItems()</code> for details</p>\n"
    },
    "getAllNodes": {
     "!type": "fn()",
     "!doc": "<p>returns a list (an array) of all valid child nodes</p>\n"
    },
    "getItem": {
     "!type": "fn(key: string)",
     "!doc": "<p>returns a data item by its key (id)</p>\n"
    },
    "getSelectedNodes": {
     "!type": "fn()",
     "!doc": "<p>returns a list (an array) of selected nodes</p>\n"
    },
    "insertNodes": {
     "!type": "fn(addSelected: bool, data: [], before: bool, anchor: +Node) -> fn()",
     "!doc": "<p>inserts new data items (see <code>dojo/dnd/Container.insertNodes()</code> method for details)</p>\n"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, Ctor: ?)"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    },
    "selectAll": {
     "!type": "fn()",
     "!doc": "<p>selects all items</p>\n"
    },
    "selectNone": {
     "!type": "fn()",
     "!doc": "<p>unselects all items</p>\n"
    },
    "setItem": {
     "!type": "fn(key: string, data: +Container.Item)",
     "!doc": "<p>associates a data item with its key (id)</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>collects valid child items and populate the map</p>\n"
    },
    "sync": {
     "!type": "fn() -> fn()",
     "!doc": "<p>sync up the node list with the data map</p>\n"
    }
   }
  },
  "dojo/dnd/common": {
   "_defaultCreatorNodes": {},
   "_empty": {},
   "_manager": {},
   "_uniqueId": {},
   "getCopyKeyState": {
    "!type": "fn(evt: ?)"
   },
   "getUniqueId": {
    "!type": "fn() -> string",
    "!doc": "<p>returns a unique string for use with any DOM element</p>\n"
   },
   "isFormElement": {
    "!type": "fn(e: +Event)",
    "!doc": "<p>returns true if user clicked on a form element</p>\n"
   },
   "manager": {
    "!type": "fn()",
    "!doc": "<p>Returns the current DnD manager.  Creates one if it is not created yet.</p>\n"
   }
  },
  "dojo/dnd/common._empty": {},
  "dojo/dnd/common._defaultCreatorNodes": {
   "div": {},
   "ol": {},
   "p": {},
   "ul": {}
  },
  "dojo/dnd/Selector": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/dnd/Container.prototype",
    "allowNested": {
     "!doc": "<p>Indicates whether to allow dnd item nodes to be nested within other elements.\nBy default this is false, indicating that only direct children of the container can\nbe draggable dnd item nodes</p>\n"
    },
    "current": {
     "!doc": "<p>The DOM node the mouse is currently hovered over</p>\n"
    },
    "map": {
     "!doc": "<p>Map from an item&#39;s id (which is also the DOMNode&#39;s id) to\nthe dojo/dnd/Container.Item itself.</p>\n"
    },
    "selection": {
     "!doc": "<p>The set of id&#39;s that are currently selected, such that this.selection[id] == 1\nif the node w/that id is selected.  Can iterate over selected node&#39;s id&#39;s like:\n\n</p>\n<pre><code>for(var id in this.selection)</code></pre>\n"
    },
    "singular": {},
    "skipForm": {},
    "clearItems": {
     "!type": "fn()",
     "!doc": "<p>removes all data items from the map</p>\n"
    },
    "creator": {
     "!type": "fn()",
     "!doc": "<p>creator function, dummy at the moment</p>\n"
    },
    "deleteSelectedNodes": {
     "!type": "fn() -> fn()",
     "!doc": "<p>deletes all selected items</p>\n"
    },
    "delItem": {
     "!type": "fn(key: string)",
     "!doc": "<p>removes a data item from the map by its key (id)</p>\n"
    },
    "destroy": {
     "!type": "fn()",
     "!doc": "<p>prepares the object to be garbage-collected</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "forInItems": {
     "!type": "fn(f: fn(), o?: ?) -> string",
     "!doc": "<p>iterates over a data map skipping members that\nare present in the empty object (IE and/or 3rd-party libraries).</p>\n"
    },
    "forInSelectedItems": {
     "!type": "fn(f: fn(), o?: ?)",
     "!doc": "<p>iterates over selected items;\nsee <code>dojo/dnd/Container.forInItems()</code> for details</p>\n"
    },
    "getAllNodes": {
     "!type": "fn()",
     "!doc": "<p>returns a list (an array) of all valid child nodes</p>\n"
    },
    "getItem": {
     "!type": "fn(key: string)",
     "!doc": "<p>returns a data item by its key (id)</p>\n"
    },
    "getSelectedNodes": {
     "!type": "fn()",
     "!doc": "<p>returns a list (an array) of selected nodes</p>\n"
    },
    "insertNodes": {
     "!type": "fn(addSelected: bool, data: [], before: bool, anchor: +Node) -> fn()",
     "!doc": "<p>inserts new data items (see <code>dojo/dnd/Container.insertNodes()</code> method for details)</p>\n"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, Ctor: ?)"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    },
    "selectAll": {
     "!type": "fn()",
     "!doc": "<p>selects all items</p>\n"
    },
    "selectNone": {
     "!type": "fn()",
     "!doc": "<p>unselects all items</p>\n"
    },
    "setItem": {
     "!type": "fn(key: string, data: +Container.Item)",
     "!doc": "<p>associates a data item with its key (id)</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>collects valid child items and populate the map</p>\n"
    },
    "sync": {
     "!type": "fn() -> fn()",
     "!doc": "<p>sync up the node list with the data map</p>\n"
    }
   }
  },
  "dojo/touch": {
   "cancel": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to &#39;touchcancel&#39;|&#39;mouseleave&#39; for the given node</p>\n"
   },
   "enter": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to mouse.enter or touch equivalent for the given node</p>\n"
   },
   "leave": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to mouse.leave or touch equivalent for the given node</p>\n"
   },
   "move": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener that fires when the mouse cursor or a finger is dragged over the given node.</p>\n"
   },
   "out": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to &#39;mouseout&#39; or touch equivalent for the given node</p>\n"
   },
   "over": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to &#39;mouseover&#39; or touch equivalent for the given node</p>\n"
   },
   "press": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to &#39;touchstart&#39;|&#39;mousedown&#39; for the given node</p>\n"
   },
   "release": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to releasing the mouse button while the cursor is over the given node\n(i.e. &quot;mouseup&quot;) or for removing the finger from the screen while touching the given node.</p>\n"
   }
  },
  "dojo/dnd/Container": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/Evented.prototype",
    "allowNested": {
     "!doc": "<p>Indicates whether to allow dnd item nodes to be nested within other elements.\nBy default this is false, indicating that only direct children of the container can\nbe draggable dnd item nodes</p>\n"
    },
    "current": {
     "!doc": "<p>The DOM node the mouse is currently hovered over</p>\n"
    },
    "map": {
     "!doc": "<p>Map from an item&#39;s id (which is also the DOMNode&#39;s id) to\nthe dojo/dnd/Container.Item itself.</p>\n"
    },
    "skipForm": {},
    "clearItems": {
     "!type": "fn()",
     "!doc": "<p>removes all data items from the map</p>\n"
    },
    "creator": {
     "!type": "fn()",
     "!doc": "<p>creator function, dummy at the moment</p>\n"
    },
    "delItem": {
     "!type": "fn(key: string)",
     "!doc": "<p>removes a data item from the map by its key (id)</p>\n"
    },
    "destroy": {
     "!type": "fn()",
     "!doc": "<p>prepares this object to be garbage-collected</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "forInItems": {
     "!type": "fn(f: fn(), o?: ?) -> string",
     "!doc": "<p>iterates over a data map skipping members that\nare present in the empty object (IE and/or 3rd-party libraries).</p>\n"
    },
    "getAllNodes": {
     "!type": "fn()",
     "!doc": "<p>returns a list (an array) of all valid child nodes</p>\n"
    },
    "getItem": {
     "!type": "fn(key: string)",
     "!doc": "<p>returns a data item by its key (id)</p>\n"
    },
    "insertNodes": {
     "!type": "fn(data: ?, before: bool, anchor: +Node) -> fn()",
     "!doc": "<p>inserts an array of new nodes before/after an anchor node</p>\n"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, Ctor: ?)"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    },
    "setItem": {
     "!type": "fn(key: string, data: +Container.Item)",
     "!doc": "<p>associates a data item with its key (id)</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>collects valid child items and populate the map</p>\n"
    },
    "sync": {
     "!type": "fn() -> fn()",
     "!doc": "<p>sync up the node list with the data map</p>\n"
    }
   },
   "Item": {
    "!type": "fn()",
    "!doc": "<p>Represents (one of) the source node(s) being dragged.\nContains (at least) the &quot;type&quot; and &quot;data&quot; attributes.</p>\n"
   }
  },
  "dojo/dnd/Container.__ContainerArgs": {
   "!type": "fn()",
   "prototype": {
    "_skipStartup": {
     "!doc": "<p>skip startup(), which collects children, for deferred initialization\n(this is used in the markup mode)</p>\n"
    },
    "dropParent": {
     "!doc": "<p>node or node&#39;s id to use as the parent node for dropped items\n(must be underneath the &#39;node&#39; parameter in the DOM)</p>\n"
    },
    "skipForm": {
     "!doc": "<p>don&#39;t start the drag operation, if clicked on form elements</p>\n"
    },
    "creator": {
     "!type": "fn()",
     "!doc": "<p>a creator function, which takes a data item, and returns an object like that:\n{node: newNode, data: usedData, type: arrayOfStrings}</p>\n"
    }
   }
  },
  "dojo/dnd/Manager": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/Evented.prototype",
    "OFFSET_X": {},
    "OFFSET_Y": {},
    "canDrop": {
     "!type": "fn(flag: ?)",
     "!doc": "<p>called to notify if the current target can accept items</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "makeAvatar": {
     "!type": "fn()",
     "!doc": "<p>makes the avatar; it is separate to be overwritten dynamically, if needed</p>\n"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    },
    "outSource": {
     "!type": "fn(source: ?)",
     "!doc": "<p>called when a source detected a mouse-out condition</p>\n"
    },
    "overSource": {
     "!type": "fn(source: ?)",
     "!doc": "<p>called when a source detected a mouse-over condition</p>\n"
    },
    "startDrag": {
     "!type": "fn(source: ?, nodes: [], copy: bool)",
     "!doc": "<p>called to initiate the DnD operation</p>\n"
    },
    "stopDrag": {
     "!type": "fn()",
     "!doc": "<p>stop the DnD in progress</p>\n"
    },
    "updateAvatar": {
     "!type": "fn()",
     "!doc": "<p>updates the avatar; it is separate to be overwritten dynamically, if needed</p>\n"
    }
   },
   "manager": {
    "!type": "fn()",
    "!doc": "<p>Returns the current DnD manager.  Creates one if it is not created yet.</p>\n"
   }
  },
  "dojo/dnd/autoscroll": {
   "_validNodes": {},
   "_validOverflow": {},
   "H_AUTOSCROLL_VALUE": {},
   "H_TRIGGER_AUTOSCROLL": {},
   "V_AUTOSCROLL_VALUE": {},
   "V_TRIGGER_AUTOSCROLL": {},
   "autoScroll": {
    "!type": "fn(e: +Event)",
    "!doc": "<p>a handler for mousemove and touchmove events, which scrolls the window, if\nnecessary</p>\n"
   },
   "autoScrollNodes": {
    "!type": "fn(e: +Event)",
    "!doc": "<p>a handler for mousemove and touchmove events, which scrolls the first available\nDom element, it falls back to exports.autoScroll()</p>\n"
   },
   "autoScrollStart": {
    "!type": "fn(d: +Document)",
    "!doc": "<p>Called at the start of a drag.</p>\n"
   },
   "getViewport": {
    "!type": "fn(doc?: +Document)",
    "!doc": "<p>Returns the dimensions and scroll position of the viewable area of a browser window</p>\n"
   }
  },
  "dojo/dnd/autoscroll._validNodes": {
   "div": {},
   "p": {},
   "td": {}
  },
  "dojo/dnd/autoscroll._validOverflow": {
   "auto": {},
   "scroll": {}
  },
  "dojo/window": {
   "get": {
    "!type": "fn(doc: +Document)",
    "!doc": "<p>Get window object associated with document doc.</p>\n"
   },
   "getBox": {
    "!type": "fn(doc?: +Document)",
    "!doc": "<p>Returns the dimensions and scroll position of the viewable area of a browser window</p>\n"
   },
   "scrollIntoView": {
    "!type": "fn(node: +Node, pos?: ?)",
    "!doc": "<p>Scroll the passed node into view using minimal movement, if it is not already.</p>\n"
   }
  },
  "dojo/dnd/Avatar": {
   "!type": "fn()",
   "prototype": {
    "manager": {
     "!doc": "<p>a DnD manager object</p>\n"
    },
    "construct": {
     "!type": "fn()",
     "!doc": "<p>constructor function;\nit is separate so it can be (dynamically) overwritten in case of need</p>\n"
    },
    "destroy": {
     "!type": "fn()",
     "!doc": "<p>destructor for the avatar; called to remove all references so it can be garbage-collected</p>\n"
    },
    "update": {
     "!type": "fn()",
     "!doc": "<p>updates the avatar to reflect the current DnD state</p>\n"
    }
   }
  },
  "dojo/hccss": {
   "!type": "fn()",
   "!doc": "<p>Test if computer is in high contrast mode (i.e. if browser is not displaying background images).\nDefines <code>has(&quot;highcontrast&quot;)</code> and sets <code>dj_a11y</code> CSS class on <code>&lt;body&gt;</code> if machine is in high contrast mode.\nReturns <code>has()</code> method;</p>\n"
  },
  "dojo/dnd/Moveable": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/Evented.prototype",
    "delay": {},
    "handle": {},
    "skip": {},
    "destroy": {
     "!type": "fn()",
     "!doc": "<p>stops watching for possible move, deletes all references, so the object can be garbage-collected</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, Ctor: ?)"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    }
   }
  },
  "dojo/dnd/Moveable.__MoveableArgs": {
   "!type": "fn()",
   "prototype": {
    "delay": {
     "!doc": "<p>delay move by this number of pixels</p>\n"
    },
    "handle": {
     "!doc": "<p>A node (or node&#39;s id), which is used as a mouse handle.\nIf omitted, the node itself is used as a handle.</p>\n"
    },
    "mover": {
     "!doc": "<p>a constructor of custom Mover</p>\n"
    },
    "skip": {
     "!doc": "<p>skip move of form elements</p>\n"
    }
   }
  },
  "dojo/dnd/Mover": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/Evented.prototype",
    "destroy": {
     "!type": "fn()",
     "!doc": "<p>stops the move, deletes all references, so the object can be garbage-collected</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    }
   }
  },
  "dojo/dnd/Target": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/dnd/Source.prototype",
    "accept": {},
    "allowNested": {
     "!doc": "<p>Indicates whether to allow dnd item nodes to be nested within other elements.\nBy default this is false, indicating that only direct children of the container can\nbe draggable dnd item nodes</p>\n"
    },
    "autoSync": {},
    "copyOnly": {},
    "current": {
     "!doc": "<p>The DOM node the mouse is currently hovered over</p>\n"
    },
    "delay": {},
    "generateText": {},
    "horizontal": {},
    "isSource": {},
    "map": {
     "!doc": "<p>Map from an item&#39;s id (which is also the DOMNode&#39;s id) to\nthe dojo/dnd/Container.Item itself.</p>\n"
    },
    "selection": {
     "!doc": "<p>The set of id&#39;s that are currently selected, such that this.selection[id] == 1\nif the node w/that id is selected.  Can iterate over selected node&#39;s id&#39;s like:\n\n</p>\n<pre><code>for(var id in this.selection)</code></pre>\n"
    },
    "selfAccept": {},
    "selfCopy": {},
    "singular": {},
    "skipForm": {},
    "withHandles": {},
    "checkAcceptance": {
     "!type": "fn(source: ?, nodes: []) -> bool",
     "!doc": "<p>checks if the target can accept nodes from this source</p>\n"
    },
    "clearItems": {
     "!type": "fn()",
     "!doc": "<p>removes all data items from the map</p>\n"
    },
    "copyState": {
     "!type": "fn(keyPressed: bool, self?: bool) -> bool",
     "!doc": "<p>Returns true if we need to copy items, false to move.\nIt is separated to be overwritten dynamically, if needed.</p>\n"
    },
    "creator": {
     "!type": "fn()",
     "!doc": "<p>creator function, dummy at the moment</p>\n"
    },
    "deleteSelectedNodes": {
     "!type": "fn() -> fn()",
     "!doc": "<p>deletes all selected items</p>\n"
    },
    "delItem": {
     "!type": "fn(key: string)",
     "!doc": "<p>removes a data item from the map by its key (id)</p>\n"
    },
    "destroy": {
     "!type": "fn()",
     "!doc": "<p>prepares the object to be garbage-collected</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "forInItems": {
     "!type": "fn(f: fn(), o?: ?) -> string",
     "!doc": "<p>iterates over a data map skipping members that\nare present in the empty object (IE and/or 3rd-party libraries).</p>\n"
    },
    "forInSelectedItems": {
     "!type": "fn(f: fn(), o?: ?)",
     "!doc": "<p>iterates over selected items;\nsee <code>dojo/dnd/Container.forInItems()</code> for details</p>\n"
    },
    "getAllNodes": {
     "!type": "fn()",
     "!doc": "<p>returns a list (an array) of all valid child nodes</p>\n"
    },
    "getItem": {
     "!type": "fn(key: string)",
     "!doc": "<p>returns a data item by its key (id)</p>\n"
    },
    "getSelectedNodes": {
     "!type": "fn()",
     "!doc": "<p>returns a list (an array) of selected nodes</p>\n"
    },
    "insertNodes": {
     "!type": "fn(addSelected: bool, data: [], before: bool, anchor: +Node) -> fn()",
     "!doc": "<p>inserts new data items (see <code>dojo/dnd/Container.insertNodes()</code> method for details)</p>\n"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, Ctor: ?)"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    },
    "selectAll": {
     "!type": "fn()",
     "!doc": "<p>selects all items</p>\n"
    },
    "selectNone": {
     "!type": "fn()",
     "!doc": "<p>unselects all items</p>\n"
    },
    "setItem": {
     "!type": "fn(key: string, data: +Container.Item)",
     "!doc": "<p>associates a data item with its key (id)</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>collects valid child items and populate the map</p>\n"
    },
    "sync": {
     "!type": "fn() -> fn()",
     "!doc": "<p>sync up the node list with the data map</p>\n"
    }
   }
  },
  "dojo/dnd/TimedMoveable": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/dnd/Moveable.prototype",
    "delay": {},
    "handle": {},
    "skip": {},
    "timeout": {},
    "destroy": {
     "!type": "fn()",
     "!doc": "<p>stops watching for possible move, deletes all references, so the object can be garbage-collected</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, Ctor: ?)"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    }
   }
  },
  "dojo/dnd/move": {
   "boxConstrainedMoveable": {
    "!type": "fn()"
   },
   "constrainedMoveable": {
    "!type": "fn()"
   },
   "parentConstrainedMoveable": {
    "!type": "fn()"
   }
  },
  "dojo/dnd/move.constrainedMoveable": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/dnd/Moveable.prototype",
    "delay": {},
    "handle": {},
    "skip": {},
    "within": {},
    "constraints": {
     "!type": "fn()"
    },
    "destroy": {
     "!type": "fn()",
     "!doc": "<p>stops watching for possible move, deletes all references, so the object can be garbage-collected</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, Ctor: ?)"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    }
   }
  },
  "dojo/dnd/move.boxConstrainedMoveable": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/dnd/Moveable.prototype",
    "box": {
     "!doc": "<p>object attributes (for markup)</p>\n"
    },
    "delay": {},
    "handle": {},
    "skip": {},
    "within": {},
    "constraints": {
     "!type": "fn()"
    },
    "destroy": {
     "!type": "fn()",
     "!doc": "<p>stops watching for possible move, deletes all references, so the object can be garbage-collected</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, Ctor: ?)"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    }
   }
  },
  "dojo/dnd/move.parentConstrainedMoveable": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/dnd/Moveable.prototype",
    "area": {
     "!doc": "<p>object attributes (for markup)</p>\n"
    },
    "delay": {},
    "handle": {},
    "skip": {},
    "within": {},
    "constraints": {
     "!type": "fn()"
    },
    "destroy": {
     "!type": "fn()",
     "!doc": "<p>stops watching for possible move, deletes all references, so the object can be garbage-collected</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, Ctor: ?)"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    }
   }
  },
  "dojo/fx/Toggler": {
   "!type": "fn()",
   "prototype": {
    "_hideAnim": {},
    "_hideArgs": {},
    "_isHiding": {},
    "_isShowing": {},
    "_showAnim": {},
    "_showArgs": {},
    "hideDuration": {
     "!doc": "<p>Time in milliseconds to run the hide Animation</p>\n"
    },
    "node": {
     "!doc": "<p>the node to target for the showing and hiding animations</p>\n"
    },
    "showDuration": {
     "!doc": "<p>Time in milliseconds to run the show Animation</p>\n"
    },
    "hide": {
     "!type": "fn(delay?: +Integer)",
     "!doc": "<p>Toggle the node to hidden</p>\n"
    },
    "hideFunc": {
     "!type": "fn(args: ?)",
     "!doc": "<p>The function that returns the <code>dojo.Animation</code> to hide the node</p>\n"
    },
    "show": {
     "!type": "fn(delay?: +Integer)",
     "!doc": "<p>Toggle the node to showing</p>\n"
    },
    "showFunc": {
     "!type": "fn(args: ?)",
     "!doc": "<p>The function that returns the <code>dojo.Animation</code> to show the node</p>\n"
    }
   }
  },
  "dojo/fx/easing": {
   "backIn": {
    "!type": "fn(n?: +Decimal) -> number",
    "!doc": "<p>An easing function that starts away from the target,\nand quickly accelerates towards the end value.\n\n</p>\n<p>Use caution when the easing will cause values to become\nnegative as some properties cannot be set to negative values.</p>\n"
   },
   "backInOut": {
    "!type": "fn(n?: +Decimal) -> number",
    "!doc": "<p>An easing function combining the effects of <code>backIn</code> and <code>backOut</code></p>\n"
   },
   "backOut": {
    "!type": "fn(n?: +Decimal) -> number",
    "!doc": "<p>An easing function that pops past the range briefly, and slowly comes back.</p>\n"
   },
   "bounceIn": {
    "!type": "fn(n?: +Decimal) -> number",
    "!doc": "<p>An easing function that &#39;bounces&#39; near the beginning of an Animation</p>\n"
   },
   "bounceInOut": {
    "!type": "fn(n?: +Decimal) -> number",
    "!doc": "<p>An easing function that &#39;bounces&#39; at the beginning and end of the Animation</p>\n"
   },
   "bounceOut": {
    "!type": "fn(n?: +Decimal) -> number",
    "!doc": "<p>An easing function that &#39;bounces&#39; near the end of an Animation</p>\n"
   },
   "circIn": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "circInOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "circOut": {
    "!type": "fn(n?: +Decimal)"
   },
   "cubicIn": {
    "!type": "fn(n?: +Decimal)"
   },
   "cubicInOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "cubicOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "elasticIn": {
    "!type": "fn(n?: +Decimal) -> +Decimal",
    "!doc": "<p>An easing function the elastically snaps from the start value</p>\n"
   },
   "elasticInOut": {
    "!type": "fn(n?: +Decimal) -> number",
    "!doc": "<p>An easing function that elasticly snaps around the value, near\nthe beginning and end of the Animation.</p>\n"
   },
   "elasticOut": {
    "!type": "fn(n?: +Decimal) -> +Decimal",
    "!doc": "<p>An easing function that elasticly snaps around the target value,\nnear the end of the Animation</p>\n"
   },
   "expoIn": {
    "!type": "fn(n?: +Decimal)"
   },
   "expoInOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "expoOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "linear": {
    "!type": "fn(n?: +Decimal) -> +Decimal",
    "!doc": "<p>A linear easing function</p>\n"
   },
   "quadIn": {
    "!type": "fn(n?: +Decimal)"
   },
   "quadInOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "quadOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "quartIn": {
    "!type": "fn(n?: +Decimal)"
   },
   "quartInOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "quartOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "quintIn": {
    "!type": "fn(n?: +Decimal)"
   },
   "quintInOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "quintOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "sineIn": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "sineInOut": {
    "!type": "fn(n?: +Decimal) -> number"
   },
   "sineOut": {
    "!type": "fn(n?: +Decimal)"
   }
  },
  "dojo/gears": {
   "available": {
    "!doc": "<p>True if client is using Google Gears</p>\n"
   }
  },
  "dojo/gears.available": {},
  "dojo/hash": {
   "!type": "fn(hash?: string, replace?: bool)",
   "!doc": "<p>Gets or sets the hash string in the browser URL.</p>\n"
  },
  "dojo/io/iframe": {
   "create": {
    "!type": "fn(fname: string, onloadstr: string, uri: string)",
    "!doc": "<p>Creates a hidden iframe in the page. Used mostly for IO\ntransports.  You do not need to call this to start a\ndojo/io/iframe request. Just call send().</p>\n"
   },
   "doc": {
    "!type": "fn(iframeNode: ?)",
    "!doc": "<p>Returns the document object associated with the iframe DOM Node argument.</p>\n"
   },
   "setSrc": {
    "!type": "fn(iframe: ?, src: ?, replace: ?)",
    "!doc": "<p>Sets the URL that is loaded in an IFrame. The replace parameter\nindicates whether location.replace() should be used when\nchanging the location of the iframe.</p>\n"
   }
  },
  "dojo/request/iframe": {
   "!type": "fn(url: string, options?: +dojo) -> +dojo",
   "!doc": "<p>Sends a request using an iframe element with the given URL and options.</p>\n",
   "_currentDfd": {},
   "_dfdQueue": {},
   "_iframeName": {},
   "create": {
    "!type": "fn(name: ?, onloadstr: ?, uri: ?)"
   },
   "doc": {
    "!type": "fn(iframeNode: ?)"
   },
   "get": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP GET request using an iframe element with the given URL and options.</p>\n"
   },
   "post": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP POST request using an iframe element with the given URL and options.</p>\n"
   },
   "setSrc": {
    "!type": "fn(_iframe: ?, src: ?, replace: ?)"
   }
  },
  "dojo/request/iframe.__BaseOptions": {
   "!type": "fn()",
   "prototype": {
    "data": {
     "!doc": "<p>Data to transfer. When making a GET request, this will\nbe converted to key=value parameters and appended to the\nURL.</p>\n"
    },
    "form": {
     "!doc": "<p>A form node to use to submit data to the server.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>How to handle the response from the server.  Default is\n&#39;text&#39;.  Other values are &#39;json&#39;, &#39;javascript&#39;, and &#39;xml&#39;.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Whether to append a cache-busting parameter to the URL.</p>\n"
    },
    "query": {
     "!doc": "<p>Query parameters to append to the URL.</p>\n"
    },
    "timeout": {
     "!doc": "<p>Milliseconds to wait for the response.  If this time\npasses, the then the promise is rejected.</p>\n"
    }
   }
  },
  "dojo/request/iframe.__MethodOptions": {
   "!type": "fn()",
   "prototype": {
    "method": {
     "!doc": "<p>The HTTP method to use to make the request. Must be\nuppercase. Only <code>&quot;GET&quot;</code> and <code>&quot;POST&quot;</code> are accepted.\nDefault is <code>&quot;POST&quot;</code>.</p>\n"
    }
   }
  },
  "dojo/request/iframe.__Options": {
   "!type": "fn()",
   "prototype": {
    "data": {
     "!doc": "<p>Data to transfer. When making a GET request, this will\nbe converted to key=value parameters and appended to the\nURL.</p>\n"
    },
    "form": {
     "!doc": "<p>A form node to use to submit data to the server.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>How to handle the response from the server.  Default is\n&#39;text&#39;.  Other values are &#39;json&#39;, &#39;javascript&#39;, and &#39;xml&#39;.</p>\n"
    },
    "method": {
     "!doc": "<p>The HTTP method to use to make the request. Must be\nuppercase. Only <code>&quot;GET&quot;</code> and <code>&quot;POST&quot;</code> are accepted.\nDefault is <code>&quot;POST&quot;</code>.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Whether to append a cache-busting parameter to the URL.</p>\n"
    },
    "query": {
     "!doc": "<p>Query parameters to append to the URL.</p>\n"
    },
    "timeout": {
     "!doc": "<p>Milliseconds to wait for the response.  If this time\npasses, the then the promise is rejected.</p>\n"
    }
   }
  },
  "dojo/io/script": {
   "_counter": {},
   "_deadScripts": {},
   "attach": {
    "!type": "fn(id: ?, url: ?, frameDocument: ?)",
    "!doc": "<p>creates a new <code>&lt;script&gt;</code> tag pointing to the specified URL and\nadds it to the document.</p>\n"
   },
   "get": {
    "!type": "fn(args: ?)",
    "!doc": "<p>sends a get request using a dynamically created script tag.</p>\n"
   },
   "remove": {
    "!type": "fn(id: ?, frameDocument: ?)",
    "!doc": "<p>removes the script element with the given id, from the given frameDocument.\nIf no frameDocument is passed, the current document is used.</p>\n"
   }
  },
  "dojo/request/script": {
   "!type": "fn(url: string, options?: +dojo) -> +dojo",
   "!doc": "<p>Sends a request using a script element with the given URL and options.</p>\n",
   "_callbacksProperty": {},
   "get": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP GET request using a script element with the given URL and options.</p>\n"
   }
  },
  "dojo/request/script.__BaseOptions": {
   "!type": "fn()",
   "prototype": {
    "checkString": {
     "!doc": "<p>A string of JavaScript that when evaluated like so:\n&quot;typeof(&quot; + checkString + &quot;) != &#39;undefined&#39;&quot;\nbeing true means that the script fetched has been loaded.\nDo not use this if doing a JSONP type of call (use <code>jsonp</code> instead).</p>\n"
    },
    "data": {
     "!doc": "<p>Data to transfer.  This is ignored for GET and DELETE\nrequests.</p>\n"
    },
    "frameDoc": {
     "!doc": "<p>The Document object of a child iframe. If this is passed in, the script\nwill be attached to that document. This can be helpful in some comet long-polling\nscenarios with Firefox and Opera.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>How to handle the response from the server.  Default is\n&#39;text&#39;.  Other values are &#39;json&#39;, &#39;javascript&#39;, and &#39;xml&#39;.</p>\n"
    },
    "jsonp": {
     "!doc": "<p>The URL parameter name that indicates the JSONP callback string.\nFor instance, when using Yahoo JSONP calls it is normally,\njsonp: &quot;callback&quot;. For AOL JSONP calls it is normally\njsonp: &quot;c&quot;.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Whether to append a cache-busting parameter to the URL.</p>\n"
    },
    "query": {
     "!doc": "<p>Query parameters to append to the URL.</p>\n"
    },
    "timeout": {
     "!doc": "<p>Milliseconds to wait for the response.  If this time\npasses, the then the promise is rejected.</p>\n"
    }
   }
  },
  "dojo/request/script.__MethodOptions": {
   "!type": "fn()",
   "prototype": {
    "method": {
     "!doc": "<p>This option is ignored. All requests using this transport are\nGET requests.</p>\n"
    }
   }
  },
  "dojo/request/script.__Options": {
   "!type": "fn()",
   "prototype": {
    "checkString": {
     "!doc": "<p>A string of JavaScript that when evaluated like so:\n&quot;typeof(&quot; + checkString + &quot;) != &#39;undefined&#39;&quot;\nbeing true means that the script fetched has been loaded.\nDo not use this if doing a JSONP type of call (use <code>jsonp</code> instead).</p>\n"
    },
    "data": {
     "!doc": "<p>Data to transfer.  This is ignored for GET and DELETE\nrequests.</p>\n"
    },
    "frameDoc": {
     "!doc": "<p>The Document object of a child iframe. If this is passed in, the script\nwill be attached to that document. This can be helpful in some comet long-polling\nscenarios with Firefox and Opera.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>How to handle the response from the server.  Default is\n&#39;text&#39;.  Other values are &#39;json&#39;, &#39;javascript&#39;, and &#39;xml&#39;.</p>\n"
    },
    "jsonp": {
     "!doc": "<p>The URL parameter name that indicates the JSONP callback string.\nFor instance, when using Yahoo JSONP calls it is normally,\njsonp: &quot;callback&quot;. For AOL JSONP calls it is normally\njsonp: &quot;c&quot;.</p>\n"
    },
    "method": {
     "!doc": "<p>This option is ignored. All requests using this transport are\nGET requests.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Whether to append a cache-busting parameter to the URL.</p>\n"
    },
    "query": {
     "!doc": "<p>Query parameters to append to the URL.</p>\n"
    },
    "timeout": {
     "!doc": "<p>Milliseconds to wait for the response.  If this time\npasses, the then the promise is rejected.</p>\n"
    }
   }
  },
  "dojo/loadInit": {
   "dynamic": {},
   "load": {},
   "normalize": {
    "!type": "fn(id: ?)"
   }
  },
  "dojo/main": {
   "_blockAsync": {},
   "_contentHandlers": {
    "!doc": "<p>A map of available XHR transport handle types. Name matches the\n<code>handleAs</code> attribute passed to XHR calls.</p>\n"
   },
   "_defaultContext": {},
   "_hasResource": {},
   "_initFired": {},
   "_name": {},
   "_nodeDataCache": {},
   "_NodeListCtor": {},
   "_postLoad": {},
   "_windowUnloaders": {},
   "back": {
    "!doc": "<p>Browser history management resources</p>\n"
   },
   "baseUrl": {},
   "behavior": {},
   "cldr": {},
   "colors": {},
   "config": {
    "!doc": "<p>This module defines the user configuration during bootstrap.</p>\n"
   },
   "connectPublisher": {},
   "contentHandlers": {
    "!doc": "<p>A map of available XHR transport handle types. Name matches the\n<code>handleAs</code> attribute passed to XHR calls.</p>\n"
   },
   "currency": {
    "!doc": "<p>localized formatting and parsing routines for currencies</p>\n"
   },
   "data": {},
   "date": {},
   "dijit": {},
   "dnd": {},
   "doc": {
    "!doc": "<p>Alias for the current document. &#39;doc&#39; can be modified\nfor temporary context shifting. See also withDoc().</p>\n"
   },
   "dojox": {},
   "fx": {
    "!doc": "<p>Effects library on top of Base animations</p>\n"
   },
   "gears": {
    "!doc": "<p>TODOC</p>\n"
   },
   "global": {
    "!doc": "<p>Alias for the current window. &#39;global&#39; can be modified\nfor temporary context shifting. See also withGlobal().</p>\n"
   },
   "html": {
    "!doc": "<p>TODOC</p>\n"
   },
   "i18n": {
    "!doc": "<p>This module implements the dojo/i18n! plugin and the v1.6- i18n API</p>\n"
   },
   "io": {},
   "isAir": {
    "!doc": "<p>True if client is Adobe Air</p>\n"
   },
   "isAndroid": {
    "!doc": "<p>Version as a Number if client is android browser. undefined otherwise.</p>\n"
   },
   "isAsync": {},
   "isBrowser": {},
   "isChrome": {
    "!doc": "<p>Version as a Number if client is Chrome browser. undefined otherwise.</p>\n"
   },
   "isCopyKey": {},
   "isFF": {},
   "isIE": {
    "!doc": "<p>Version as a Number if client is MSIE(PC). undefined otherwise. Corresponds to\nmajor detected IE version (6, 7, 8, etc.)</p>\n"
   },
   "isIos": {
    "!doc": "<p>Version as a Number if client is iPhone, iPod, or iPad. undefined otherwise.</p>\n"
   },
   "isKhtml": {
    "!doc": "<p>Version as a Number if client is a KHTML browser. undefined otherwise. Corresponds to major\ndetected version.</p>\n"
   },
   "isMac": {
    "!doc": "<p>True if the client runs on Mac</p>\n"
   },
   "isMoz": {},
   "isMozilla": {},
   "isOpera": {
    "!doc": "<p>Version as a Number if client is Opera. undefined otherwise. Corresponds to\nmajor detected version.</p>\n"
   },
   "isQuirks": {},
   "isSafari": {
    "!doc": "<p>Version as a Number if client is Safari or iPhone. undefined otherwise.</p>\n"
   },
   "isSpidermonkey": {},
   "isWebKit": {
    "!doc": "<p>Version as a Number if client is a WebKit-derived browser (Konqueror,\nSafari, Chrome, etc.). undefined otherwise.</p>\n"
   },
   "isWii": {
    "!doc": "<p>True if client is Wii</p>\n"
   },
   "keys": {
    "!doc": "<p>Definitions for common key values.  Client code should test keyCode against these named constants,\nas the actual codes can vary by browser.</p>\n"
   },
   "locale": {},
   "mouseButtons": {},
   "number": {
    "!doc": "<p>localized formatting and parsing routines for Number</p>\n"
   },
   "parser": {},
   "publish": {},
   "query": {},
   "regexp": {
    "!doc": "<p>Regular expressions and Builder resources</p>\n"
   },
   "rpc": {},
   "scopeMap": {},
   "store": {},
   "string": {
    "!doc": "<p>String utilities for Dojo</p>\n"
   },
   "subscribe": {},
   "tests": {
    "!doc": "<p>D.O.H. Test files for Dojo unit testing.</p>\n"
   },
   "toJsonIndentStr": {},
   "touch": {
    "!doc": "<p>This module provides unified touch event handlers by exporting\npress, move, release and cancel which can also run well on desktop.\nBased on <a href=\"http://dvcs.w3.org/hg/webevents/raw-file/tip/touchevents.html\">http://dvcs.w3.org/hg/webevents/raw-file/tip/touchevents.html</a>\nAlso, if the dojoClick property is set to truthy on a DOM node, dojo/touch generates\nclick events immediately for this node and its descendants (except for descendants that\nhave a dojoClick property set to falsy), to avoid the delay before native browser click events,\nand regardless of whether evt.preventDefault() was called in a touch.press event listener.\n</p>\n"
   },
   "version": {
    "!doc": "<p>Version number of the Dojo Toolkit</p>\n"
   },
   "window": {
    "!doc": "<p>TODOC</p>\n"
   },
   "AdapterRegistry": {
    "!type": "fn(returnWrappers?: bool)",
    "!doc": "<p>A registry to make contextual calling/searching easier.</p>\n"
   },
   "addClass": {
    "!type": "fn(node: string|+Node, classStr: string|[])",
    "!doc": "<p>Adds the specified classes to the end of the class list on the\npassed node. Will not re-apply duplicate classes.\n</p>\n"
   },
   "addOnLoad": {
    "!type": "fn(priority?: +Integer, context: ?, callback?: fn())",
    "!doc": "<p>Add a function to execute on DOM content loaded and all requested modules have arrived and been evaluated.\nIn most cases, the <code>domReady</code> plug-in should suffice and this method should not be needed.\n\n</p>\n<p>When called in a non-browser environment, just checks that all requested modules have arrived and been\nevaluated.</p>\n"
   },
   "addOnUnload": {
    "!type": "fn(obj?: ?|fn(), functionName?: string|fn())",
    "!doc": "<p>Registers a function to be triggered when the page unloads.\nDeprecated, use on(window, &quot;beforeunload&quot;, lang.hitch(obj, functionName)) instead.</p>\n"
   },
   "addOnWindowUnload": {
    "!type": "fn(obj?: ?, functionName?: string|fn())",
    "!doc": "<p>registers a function to be triggered when window.onunload fires.\nBe careful trying to modify the DOM or access JavaScript properties\nduring this phase of page unloading: they may not always be available.\nConsider dojo.addOnUnload() if you need to modify the DOM or do heavy\nJavaScript work.</p>\n"
   },
   "anim": {
    "!type": "fn(node: +Node|string, properties: ?, duration?: +Integer, easing?: fn(), onEnd?: fn(), delay?: +Integer)",
    "!doc": "<p>A simpler interface to <code>animateProperty()</code>, also returns\nan instance of <code>Animation</code> but begins the animation\nimmediately, unlike nearly every other Dojo animation API.</p>\n"
   },
   "animateProperty": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Returns an animation that will transition the properties of\nnode defined in <code>args</code> depending how they are defined in\n<code>args.properties</code>\n</p>\n"
   },
   "Animation": {
    "!type": "fn(args: ?)",
    "!doc": "<p>A generic animation class that fires callbacks into its handlers\nobject at various states.</p>\n"
   },
   "attr": {
    "!type": "fn(node: +Node|string, name: string|?, value?: string)",
    "!doc": "<p>Gets or sets an attribute on an HTML element.</p>\n"
   },
   "blendColors": {
    "!type": "fn(start: +dojo, end: +dojo, weight: number, obj?: +dojo)",
    "!doc": "<p>Blend colors end and start with weight from 0 to 1, 0.5 being a 50/50 blend,\ncan reuse a previously allocated Color object for the result</p>\n"
   },
   "body": {
    "!type": "fn(doc?: +Document)",
    "!doc": "<p>Return the body element of the specified document or of dojo/_base/window::doc.</p>\n"
   },
   "byId": {
    "!type": "fn(id: string|+Node, doc?: +Document)",
    "!doc": "<p>Returns DOM node with matching <code>id</code> attribute or falsy value (ex: null or undefined)\nif not found.  If <code>id</code> is a DomNode, this function is a no-op.\n</p>\n"
   },
   "cache": {
    "!type": "fn(module: string|?, url: string, value?: string|?)",
    "!doc": "<p>A getter and setter for storing the string content associated with the\nmodule and url arguments.</p>\n"
   },
   "clearCache": {
    "!type": "fn()"
   },
   "Color": {
    "!type": "fn(color: []|string|?)",
    "!doc": "<p>Takes a named string, hex string, array of rgb or rgba values,\nan object with r, g, b, and a properties, or another <code>Color</code> object\nand creates a new Color instance to work from.\n</p>\n"
   },
   "colorFromArray": {
    "!type": "fn(a: [], obj?: +dojo)",
    "!doc": "<p>Builds a <code>Color</code> from a 3 or 4 element array, mapping each\nelement in sequence to the rgb(a) values of the color.</p>\n"
   },
   "colorFromHex": {
    "!type": "fn(color: string, obj?: +dojo)",
    "!doc": "<p>Converts a hex string with a &#39;#&#39; prefix to a color object.\nSupports 12-bit #rgb shorthand. Optionally accepts a\n<code>Color</code> object to update with the parsed value.\n</p>\n"
   },
   "colorFromRgb": {
    "!type": "fn(color: string, obj?: +dojo)",
    "!doc": "<p>get rgb(a) array from css-style color declarations</p>\n"
   },
   "colorFromString": {
    "!type": "fn(str: string, obj?: +dojo)",
    "!doc": "<p>Parses <code>str</code> for a color value. Accepts hex, rgb, and rgba\nstyle color values.</p>\n"
   },
   "connect": {
    "!type": "fn(obj?: ?, event: string, context: ?|?, method: string|fn(), dontFix?: bool)",
    "!doc": "<p><code>dojo.connect</code> is a deprecated event handling and delegation method in\nDojo. It allows one function to &quot;listen in&quot; on the execution of\nany other, triggering the second whenever the first is called. Many\nlisteners may be attached to a function, and source functions may\nbe either regular function calls or DOM events.\n</p>\n"
   },
   "contentBox": {
    "!type": "fn(node: +Node|string, box?: ?)",
    "!doc": "<p>Getter/setter for the content-box of node.</p>\n"
   },
   "cookie": {
    "!type": "fn(name: string, value?: string, props?: ?)",
    "!doc": "<p>Get or set a cookie.</p>\n"
   },
   "coords": {
    "!type": "fn(node: +Node|string, includeScroll?: bool)",
    "!doc": "<p>Deprecated: Use position() for border-box x/y/w/h\nor marginBox() for margin-box w/h/l/t.\n\n</p>\n<p>Returns an object that measures margin-box (w)idth/(h)eight\nand absolute position x/y of the border-box. Also returned\nis computed (l)eft and (t)op values in pixels from the\nnode&#39;s offsetParent as returned from marginBox().\nReturn value will be in the form:\n\n</p>\n<pre><code>{ l: 50, t: 200, w: 300: h: 150, x: 100, y: 300 }</code></pre>\n<p>Does not act as a setter. If includeScroll is passed, the x and\n\n</p>\n<p>y params are affected as one would expect in dojo.position().</p>\n"
   },
   "create": {
    "!type": "fn(tag: +Node|string, attrs: ?, refNode?: +Node|string, pos?: string)",
    "!doc": "<p>Create an element, allowing for optional attribute decoration\nand placement.</p>\n"
   },
   "declare": {
    "!type": "fn(className?: string, superclass: fn()|fn(), props: ?) -> +dojo",
    "!doc": "<p>Create a feature-rich constructor from compact notation.</p>\n"
   },
   "Deferred": {
    "!type": "fn(canceller?: fn())",
    "!doc": "<p>Deprecated.   This module defines the legacy dojo/_base/Deferred API.\nNew code should use dojo/Deferred instead.</p>\n"
   },
   "DeferredList": {
    "!type": "fn(list: [], fireOnOneCallback?: bool, fireOnOneErrback?: bool, consumeErrors?: bool, canceller?: fn())",
    "!doc": "<p>Deprecated, use dojo/promise/all instead.\nProvides event handling for a group of Deferred objects.</p>\n"
   },
   "deprecated": {
    "!type": "fn(behaviour: string, extra?: string, removal?: string)",
    "!doc": "<p>Log a debug message to indicate that a behavior has been\ndeprecated.</p>\n"
   },
   "destroy": {
    "!type": "fn(node: ?)"
   },
   "disconnect": {
    "!type": "fn(handle: +Handle)",
    "!doc": "<p>Remove a link created by dojo.connect.</p>\n"
   },
   "docScroll": {
    "!type": "fn(doc?: +Document)",
    "!doc": "<p>Returns an object with {node, x, y} with corresponding offsets.</p>\n"
   },
   "empty": {
    "!type": "fn(node: ?)"
   },
   "eval": {
    "!type": "fn(scriptText: ?)",
    "!doc": "<p>A legacy method created for use exclusively by internal Dojo methods. Do not use this method\ndirectly unless you understand its possibly-different implications on the platforms your are targeting.</p>\n"
   },
   "every": {
    "!type": "fn(arr: []|string, callback: fn()|string, thisObject?: ?) -> bool",
    "!doc": "<p>Determines whether or not every item in arr satisfies the\ncondition implemented by callback.</p>\n"
   },
   "exit": {
    "!type": "fn(exitcode: ?)"
   },
   "experimental": {
    "!type": "fn(moduleName: string, extra?: string)",
    "!doc": "<p>Marks code as experimental.</p>\n"
   },
   "fadeIn": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Returns an animation that will fade node defined in &#39;args&#39; from\nits current opacity to fully opaque.</p>\n"
   },
   "fadeOut": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Returns an animation that will fade node defined in &#39;args&#39;\nfrom its current opacity to fully transparent.</p>\n"
   },
   "fieldToObject": {
    "!type": "fn(inputNode: +Node|string)",
    "!doc": "<p>Serialize a form field to a JavaScript object.</p>\n"
   },
   "filter": {
    "!type": "fn(arr: [], callback: fn()|string, thisObject?: ?) -> []",
    "!doc": "<p>Returns a new Array with those items from arr that match the\ncondition implemented by callback.</p>\n"
   },
   "fixEvent": {
    "!type": "fn(evt: +Event, sender: +Node) -> +Event",
    "!doc": "<p>normalizes properties on the event object including event\nbubbling methods, keystroke normalization, and x/y positions</p>\n"
   },
   "fixIeBiDiScrollLeft": {
    "!type": "fn(scrollLeft: number, doc?: +Document) -> number",
    "!doc": "<p>In RTL direction, scrollLeft should be a negative value, but IE\nreturns a positive one. All codes using documentElement.scrollLeft\nmust call this function to fix this error, otherwise the position\nwill offset to right when there is a horizontal scrollbar.</p>\n"
   },
   "forEach": {
    "!type": "fn(arr: []|string, callback: fn()|string, thisObject?: ?)",
    "!doc": "<p>for every item in arr, callback is invoked. Return values are ignored.\nIf you want to break out of the loop, consider using array.every() or array.some().\nforEach does not allow breaking out of the loop over the items in arr.</p>\n"
   },
   "formToJson": {
    "!type": "fn(formNode: +Node|string, prettyPrint?: bool) -> string",
    "!doc": "<p>Create a serialized JSON string from a form node or string\nID identifying the form to serialize</p>\n"
   },
   "formToObject": {
    "!type": "fn(formNode: +Node|string)",
    "!doc": "<p>Serialize a form node to a JavaScript object.</p>\n"
   },
   "formToQuery": {
    "!type": "fn(formNode: +Node|string) -> string",
    "!doc": "<p>Returns a URL-encoded string representing the form passed as either a\nnode or string ID identifying the form to serialize</p>\n"
   },
   "fromJson": {
    "!type": "fn(js: string)",
    "!doc": "<p>Parses a JavaScript expression and returns a JavaScript value.</p>\n"
   },
   "getAttr": {
    "!type": "fn(node: +Node|string, name: string)",
    "!doc": "<p>Gets an attribute on an HTML element.</p>\n"
   },
   "getBorderExtents": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>returns an object with properties useful for noting the border\ndimensions.</p>\n"
   },
   "getComputedStyle": {
    "!type": "fn(node: +Node)",
    "!doc": "<p>Returns a &quot;computed style&quot; object.\n</p>\n"
   },
   "getContentBox": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>Returns an object that encodes the width, height, left and top\npositions of the node&#39;s content box, irrespective of the\ncurrent box model.</p>\n"
   },
   "getIeDocumentElementOffset": {
    "!type": "fn(doc?: +Document)",
    "!doc": "<p>returns the offset in x and y from the document body to the\nvisual edge of the page for IE</p>\n"
   },
   "getL10nName": {
    "!type": "fn(moduleName: ?, bundleName: ?, locale: ?) -> string"
   },
   "getMarginBox": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>returns an object that encodes the width, height, left and top\npositions of the node&#39;s margin box.</p>\n"
   },
   "getMarginExtents": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>returns object with properties useful for box fitting with\nregards to box margins (i.e., the outer-box).\n\n</p>\n<ul>\n<li>l/t = marginLeft, marginTop, respectively</li>\n<li>w = total width, margin inclusive</li>\n<li>h = total height, margin inclusive</li>\n</ul>\n<p>The w/h are used for calculating boxes.\nNormally application code will not need to invoke this\ndirectly, and will use the ...box... functions instead.</p>\n"
   },
   "getMarginSize": {
    "!type": "fn(node: +Node|string, computedStyle?: ?)",
    "!doc": "<p>returns an object that encodes the width and height of\nthe node&#39;s margin box</p>\n"
   },
   "getNodeProp": {
    "!type": "fn(node: +Node|string, name: string)",
    "!doc": "<p>Returns an effective value of a property or an attribute.</p>\n"
   },
   "getPadBorderExtents": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>Returns object with properties useful for box fitting with\nregards to padding.</p>\n"
   },
   "getPadExtents": {
    "!type": "fn(node: +Node, computedStyle?: ?)",
    "!doc": "<p>Returns object with special values specifically useful for node\nfitting.</p>\n"
   },
   "getProp": {
    "!type": "fn(node: +Node|string, name: string)",
    "!doc": "<p>Gets a property on an HTML element.</p>\n"
   },
   "getStyle": {
    "!type": "fn(node: +Node|string, name?: string)",
    "!doc": "<p>Accesses styles on a node.</p>\n"
   },
   "hasAttr": {
    "!type": "fn(node: +Node|string, name: string) -> bool",
    "!doc": "<p>Returns true if the requested attribute is specified on the\ngiven element, and false otherwise.</p>\n"
   },
   "hasClass": {
    "!type": "fn(node: string|+Node, classStr: string) -> bool",
    "!doc": "<p>Returns whether or not the specified classes are a portion of the\nclass list currently applied to the node.</p>\n"
   },
   "hash": {
    "!type": "fn(hash?: string, replace?: bool)",
    "!doc": "<p>Gets or sets the hash string in the browser URL.</p>\n"
   },
   "indexOf": {
    "!type": "fn(arr: [], value: ?, fromIndex?: +Integer, findLast?: bool) -> number",
    "!doc": "<p>locates the first index of the provided value in the\npassed array. If the value is not found, -1 is returned.</p>\n"
   },
   "isBodyLtr": {
    "!type": "fn(doc?: +Document) -> bool",
    "!doc": "<p>Returns true if the current language is left-to-right, and false otherwise.</p>\n"
   },
   "isDescendant": {
    "!type": "fn(node: +Node|string, ancestor: +Node|string) -> bool",
    "!doc": "<p>Returns true if node is a descendant of ancestor</p>\n"
   },
   "lastIndexOf": {
    "!type": "fn(arr: ?, value: ?, fromIndex?: +Integer) -> number",
    "!doc": "<p>locates the last index of the provided value in the passed\narray. If the value is not found, -1 is returned.</p>\n"
   },
   "loadInit": {
    "!type": "fn(f: ?)"
   },
   "map": {
    "!type": "fn(arr: []|string, callback: fn()|string, thisObject?: ?, Ctr: ?) -> []",
    "!doc": "<p>applies callback to each element of arr and returns\nan Array with the results</p>\n"
   },
   "marginBox": {
    "!type": "fn(node: +Node|string, box?: ?)",
    "!doc": "<p>Getter/setter for the margin-box of node.</p>\n"
   },
   "moduleUrl": {
    "!type": "fn(module: string, url?: string) -> string",
    "!doc": "<p>Returns a URL relative to a module.</p>\n"
   },
   "NodeList": {
    "!type": "fn(array: ?) -> []",
    "!doc": "<p>Array-like object which adds syntactic\nsugar for chaining, common iteration operations, animation, and\nnode manipulation. NodeLists are most often returned as the\nresult of dojo/query() calls.</p>\n"
   },
   "objectToQuery": {
    "!type": "fn(map: ?)",
    "!doc": "<p>takes a name/value mapping object and returns a string representing\na URL-encoded version of that object.</p>\n"
   },
   "place": {
    "!type": "fn(node: +Node|+DocumentFragment|string, refNode: +Node|string, position?: string|number) -> +Node",
    "!doc": "<p>Attempt to insert node into the DOM, choosing from various positioning options.\nReturns the first argument resolved to a DOM node.</p>\n"
   },
   "platformRequire": {
    "!type": "fn(modMap: ?)",
    "!doc": "<p>require one or more modules based on which host environment\nDojo is currently operating in</p>\n"
   },
   "popContext": {
    "!type": "fn()",
    "!doc": "<p>If the context stack contains elements, ensure that\nsubsequent code executes in the <em>previous</em> context to the\ncurrent context. The current context set ([global,\ndocument]) is returned.</p>\n"
   },
   "position": {
    "!type": "fn(node: +Node|string, includeScroll?: bool)",
    "!doc": "<p>Gets the position and size of the passed element relative to\nthe viewport (if includeScroll==false), or relative to the\ndocument root (if includeScroll==true).\n</p>\n"
   },
   "prop": {
    "!type": "fn(node: +Node|string, name: string|?, value?: string)",
    "!doc": "<p>Gets or sets a property on an HTML element.</p>\n"
   },
   "provide": {
    "!type": "fn(mid: ?)"
   },
   "pushContext": {
    "!type": "fn(g?: ?|string, d?: +MDocumentElement)",
    "!doc": "<p>causes subsequent calls to Dojo methods to assume the\npassed object and, optionally, document as the default\nscopes to use. A 2-element array of the previous global and\ndocument are returned.</p>\n"
   },
   "queryToObject": {
    "!type": "fn(str: string)",
    "!doc": "<p>Create an object representing a de-serialized query section of a\nURL. Query keys with multiple values are returned in an array.\n</p>\n"
   },
   "rawXhrPost": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP POST request to the server. In addition to the properties\nlisted for the dojo.__XhrArgs type, the following property is allowed:</p>\n"
   },
   "rawXhrPut": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP PUT request to the server. In addition to the properties\nlisted for the dojo.__XhrArgs type, the following property is allowed:</p>\n"
   },
   "ready": {
    "!type": "fn(priority?: +Integer, context: ?, callback?: fn())",
    "!doc": "<p>Add a function to execute on DOM content loaded and all requested modules have arrived and been evaluated.\nIn most cases, the <code>domReady</code> plug-in should suffice and this method should not be needed.\n\n</p>\n<p>When called in a non-browser environment, just checks that all requested modules have arrived and been\nevaluated.</p>\n"
   },
   "registerModulePath": {
    "!type": "fn(moduleName: string, prefix: string)",
    "!doc": "<p>Maps a module name to a path</p>\n"
   },
   "removeAttr": {
    "!type": "fn(node: +Node|string, name: string)",
    "!doc": "<p>Removes an attribute from an HTML element.</p>\n"
   },
   "removeClass": {
    "!type": "fn(node: string|+Node, classStr?: string|[])",
    "!doc": "<p>Removes the specified classes from node. No <code>contains()</code>\ncheck is required.\n</p>\n"
   },
   "replaceClass": {
    "!type": "fn(node: string|+Node, addClassStr: string|[], removeClassStr?: string|[])",
    "!doc": "<p>Replaces one or more classes on a node if not present.\nOperates more quickly than calling dojo.removeClass and dojo.addClass\n</p>\n"
   },
   "require": {
    "!type": "fn(moduleName: string, omitModuleCheck?: bool)",
    "!doc": "<p>loads a Javascript module from the appropriate URI\n</p>\n"
   },
   "requireAfterIf": {
    "!type": "fn(condition: bool, moduleName: string, omitModuleCheck?: bool)",
    "!doc": "<p>If the condition is true then call <code>dojo.require()</code> for the specified\nresource\n</p>\n"
   },
   "requireIf": {
    "!type": "fn(condition: bool, moduleName: string, omitModuleCheck?: bool)",
    "!doc": "<p>If the condition is true then call <code>dojo.require()</code> for the specified\nresource\n</p>\n"
   },
   "requireLocalization": {
    "!type": "fn(moduleName: string, bundleName: string, locale?: string)"
   },
   "safeMixin": {
    "!type": "fn(target: ?, source: ?)",
    "!doc": "<p>Mix in properties skipping a constructor and decorating functions\nlike it is done by declare().</p>\n"
   },
   "setAttr": {
    "!type": "fn(node: +Node|string, name: string|?, value?: string)",
    "!doc": "<p>Sets an attribute on an HTML element.</p>\n"
   },
   "setContentSize": {
    "!type": "fn(node: +Node, box: ?, computedStyle?: ?)",
    "!doc": "<p>Sets the size of the node&#39;s contents, irrespective of margins,\npadding, or borders.</p>\n"
   },
   "setContext": {
    "!type": "fn(globalObject: ?, globalDocument: +DocumentElement)",
    "!doc": "<p>changes the behavior of many core Dojo functions that deal with\nnamespace and DOM lookup, changing them to work in a new global\ncontext (e.g., an iframe). The varibles dojo.global and dojo.doc\nare modified as a result of calling this function and the result of\n<code>dojo.body()</code> likewise differs.</p>\n"
   },
   "setMarginBox": {
    "!type": "fn(node: +Node, box: ?, computedStyle?: ?)",
    "!doc": "<p>sets the size of the node&#39;s margin box and placement\n(left/top), irrespective of box model. Think of it as a\npassthrough to setBox that handles box-model vagaries for\nyou.</p>\n"
   },
   "setProp": {
    "!type": "fn(node: +Node|string, name: string|?, value?: string)",
    "!doc": "<p>Sets a property on an HTML element.</p>\n"
   },
   "setSelectable": {
    "!type": "fn(node: ?, selectable: ?)"
   },
   "setStyle": {
    "!type": "fn(node: +Node|string, name: string|?, value?: string) -> string",
    "!doc": "<p>Sets styles on a node.</p>\n"
   },
   "some": {
    "!type": "fn(arr: []|string, callback: fn()|string, thisObject?: ?) -> bool",
    "!doc": "<p>Determines whether or not any item in arr satisfies the\ncondition implemented by callback.</p>\n"
   },
   "Stateful": {
    "!type": "fn()"
   },
   "stopEvent": {
    "!type": "fn(evt: +Event)",
    "!doc": "<p>prevents propagation and clobbers the default action of the\npassed event</p>\n"
   },
   "style": {
    "!type": "fn(node: +Node|string, name?: string|?, value?: string)",
    "!doc": "<p>Accesses styles on a node. If 2 arguments are\npassed, acts as a getter. If 3 arguments are passed, acts\nas a setter.</p>\n"
   },
   "toDom": {
    "!type": "fn(frag: string, doc?: +DocumentNode)",
    "!doc": "<p>instantiates an HTML fragment returning the corresponding DOM.</p>\n"
   },
   "toggleClass": {
    "!type": "fn(node: string|+Node, classStr: string|[], condition?: bool) -> bool",
    "!doc": "<p>Adds a class to node if not present, or removes if present.\nPass a boolean condition if you want to explicitly add or remove.\nReturns the condition that was specified directly or indirectly.\n</p>\n"
   },
   "toJson": {
    "!type": "fn(it: ?, prettyPrint?: bool)",
    "!doc": "<p>Returns a <a href=\"http://json.org\">JSON</a> serialization of an object.</p>\n"
   },
   "toPixelValue": {
    "!type": "fn(node: +Node, value: string) -> number",
    "!doc": "<p>converts style value to pixels on IE or return a numeric value.</p>\n"
   },
   "unsubscribe": {
    "!type": "fn(handle: +Handle)",
    "!doc": "<p>Remove a topic listener.</p>\n"
   },
   "when": {
    "!type": "fn(valueOrPromise: ?, callback?: fn(), errback?: fn(), progback?: fn()) -> +dojo",
    "!doc": "<p>Transparently applies callbacks to values and/or promises.</p>\n"
   },
   "windowUnloaded": {
    "!type": "fn()",
    "!doc": "<p>signal fired by impending window destruction. You may use\ndojo.addOnWIndowUnload() or dojo.connect() to this method to perform\npage/application cleanup methods. See dojo.addOnWindowUnload for more info.</p>\n"
   },
   "withDoc": {
    "!type": "fn(documentObject: +DocumentElement, callback: fn(), thisObject?: ?, cbArguments?: [])",
    "!doc": "<p>Invoke callback with documentObject as dojo/_base/window::doc.</p>\n"
   },
   "withGlobal": {
    "!type": "fn(globalObject: ?, callback: fn(), thisObject?: ?, cbArguments?: [])",
    "!doc": "<p>Invoke callback with globalObject as dojo.global and\nglobalObject.document as dojo.doc.</p>\n"
   },
   "xhr": {
    "!type": "fn(method: ?, args: ?)"
   },
   "xhrDelete": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP DELETE request to the server.</p>\n"
   },
   "xhrGet": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP GET request to the server.</p>\n"
   },
   "xhrPost": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP POST request to the server. In addition to the properties\nlisted for the dojo.__XhrArgs type, the following property is allowed:</p>\n"
   },
   "xhrPut": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sends an HTTP PUT request to the server. In addition to the properties\nlisted for the dojo.__XhrArgs type, the following property is allowed:</p>\n"
   }
  },
  "dojo/main.config": {
   "addOnLoad": {
    "!doc": "<p>Adds a callback via dojo/ready. Useful when Dojo is added after\nthe page loads and djConfig.afterOnLoad is true. Supports the same\narguments as dojo/ready. When using a function reference, use\n<code>djConfig.addOnLoad = function(){};</code>. For object with function name use\n<code>djConfig.addOnLoad = [myObject, &quot;functionName&quot;];</code> and for object with\nfunction reference use\n<code>djConfig.addOnLoad = [myObject, function(){}];</code></p>\n"
   },
   "afterOnLoad": {},
   "baseUrl": {},
   "callback": {
    "!doc": "<p>Defines a callback to be used when dependencies are defined before \nthe loader has been loaded. When provided, they cause the loader to \nexecute require(deps, callback) once it has finished loading. \nShould be used with deps.</p>\n"
   },
   "debugContainerId": {},
   "debugHeight": {},
   "defaultDuration": {
    "!doc": "<p>Default duration, in milliseconds, for wipe and fade animations within dijits.\nAssigned to dijit.defaultDuration.</p>\n"
   },
   "deferredInstrumentation": {
    "!doc": "<p>Whether deferred instrumentation should be loaded or included\nin builds.</p>\n"
   },
   "deps": {
    "!doc": "<p>Defines dependencies to be used before the loader has been loaded.\nWhen provided, they cause the loader to execute require(deps, callback) \nonce it has finished loading. Should be used with callback.</p>\n"
   },
   "dojoBlankHtmlUrl": {
    "!doc": "<p>Used by some modules to configure an empty iframe. Used by dojo/io/iframe and\ndojo/back, and dijit/popup support in IE where an iframe is needed to make sure native\ncontrols do not bleed through the popups. Normally this configuration variable\ndoes not need to be set, except when using cross-domain/CDN Dojo builds.\nSave dojo/resources/blank.html to your domain and set <code>djConfig.dojoBlankHtmlUrl</code>\nto the path on your domain your copy of blank.html.</p>\n"
   },
   "extraLocale": {
    "!doc": "<p>No default value. Specifies additional locales whose\nresources should also be loaded alongside the default locale when\ncalls to <code>dojo.requireLocalization()</code> are processed.</p>\n"
   },
   "ioPublish": {
    "!doc": "<p>Set this to true to enable publishing of topics for the different phases of\nIO operations. Publishing is done via dojo/topic.publish(). See dojo/main.__IoPublish for a list\nof topics that are published.</p>\n"
   },
   "isDebug": {
    "!doc": "<p>Defaults to <code>false</code>. If set to <code>true</code>, ensures that Dojo provides\nextended debugging feedback via Firebug. If Firebug is not available\non your platform, setting <code>isDebug</code> to <code>true</code> will force Dojo to\npull in (and display) the version of Firebug Lite which is\nintegrated into the Dojo distribution, thereby always providing a\ndebugging/logging console when <code>isDebug</code> is enabled. Note that\nFirebug&#39;s <code>console.*</code> methods are ALWAYS defined by Dojo. If\n<code>isDebug</code> is false and you are on a platform without Firebug, these\nmethods will be defined as no-ops.</p>\n"
   },
   "locale": {
    "!doc": "<p>The locale to assume for loading localized resources in this page,\nspecified according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a>.\nMust be specified entirely in lowercase, e.g. <code>en-us</code> and <code>zh-cn</code>.\nSee the documentation for <code>dojo.i18n</code> and <code>dojo.requireLocalization</code>\nfor details on loading localized resources. If no locale is specified,\nDojo assumes the locale of the user agent, according to <code>navigator.userLanguage</code>\nor <code>navigator.language</code> properties.</p>\n"
   },
   "modulePaths": {
    "!doc": "<p>A map of module names to paths relative to <code>dojo.baseUrl</code>. The\nkey/value pairs correspond directly to the arguments which\n<code>dojo.registerModulePath</code> accepts. Specifying\n<code>djConfig.modulePaths = { &quot;foo&quot;: &quot;../../bar&quot; }</code> is the equivalent\nof calling <code>dojo.registerModulePath(&quot;foo&quot;, &quot;../../bar&quot;);</code>. Multiple\nmodules may be configured via <code>djConfig.modulePaths</code>.</p>\n"
   },
   "parseOnLoad": {
    "!doc": "<p>Run the parser after the page is loaded</p>\n"
   },
   "require": {
    "!doc": "<p>An array of module names to be loaded immediately after dojo.js has been included\nin a page.</p>\n"
   },
   "transparentColor": {
    "!doc": "<p>Array containing the r, g, b components used as transparent color in dojo.Color;\nif undefined, [255,255,255] (white) will be used.</p>\n"
   },
   "urchin": {
    "!doc": "<p>Used by <code>dojox.analytics.Urchin</code> as the default UA-123456-7 account\nnumber used when being created. Alternately, you can pass an acct:&quot;&quot;\nparameter to the constructor a la: new dojox.analytics.Urchin({ acct:&quot;UA-123456-7&quot; });</p>\n"
   },
   "useCustomLogger": {
    "!doc": "<p>If set to a value that evaluates to true such as a string or array and\nisDebug is true and Firebug is not available or running, then it bypasses\nthe creation of Firebug Lite allowing you to define your own console object.</p>\n"
   },
   "useDeferredInstrumentation": {
    "!doc": "<p>Whether the deferred instrumentation should be used.\n\n</p>\n<ul>\n<li><code>&quot;report-rejections&quot;</code>: report each rejection as it occurs.</li>\n<li><code>true</code> or <code>1</code> or <code>&quot;report-unhandled-rejections&quot;</code>: wait 1 second\n  in an attempt to detect unhandled rejections.</li>\n</ul>\n"
   }
  },
  "dojo/main.global": {
   "undefined_callbacks": {},
   "$": {
    "!type": "fn()"
   },
   "GoogleSearchStoreCallback_undefined_NaN": {
    "!type": "fn(start: ?, data: ?, responseCode: ?, errorMsg: ?)"
   },
   "jQuery": {
    "!type": "fn()"
   },
   "swfIsInHTML": {
    "!type": "fn()"
   },
   "undefined_onload": {
    "!type": "fn()"
   }
  },
  "dojo/main.dijit": {
   "_editor": {},
   "_tree": {},
   "form": {},
   "layout": {},
   "range": {
    "!doc": "<p>W3C range API</p>\n"
   },
   "registry": {},
   "tree": {},
   "byId": {
    "!type": "fn(id: ?)"
   },
   "Calendar": {
    "!type": "fn()"
   },
   "CalendarLite": {
    "!type": "fn()"
   },
   "CheckedMenuItem": {
    "!type": "fn()"
   },
   "ColorPalette": {
    "!type": "fn()"
   },
   "Declaration": {
    "!type": "fn()"
   },
   "Destroyable": {
    "!type": "fn()"
   },
   "Dialog": {
    "!type": "fn()"
   },
   "DialogUnderlay": {
    "!type": "fn()"
   },
   "DropDownMenu": {
    "!type": "fn()"
   },
   "Dye": {
    "!type": "fn()"
   },
   "Editor": {
    "!type": "fn()"
   },
   "Fieldset": {
    "!type": "fn()"
   },
   "InlineEditBox": {
    "!type": "fn()"
   },
   "Menu": {
    "!type": "fn()"
   },
   "MenuBar": {
    "!type": "fn()"
   },
   "MenuBarItem": {
    "!type": "fn()"
   },
   "MenuItem": {
    "!type": "fn()"
   },
   "MenuSeparator": {
    "!type": "fn()"
   },
   "PopupMenuBarItem": {
    "!type": "fn()"
   },
   "PopupMenuItem": {
    "!type": "fn()"
   },
   "ProgressBar": {
    "!type": "fn()"
   },
   "RadioButtonMenuItem": {
    "!type": "fn()"
   },
   "TitlePane": {
    "!type": "fn()"
   },
   "Toolbar": {
    "!type": "fn()"
   },
   "ToolbarSeparator": {
    "!type": "fn()"
   },
   "Tooltip": {
    "!type": "fn()"
   },
   "TooltipDialog": {
    "!type": "fn()"
   },
   "Tree": {
    "!type": "fn()"
   },
   "WidgetSet": {
    "!type": "fn()"
   }
  },
  "dojo/main.dojox": {
   "analytics": {},
   "app": {},
   "atom": {},
   "av": {},
   "BidiComplex": {},
   "calc": {},
   "calendar": {},
   "charting": {},
   "collections": {},
   "color": {},
   "css3": {},
   "data": {},
   "date": {},
   "dgauges": {},
   "dnd": {},
   "drawing": {},
   "dtl": {},
   "editor": {},
   "embed": {},
   "encoding": {},
   "enhanced": {},
   "flash": {
    "!doc": "<p>Utilities to embed and communicate with the Flash player from Javascript</p>\n"
   },
   "form": {},
   "fx": {},
   "gantt": {},
   "gauges": {},
   "geo": {},
   "gesture": {},
   "gfx": {},
   "gfx3d": {},
   "grid": {},
   "help": {
    "!doc": "<p>Adds the help function to all variables.</p>\n"
   },
   "highlight": {},
   "html": {},
   "image": {},
   "io": {},
   "jq": {},
   "json": {},
   "jsonPath": {},
   "lang": {},
   "layout": {},
   "math": {},
   "mdnd": {},
   "mobile": {},
   "mvc": {},
   "openlayers": {},
   "rails": {},
   "robot": {},
   "rpc": {},
   "secure": {},
   "sketch": {},
   "sql": {},
   "storage": {},
   "string": {},
   "testing": {},
   "timing": {},
   "treemap": {},
   "uuid": {},
   "validate": {},
   "widget": {},
   "xml": {},
   "sprintf": {
    "!type": "fn(format: string, filler: +mixed...)"
   }
  },
  "dojo/main.scopeMap": {
   "dijit": {},
   "dojo": {},
   "dojox": {}
  },
  "dojo/main.version": {
   "flag": {},
   "major": {},
   "minor": {},
   "patch": {},
   "revision": {},
   "toString": {
    "!type": "fn() -> string"
   }
  },
  "dojo/main._hasResource": {},
  "dojo/main.doc": {
   "documentElement": {},
   "dojoClick": {}
  },
  "dojo/main.mouseButtons": {
   "LEFT": {
    "!doc": "<p>Numeric value of the left mouse button for the platform.</p>\n"
   },
   "MIDDLE": {
    "!doc": "<p>Numeric value of the middle mouse button for the platform.</p>\n"
   },
   "RIGHT": {
    "!doc": "<p>Numeric value of the right mouse button for the platform.</p>\n"
   },
   "isButton": {
    "!type": "fn(e: +Event, button: number) -> bool",
    "!doc": "<p>Checks an event object for a pressed button</p>\n"
   },
   "isLeft": {
    "!type": "fn(e: +Event) -> bool",
    "!doc": "<p>Checks an event object for the pressed left button</p>\n"
   },
   "isMiddle": {
    "!type": "fn(e: +Event) -> bool",
    "!doc": "<p>Checks an event object for the pressed middle button</p>\n"
   },
   "isRight": {
    "!type": "fn(e: +Event) -> bool",
    "!doc": "<p>Checks an event object for the pressed right button</p>\n"
   }
  },
  "dojo/main.keys": {
   "ALT": {},
   "BACKSPACE": {},
   "CAPS_LOCK": {},
   "CLEAR": {},
   "copyKey": {},
   "CTRL": {},
   "DELETE": {},
   "DOWN_ARROW": {},
   "DOWN_DPAD": {},
   "END": {},
   "ENTER": {},
   "ESCAPE": {},
   "F1": {},
   "F10": {},
   "F11": {},
   "F12": {},
   "F13": {},
   "F14": {},
   "F15": {},
   "F2": {},
   "F3": {},
   "F4": {},
   "F5": {},
   "F6": {},
   "F7": {},
   "F8": {},
   "F9": {},
   "HELP": {},
   "HOME": {},
   "INSERT": {},
   "LEFT_ARROW": {},
   "LEFT_DPAD": {},
   "LEFT_WINDOW": {},
   "META": {},
   "NUM_LOCK": {},
   "NUMPAD_0": {},
   "NUMPAD_1": {},
   "NUMPAD_2": {},
   "NUMPAD_3": {},
   "NUMPAD_4": {},
   "NUMPAD_5": {},
   "NUMPAD_6": {},
   "NUMPAD_7": {},
   "NUMPAD_8": {},
   "NUMPAD_9": {},
   "NUMPAD_DIVIDE": {},
   "NUMPAD_ENTER": {},
   "NUMPAD_MINUS": {},
   "NUMPAD_MULTIPLY": {},
   "NUMPAD_PERIOD": {},
   "NUMPAD_PLUS": {},
   "PAGE_DOWN": {},
   "PAGE_UP": {},
   "PAUSE": {},
   "RIGHT_ARROW": {},
   "RIGHT_DPAD": {},
   "RIGHT_WINDOW": {},
   "SCROLL_LOCK": {},
   "SELECT": {},
   "SHIFT": {},
   "SPACE": {},
   "TAB": {},
   "UP_ARROW": {},
   "UP_DPAD": {}
  },
  "dojo/main._nodeDataCache": {},
  "dojo/main.fx": {
   "easing": {
    "!doc": "<p>Collection of easing functions to use beyond the default\n<code>dojo._defaultEasing</code> function.</p>\n"
   },
   "chain": {
    "!type": "fn(animations: +dojo)",
    "!doc": "<p>Chain a list of <code>dojo/_base/fx.Animation</code>s to run in sequence\n</p>\n"
   },
   "combine": {
    "!type": "fn(animations: +dojo)",
    "!doc": "<p>Combine a list of <code>dojo/_base/fx.Animation</code>s to run in parallel\n</p>\n"
   },
   "slideTo": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Slide a node to a new top/left position\n</p>\n"
   },
   "Toggler": {
    "!type": "fn()"
   },
   "wipeIn": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Expand a node to it&#39;s natural height.\n</p>\n"
   },
   "wipeOut": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Shrink a node to nothing and hide it.\n</p>\n"
   }
  },
  "dojo/main.date": {
   "stamp": {
    "!doc": "<p>TODOC</p>\n"
   },
   "add": {
    "!type": "fn(date: +Date, interval: string, amount: number)",
    "!doc": "<p>Add to a Date in intervals of different size, from milliseconds to years</p>\n"
   },
   "compare": {
    "!type": "fn(date1: +Date, date2?: +Date, portion?: string) -> number",
    "!doc": "<p>Compare two date objects by date, time, or both.</p>\n"
   },
   "difference": {
    "!type": "fn(date1: +Date, date2?: +Date, interval?: string)",
    "!doc": "<p>Get the difference in a specific unit of time (e.g., number of\nmonths, weeks, days, etc.) between two dates, rounded to the\nnearest integer.</p>\n"
   },
   "getDaysInMonth": {
    "!type": "fn(dateObject: +Date) -> number",
    "!doc": "<p>Returns the number of days in the month used by dateObject</p>\n"
   },
   "getTimezoneName": {
    "!type": "fn(dateObject: +Date)",
    "!doc": "<p>Get the user&#39;s time zone as provided by the browser</p>\n"
   },
   "isLeapYear": {
    "!type": "fn(dateObject: +Date) -> bool",
    "!doc": "<p>Determines if the year of the dateObject is a leap year</p>\n"
   }
  },
  "dojo/main.html": {
   "set": {
    "!type": "fn(node: +Node, cont: string|+Node|+NodeList, params?: ?)",
    "!doc": "<p>inserts (replaces) the given content into the given node. dojo/dom-construct.place(cont, node, &quot;only&quot;)\nmay be a better choice for simple HTML insertion.</p>\n"
   }
  },
  "dojo/main.Stateful": {
   "!type": "fn()",
   "prototype": {
    "_attrPairNames": {
     "!doc": "<p>Used across all instances a hash to cache attribute names and their getter \nand setter names.</p>\n"
    },
    "get": {
     "!type": "fn(name: string)",
     "!doc": "<p>Get a property on a Stateful instance.</p>\n"
    },
    "postscript": {
     "!type": "fn(params?: ?)"
    },
    "set": {
     "!type": "fn(name: string, value: ?) -> fn()",
     "!doc": "<p>Set a property on a Stateful instance</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dojo/main.contentHandlers": {
   "auto": {
    "!type": "fn(xhr: ?)"
   },
   "javascript": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which evaluates the response data, expecting it to be valid JavaScript</p>\n"
   },
   "json": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which returns a JavaScript object created from the response data</p>\n"
   },
   "json-comment-filtered": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which expects comment-filtered JSON.</p>\n"
   },
   "json-comment-optional": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which checks the presence of comment-filtered JSON and\nalternates between the <code>json</code> and <code>json-comment-filtered</code> contentHandlers.</p>\n"
   },
   "text": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which simply returns the plaintext response data</p>\n"
   },
   "xml": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler returning an XML Document parsed from the response data</p>\n"
   }
  },
  "dojo/main._contentHandlers": {
   "auto": {
    "!type": "fn(xhr: ?)"
   },
   "javascript": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which evaluates the response data, expecting it to be valid JavaScript</p>\n"
   },
   "json": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which returns a JavaScript object created from the response data</p>\n"
   },
   "json-comment-filtered": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which expects comment-filtered JSON.</p>\n"
   },
   "json-comment-optional": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which checks the presence of comment-filtered JSON and\nalternates between the <code>json</code> and <code>json-comment-filtered</code> contentHandlers.</p>\n"
   },
   "text": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler which simply returns the plaintext response data</p>\n"
   },
   "xml": {
    "!type": "fn(xhr: ?)",
    "!doc": "<p>A contentHandler returning an XML Document parsed from the response data</p>\n"
   }
  },
  "dojo/main.__IoArgs": {
   "!type": "fn()",
   "prototype": {
    "content": {
     "!doc": "<p>Contains properties with string values. These\nproperties will be serialized as name1=value2 and\npassed in the request.</p>\n"
    },
    "form": {
     "!doc": "<p>DOM node for a form. Used to extract the form values\nand send to the server.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>Acceptable values depend on the type of IO\ntransport (see specific IO calls for more information).</p>\n"
    },
    "ioPublish": {
     "!doc": "<p>Set this explicitly to false to prevent publishing of topics related to\nIO operations. Otherwise, if djConfig.ioPublish is set to true, topics\nwill be published via dojo/topic.publish() for different phases of an IO operation.\nSee dojo/main.__IoPublish for a list of topics that are published.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Default is false. If true, then a\n&quot;dojo.preventCache&quot; parameter is sent in the request\nwith a value that changes with each request\n(timestamp). Useful only with GET-type requests.</p>\n"
    },
    "rawBody": {
     "!doc": "<p>Sets the raw body for an HTTP request. If this is used, then the content\nproperty is ignored. This is mostly useful for HTTP methods that have\na body to their requests, like PUT or POST. This property can be used instead\nof postData and putData for dojo/_base/xhr.rawXhrPost and dojo/_base/xhr.rawXhrPut respectively.</p>\n"
    },
    "timeout": {
     "!doc": "<p>Milliseconds to wait for the response. If this time\npasses, the then error callbacks are called.</p>\n"
    },
    "url": {
     "!doc": "<p>URL to server endpoint.</p>\n"
    },
    "error": {
     "!type": "fn(response: ?, ioArgs: +dojo)",
     "!doc": "<p>This function will\nbe called when the request fails due to a network or server error, the url\nis invalid, etc. It will also be called if the load or handle callback throws an\nexception, unless djConfig.debugAtAllCosts is true.  This allows deployed applications\nto continue to run even when a logic error happens in the callback, while making\nit easier to troubleshoot while in debug mode.</p>\n"
    },
    "handle": {
     "!type": "fn(loadOrError: string, response: ?, ioArgs: +dojo)",
     "!doc": "<p>This function will\nbe called at the end of every request, whether or not an error occurs.</p>\n"
    },
    "load": {
     "!type": "fn(response: ?, ioArgs: +dojo)",
     "!doc": "<p>This function will be\ncalled on a successful HTTP response code.</p>\n"
    }
   }
  },
  "dojo/main.__IoCallbackArgs": {
   "!type": "fn()",
   "prototype": {
    "args": {
     "!doc": "<p>the original object argument to the IO call.</p>\n"
    },
    "canDelete": {
     "!doc": "<p>For dojo/io/script calls only, indicates\nwhether the script tag that represents the\nrequest can be deleted after callbacks have\nbeen called. Used internally to know when\ncleanup can happen on JSONP-type requests.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>The final indicator on how the response will be\nhandled.</p>\n"
    },
    "id": {
     "!doc": "<p>For dojo/io/script calls only, the internal\nscript ID used for the request.</p>\n"
    },
    "json": {
     "!doc": "<p>For dojo/io/script calls only: holds the JSON\nresponse for JSONP-type requests. Used\ninternally to hold on to the JSON responses.\nYou should not need to access it directly --\nthe same object should be passed to the success\ncallbacks directly.</p>\n"
    },
    "query": {
     "!doc": "<p>For non-GET requests, the\nname1=value1&amp;name2=value2 parameters sent up in\nthe request.</p>\n"
    },
    "url": {
     "!doc": "<p>The final URL used for the call. Many times it\nwill be different than the original args.url\nvalue.</p>\n"
    },
    "xhr": {
     "!doc": "<p>For XMLHttpRequest calls only, the\nXMLHttpRequest object that was used for the\nrequest.</p>\n"
    }
   }
  },
  "dojo/main.__IoPublish": {
   "!type": "fn()",
   "prototype": {
    "done": {
     "!doc": "<p>&quot;/dojo/io/done&quot; is sent whenever an IO request has completed,\neither by loading or by erroring. It passes the error and\nthe dojo.Deferred for the request with the topic.</p>\n"
    },
    "error": {
     "!doc": "<p>&quot;/dojo/io/error&quot; is sent whenever an IO request has errored.\nIt passes the error and the dojo.Deferred\nfor the request with the topic.</p>\n"
    },
    "load": {
     "!doc": "<p>&quot;/dojo/io/load&quot; is sent whenever an IO request has loaded\nsuccessfully. It passes the response and the dojo.Deferred\nfor the request with the topic.</p>\n"
    },
    "send": {
     "!doc": "<p>&quot;/dojo/io/send&quot; is sent whenever a new IO request is started.\nIt passes the dojo.Deferred for the request with the topic.</p>\n"
    },
    "start": {
     "!doc": "<p>&quot;/dojo/io/start&quot; is sent when there are no outstanding IO\nrequests, and a new IO request is started. No arguments\nare passed with this topic.</p>\n"
    },
    "stop": {
     "!doc": "<p>&quot;/dojo/io/stop&quot; is sent when all outstanding IO requests have\nfinished. No arguments are passed with this topic.</p>\n"
    }
   }
  },
  "dojo/main.__XhrArgs": {
   "!type": "fn()",
   "prototype": {
    "content": {
     "!doc": "<p>Contains properties with string values. These\nproperties will be serialized as name1=value2 and\npassed in the request.</p>\n"
    },
    "contentType": {
     "!doc": "<p>&quot;application/x-www-form-urlencoded&quot; is default. Set to false to\nprevent a Content-Type header from being sent, or to a string\nto send a different Content-Type.</p>\n"
    },
    "failOk": {
     "!doc": "<p>false is default. Indicates whether a request should be\nallowed to fail (and therefore no console error message in\nthe event of a failure)</p>\n"
    },
    "form": {
     "!doc": "<p>DOM node for a form. Used to extract the form values\nand send to the server.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>Acceptable values are: text (default), json, json-comment-optional,\njson-comment-filtered, javascript, xml. See <code>dojo/_base/xhr.contentHandlers</code></p>\n"
    },
    "headers": {
     "!doc": "<p>Additional HTTP headers to send in the request.</p>\n"
    },
    "ioPublish": {
     "!doc": "<p>Set this explicitly to false to prevent publishing of topics related to\nIO operations. Otherwise, if djConfig.ioPublish is set to true, topics\nwill be published via dojo/topic.publish() for different phases of an IO operation.\nSee dojo/main.__IoPublish for a list of topics that are published.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Default is false. If true, then a\n&quot;dojo.preventCache&quot; parameter is sent in the request\nwith a value that changes with each request\n(timestamp). Useful only with GET-type requests.</p>\n"
    },
    "rawBody": {
     "!doc": "<p>Sets the raw body for an HTTP request. If this is used, then the content\nproperty is ignored. This is mostly useful for HTTP methods that have\na body to their requests, like PUT or POST. This property can be used instead\nof postData and putData for dojo/_base/xhr.rawXhrPost and dojo/_base/xhr.rawXhrPut respectively.</p>\n"
    },
    "sync": {
     "!doc": "<p>false is default. Indicates whether the request should\nbe a synchronous (blocking) request.</p>\n"
    },
    "timeout": {
     "!doc": "<p>Milliseconds to wait for the response. If this time\npasses, the then error callbacks are called.</p>\n"
    },
    "url": {
     "!doc": "<p>URL to server endpoint.</p>\n"
    },
    "error": {
     "!type": "fn(response: ?, ioArgs: +dojo)",
     "!doc": "<p>This function will\nbe called when the request fails due to a network or server error, the url\nis invalid, etc. It will also be called if the load or handle callback throws an\nexception, unless djConfig.debugAtAllCosts is true.  This allows deployed applications\nto continue to run even when a logic error happens in the callback, while making\nit easier to troubleshoot while in debug mode.</p>\n"
    },
    "handle": {
     "!type": "fn(loadOrError: string, response: ?, ioArgs: +dojo)",
     "!doc": "<p>This function will\nbe called at the end of every request, whether or not an error occurs.</p>\n"
    },
    "load": {
     "!type": "fn(response: ?, ioArgs: +dojo)",
     "!doc": "<p>This function will be\ncalled on a successful HTTP response code.</p>\n"
    }
   }
  },
  "dojo/main.back": {
   "addToHistory": {
    "!type": "fn(args: ?)",
    "!doc": "<p>adds a state object (args) to the history list.</p>\n"
   },
   "getHash": {
    "!type": "fn()"
   },
   "goBack": {
    "!type": "fn()",
    "!doc": "<p>private method. Do not call this directly.</p>\n"
   },
   "goForward": {
    "!type": "fn()",
    "!doc": "<p>private method. Do not call this directly.</p>\n"
   },
   "init": {
    "!type": "fn()",
    "!doc": "<p>Initializes the undo stack. This must be called from a <script>\nblock that lives inside the <code>&lt;body&gt;</code> tag to prevent bugs on IE.\n\n</p>\n<p>Only call this method before the page&#39;s DOM is finished loading. Otherwise\nit will not work. Be careful with xdomain loading or djConfig.debugAtAllCosts scenarios,\nin order for this method to work, dojo/back will need to be part of a build layer.</p>\n"
   },
   "setHash": {
    "!type": "fn(h: ?)"
   },
   "setInitialState": {
    "!type": "fn(args: ?)",
    "!doc": "<p>Sets the state object and back callback for the very first page\nthat is loaded.\n\n</p>\n<p>It is recommended that you call this method as part of an event\nlistener that is registered via dojo/ready.</p>\n"
   }
  },
  "dojo/main.cldr": {
   "monetary": {
    "!doc": "<p>TODOC</p>\n"
   },
   "supplemental": {
    "!doc": "<p>TODOC</p>\n"
   }
  },
  "dojo/main.i18n": {
   "cache": {},
   "dynamic": {},
   "unitTests": {},
   "getL10nName": {
    "!type": "fn(moduleName: ?, bundleName: ?, locale: ?) -> string"
   },
   "getLocalization": {
    "!type": "fn(moduleName: ?, bundleName: ?, locale: ?)"
   },
   "load": {
    "!type": "fn(id: ?, require: ?, load: ?)",
    "!doc": "<p>id is in one of the following formats\n\n</p>\n<ol>\n<li><p><path>/nls/<bundle>\n =&gt; load the bundle, localized to config.locale; load all bundles localized to\n config.extraLocale (if any); return the loaded bundle localized to config.locale.</p>\n</li>\n<li><p><path>/nls/<locale>/<bundle>\n =&gt; load then return the bundle localized to <locale></p>\n</li>\n<li><p><em>preload</em><path>/nls/<module>*<JSON array of available locales>\n =&gt; for config.locale and all config.extraLocale, load all bundles found\n in the best-matching bundle rollup. A value of 1 is returned, which\n is meaningless other than to say the plugin is executing the requested\n preloads</p>\n</li>\n</ol>\n<p>In cases 1 and 2, <path> is always normalized to an absolute module id upon entry; see\nnormalize. In case 3, it <path> is assumed to be absolute; this is arranged by the builder.\n\n</p>\n<p>To load a bundle means to insert the bundle into the plugin&#39;s cache and publish the bundle\nvalue to the loader. Given <path>, <bundle>, and a particular <locale>, the cache key\n\n</p>\n<pre><code>&lt;path&gt;/nls/&lt;bundle&gt;/&lt;locale&gt;</code></pre>\n<p>will hold the value. Similarly, then plugin will publish this value to the loader by\n\n</p>\n<pre><code>define(&quot;&lt;path&gt;/nls/&lt;bundle&gt;/&lt;locale&gt;&quot;, &lt;bundle-value&gt;);</code></pre>\n<p>Given this algorithm, other machinery can provide fast load paths be preplacing\nvalues in the plugin&#39;s cache, which is public. When a load is demanded the\ncache is inspected before starting any loading. Explicitly placing values in the plugin\ncache is an advanced/experimental feature that should not be needed; use at your own risk.\n\n</p>\n<p>For the normal AMD algorithm, the root bundle is loaded first, which instructs the\nplugin what additional localized bundles are required for a particular locale. These\nadditional locales are loaded and a mix of the root and each progressively-specific\nlocale is returned. For example:\n\n</p>\n<ol>\n<li><p>The client demands &quot;dojo/i18n!some/path/nls/someBundle</p>\n</li>\n<li><p>The loader demands load(some/path/nls/someBundle)</p>\n</li>\n<li><p>This plugin require&#39;s &quot;some/path/nls/someBundle&quot;, which is the root bundle.</p>\n</li>\n<li><p>Assuming config.locale is &quot;ab-cd-ef&quot; and the root bundle indicates that localizations\nare available for &quot;ab&quot; and &quot;ab-cd-ef&quot; (note the missing &quot;ab-cd&quot;, then the plugin\nrequires &quot;some/path/nls/ab/someBundle&quot; and &quot;some/path/nls/ab-cd-ef/someBundle&quot;</p>\n</li>\n<li><p>Upon receiving all required bundles, the plugin constructs the value of the bundle\nab-cd-ef as...</p>\n<pre><code> mixin(mixin(mixin({}, require(&quot;some/path/nls/someBundle&quot;),\n     require(&quot;some/path/nls/ab/someBundle&quot;)),\n     require(&quot;some/path/nls/ab-cd-ef/someBundle&quot;));</code></pre>\n</li>\n</ol>\n<p>This value is inserted into the cache and published to the loader at the\nkey/module-id some/path/nls/someBundle/ab-cd-ef.\n\n</p>\n<p>The special preload signature (case 3) instructs the plugin to stop servicing all normal requests\n(further preload requests will be serviced) until all ongoing preloading has completed.\n\n</p>\n<p>The preload signature instructs the plugin that a special rollup module is available that contains\none or more flattened, localized bundles. The JSON array of available locales indicates which locales\nare available. Here is an example:\n\n</p>\n<pre><code>*preload*some/path/nls/someModule*[&quot;root&quot;, &quot;ab&quot;, &quot;ab-cd-ef&quot;]</code></pre>\n<p>This indicates the following rollup modules are available:\n\n</p>\n<pre><code>some/path/nls/someModule_ROOT\nsome/path/nls/someModule_ab\nsome/path/nls/someModule_ab-cd-ef</code></pre>\n<p>Each of these modules is a normal AMD module that contains one or more flattened bundles in a hash.\nFor example, assume someModule contained the bundles some/bundle/path/someBundle and\nsome/bundle/path/someOtherBundle, then some/path/nls/someModule_ab would be expressed as follows:\n\n</p>\n<pre><code>define({\n    some/bundle/path/someBundle:&lt;value of someBundle, flattened with respect to locale ab&gt;,\n    some/bundle/path/someOtherBundle:&lt;value of someOtherBundle, flattened with respect to locale ab&gt;,\n});</code></pre>\n<p>E.g., given this design, preloading for locale==&quot;ab&quot; can execute the following algorithm:\n\n</p>\n<pre><code>require([&quot;some/path/nls/someModule_ab&quot;], function(rollup){\n    for(var p in rollup){\n        var id = p + &quot;/ab&quot;,\n        cache[id] = rollup[p];\n        define(id, rollup[p]);\n    }\n});</code></pre>\n<p>Similarly, if &quot;ab-cd&quot; is requested, the algorithm can determine that &quot;ab&quot; is the best available and\nload accordingly.\n\n</p>\n<p>The builder will write such rollups for every layer if a non-empty localeList  profile property is\nprovided. Further, the builder will include the following cache entry in the cache associated with\nany layer.\n\n</p>\n<pre><code>&quot;*now&quot;:function(r){r([&#39;dojo/i18n!*preload*&lt;path&gt;/nls/&lt;module&gt;*&lt;JSON array of available locales&gt;&#39;]);}</code></pre>\n<p>The *now special cache module instructs the loader to apply the provided function to context-require\nwith respect to the particular layer being defined. This causes the plugin to hold all normal service\nrequests until all preloading is complete.\n\n</p>\n<p>Notice that this algorithm is rarely better than the standard AMD load algorithm. Consider the normal case\nwhere the target locale has a single segment and a layer depends on a single bundle:\n\n</p>\n<p>Without Preloads:\n\n</p>\n<ol>\n<li>Layer loads root bundle.</li>\n<li>bundle is demanded; plugin loads single localized bundle.</li>\n</ol>\n<p>With Preloads:\n\n</p>\n<ol>\n<li>Layer causes preloading of target bundle.</li>\n<li>bundle is demanded; service is delayed until preloading complete; bundle is returned.</li>\n</ol>\n<p>In each case a single transaction is required to load the target bundle. In cases where multiple bundles\nare required and/or the locale has multiple segments, preloads still requires a single transaction whereas\nthe normal path requires an additional transaction for each additional bundle/locale-segment. However all\nof these additional transactions can be done concurrently. Owing to this analysis, the entire preloading\nalgorithm can be discard during a build by setting the has feature dojo-preload-i18n-Api to false.</p>\n"
   },
   "normalize": {
    "!type": "fn(id: ?, toAbsMid: ?)",
    "!doc": "<p>id may be relative.\npreload has form <code>*preload*&lt;path&gt;/nls/&lt;module&gt;*&lt;flattened locales&gt;</code> and\ntherefore never looks like a relative</p>\n"
   },
   "normalizeLocale": {
    "!type": "fn(locale: ?)"
   }
  },
  "dojo/main.colors": {
   "makeGrey": {
    "!type": "fn(g: number, a?: number)",
    "!doc": "<p>creates a greyscale color with an optional alpha</p>\n"
   }
  },
  "dojo/main.regexp": {
   "buildGroupRE": {
    "!type": "fn(arr: ?|[], re: fn(), nonCapture?: bool)",
    "!doc": "<p>Builds a regular expression that groups subexpressions</p>\n"
   },
   "escapeString": {
    "!type": "fn(str: string, except?: string)",
    "!doc": "<p>Adds escape sequences for special characters in regular expressions</p>\n"
   },
   "group": {
    "!type": "fn(expression: string, nonCapture?: bool) -> string",
    "!doc": "<p>adds group match to expression</p>\n"
   }
  },
  "dojo/main.string": {
   "escape": {
    "!type": "fn(str: string) -> string",
    "!doc": "<p>Efficiently escape a string for insertion into HTML (innerHTML or attributes), replacing &amp;, &lt;, &gt;, &quot;, &#39;, and / characters.</p>\n"
   },
   "pad": {
    "!type": "fn(text: string, size: +Integer, ch?: string, end?: bool) -> number",
    "!doc": "<p>Pad a string to guarantee that it is at least <code>size</code> length by\nfilling with the character <code>ch</code> at either the start or end of the\nstring. Pads at the start, by default.</p>\n"
   },
   "rep": {
    "!type": "fn(str: string, num: +Integer) -> string",
    "!doc": "<p>Efficiently replicate a string <code>n</code> times.</p>\n"
   },
   "substitute": {
    "!type": "fn(template: string, map: ?|[], transform?: fn(), thisObject?: ?)",
    "!doc": "<p>Performs parameterized substitutions on a string. Throws an\nexception if any parameter is unmatched.</p>\n"
   },
   "trim": {
    "!type": "fn(str: string) -> string",
    "!doc": "<p>Trims whitespace from both sides of the string</p>\n"
   }
  },
  "dojo/main.number": {
   "_numberPatternRE": {},
   "format": {
    "!type": "fn(value: number, options?: ?)",
    "!doc": "<p>Format a Number as a String, using locale-specific settings</p>\n"
   },
   "parse": {
    "!type": "fn(expression: string, options?: ?) -> number",
    "!doc": "<p>Convert a properly formatted string to a primitive Number, using\nlocale-specific settings.</p>\n"
   },
   "regexp": {
    "!type": "fn(options?: ?)",
    "!doc": "<p>Builds the regular needed to parse a number</p>\n"
   },
   "round": {
    "!type": "fn(value: number, places?: number, increment?: number) -> number",
    "!doc": "<p>Rounds to the nearest value with the given number of decimal places, away from zero</p>\n"
   }
  },
  "dojo/main.currency": {
   "format": {
    "!type": "fn(value: number, options?: +__FormatOptions)",
    "!doc": "<p>Format a Number as a currency, using locale-specific settings\n</p>\n"
   },
   "parse": {
    "!type": "fn(expression: string, options?: ?)"
   },
   "regexp": {
    "!type": "fn(options?: ?)"
   }
  },
  "dojo/main.data": {
   "api": {},
   "util": {},
   "ItemFileReadStore": {
    "!type": "fn()"
   },
   "ItemFileWriteStore": {
    "!type": "fn()"
   },
   "ObjectStore": {
    "!type": "fn()"
   }
  },
  "dojo/main.dnd": {
   "autoscroll": {
    "!doc": "<p>Used by dojo/dnd/Manager to scroll document or internal node when the user\ndrags near the edge of the viewport or a scrollable node</p>\n"
   },
   "move": {},
   "AutoSource": {
    "!type": "fn()"
   },
   "Avatar": {
    "!type": "fn()"
   },
   "Container": {
    "!type": "fn()"
   },
   "Manager": {
    "!type": "fn()"
   },
   "Moveable": {
    "!type": "fn()"
   },
   "Mover": {
    "!type": "fn()"
   },
   "Selector": {
    "!type": "fn()"
   },
   "Source": {
    "!type": "fn()"
   },
   "Target": {
    "!type": "fn()"
   },
   "TimedMoveable": {
    "!type": "fn()"
   }
  },
  "dojo/main.touch": {
   "cancel": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to &#39;touchcancel&#39;|&#39;mouseleave&#39; for the given node</p>\n"
   },
   "enter": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to mouse.enter or touch equivalent for the given node</p>\n"
   },
   "leave": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to mouse.leave or touch equivalent for the given node</p>\n"
   },
   "move": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener that fires when the mouse cursor or a finger is dragged over the given node.</p>\n"
   },
   "out": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to &#39;mouseout&#39; or touch equivalent for the given node</p>\n"
   },
   "over": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to &#39;mouseover&#39; or touch equivalent for the given node</p>\n"
   },
   "press": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to &#39;touchstart&#39;|&#39;mousedown&#39; for the given node</p>\n"
   },
   "release": {
    "!type": "fn(node: +Dom, listener: fn())",
    "!doc": "<p>Register a listener to releasing the mouse button while the cursor is over the given node\n(i.e. &quot;mouseup&quot;) or for removing the finger from the screen while touching the given node.</p>\n"
   }
  },
  "dojo/main.window": {
   "get": {
    "!type": "fn(doc: +Document)",
    "!doc": "<p>Get window object associated with document doc.</p>\n"
   },
   "getBox": {
    "!type": "fn(doc?: +Document)",
    "!doc": "<p>Returns the dimensions and scroll position of the viewable area of a browser window</p>\n"
   },
   "scrollIntoView": {
    "!type": "fn(node: +Node, pos?: ?)",
    "!doc": "<p>Scroll the passed node into view using minimal movement, if it is not already.</p>\n"
   }
  },
  "dojo/main.gears": {
   "available": {
    "!doc": "<p>True if client is using Google Gears</p>\n"
   }
  },
  "dojo/main.io": {
   "iframe": {},
   "script": {
    "!doc": "<p>TODOC</p>\n"
   }
  },
  "dojo/main.rpc": {
   "JsonpService": {
    "!type": "fn()"
   },
   "JsonService": {
    "!type": "fn()"
   },
   "RpcService": {
    "!type": "fn()"
   }
  },
  "dojo/main.store": {
   "util": {},
   "Cache": {
    "!type": "fn(masterStore: ?, cachingStore: ?, options: ?)"
   },
   "DataStore": {
    "!type": "fn()"
   },
   "JsonRest": {
    "!type": "fn()"
   },
   "Memory": {
    "!type": "fn()"
   },
   "Observable": {
    "!type": "fn(store: +dojo)",
    "!doc": "<p>The Observable store wrapper takes a store and sets an observe method on query()\nresults that can be used to monitor results for changes.\n</p>\n"
   }
  },
  "dojo/main.tests": {},
  "dojo/node": {
   "load": {
    "!type": "fn(id: string, contextRequire: fn(), load: fn())"
   },
   "normalize": {
    "!type": "fn(id: string, normalize: fn())",
    "!doc": "<p>Produces a normalized CommonJS module ID to be used by Node.js <code>require</code>. Relative IDs\nare resolved relative to the requesting module&#39;s location in the filesystem and will\nreturn an ID with path separators appropriate for the local filesystem</p>\n"
   }
  },
  "dojo/on/asyncEventListener": {
   "!type": "fn(listener: ?)"
  },
  "dojo/on/debounce": {
   "!type": "fn(selector: string, delay: +Interger)",
   "!doc": "<p>event parser for custom events</p>\n"
  },
  "dojo/on/throttle": {
   "!type": "fn(selector: string, delay: +Interger)",
   "!doc": "<p>event parser for custom events</p>\n"
  },
  "dojo/throttle": {
   "!type": "fn(cb: fn(), wait: +Integer)",
   "!doc": "<p>Create a function that will only execute once per <code>wait</code> periods.</p>\n"
  },
  "dojo/promise/first": {
   "!type": "fn(objectOrArray?: ?|[]) -> +dojo",
   "!doc": "<p>Takes multiple promises and returns a new promise that is fulfilled\nwhen the first of these promises is fulfilled.</p>\n"
  },
  "dojo/promise/instrumentation": {
   "!type": "fn(Deferred: ?)",
   "!doc": "<p>Initialize instrumentation for the Deferred class.</p>\n"
  },
  "dojo/promise/tracer": {
   "on": {
    "!type": "fn(type: string, listener: fn())",
    "!doc": "<p>Subscribe to traces.</p>\n"
   }
  },
  "dojo/request/default": {
   "getPlatformDefaultId": {
    "!type": "fn()"
   },
   "load": {
    "!type": "fn(id: ?, parentRequire: ?, loaded: ?, config: ?)"
   }
  },
  "dojo/request/node": {
   "!type": "fn(url: string, options?: +dojo) -> +dojo",
   "!doc": "<p>Sends a request using the included http or https interface from node.js\nwith the given URL and options.</p>\n",
   "del": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP DELETE request using XMLHttpRequest with the given URL and options.</p>\n"
   },
   "get": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP GET request using XMLHttpRequest with the given URL and options.</p>\n"
   },
   "post": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP POST request using XMLHttpRequest with the given URL and options.</p>\n"
   },
   "put": {
    "!type": "fn(url: string, options?: +dojo) -> +dojo",
    "!doc": "<p>Send an HTTP PUT request using XMLHttpRequest with the given URL and options.</p>\n"
   }
  },
  "dojo/request/node.__BaseOptions": {
   "!type": "fn()",
   "prototype": {
    "data": {
     "!doc": "<p>Data to transfer. This is ignored for GET and DELETE\nrequests.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>How to handle the response from the server.  Default is\n&#39;text&#39;.  Other values are &#39;json&#39;, &#39;javascript&#39;, and &#39;xml&#39;.</p>\n"
    },
    "headers": {
     "!doc": "<p>Headers to use for the request.</p>\n"
    },
    "password": {
     "!doc": "<p>Password to use during the request.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Whether to append a cache-busting parameter to the URL.</p>\n"
    },
    "query": {
     "!doc": "<p>Query parameters to append to the URL.</p>\n"
    },
    "timeout": {
     "!doc": "<p>Milliseconds to wait for the response.  If this time\npasses, the then the promise is rejected.</p>\n"
    },
    "user": {
     "!doc": "<p>Username to use during the request.</p>\n"
    }
   }
  },
  "dojo/request/node.__MethodOptions": {
   "!type": "fn()",
   "prototype": {
    "method": {
     "!doc": "<p>The HTTP method to use to make the request. Must be\nuppercase. Default is <code>&quot;GET&quot;</code>.</p>\n"
    }
   }
  },
  "dojo/request/node.__Options": {
   "!type": "fn()",
   "prototype": {
    "data": {
     "!doc": "<p>Data to transfer. This is ignored for GET and DELETE\nrequests.</p>\n"
    },
    "handleAs": {
     "!doc": "<p>How to handle the response from the server.  Default is\n&#39;text&#39;.  Other values are &#39;json&#39;, &#39;javascript&#39;, and &#39;xml&#39;.</p>\n"
    },
    "headers": {
     "!doc": "<p>Headers to use for the request.</p>\n"
    },
    "method": {
     "!doc": "<p>The HTTP method to use to make the request. Must be\nuppercase. Default is <code>&quot;GET&quot;</code>.</p>\n"
    },
    "password": {
     "!doc": "<p>Password to use during the request.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Whether to append a cache-busting parameter to the URL.</p>\n"
    },
    "query": {
     "!doc": "<p>Query parameters to append to the URL.</p>\n"
    },
    "timeout": {
     "!doc": "<p>Milliseconds to wait for the response.  If this time\npasses, the then the promise is rejected.</p>\n"
    },
    "user": {
     "!doc": "<p>Username to use during the request.</p>\n"
    }
   }
  },
  "dojo/request/notify": {
   "!type": "fn(type?: string, listener?: fn())",
   "!doc": "<p>Register a listener to be notified when an event\nin dojo/request happens.</p>\n",
   "emit": {
    "!type": "fn(type: ?, event: ?, cancel: ?)"
   }
  },
  "dojo/request/registry": {
   "!type": "fn(url: ?, options: ?)",
   "load": {
    "!type": "fn(id: ?, parentRequire: ?, loaded: ?, config: ?)"
   },
   "register": {
    "!type": "fn(url: ?, provider: ?, first: ?)"
   }
  },
  "dojo/require": {
   "dynamic": {},
   "load": {},
   "normalize": {
    "!type": "fn(id: ?)"
   }
  },
  "dojo/robot": {
   "_appletDead": {},
   "_loaded": {
    "!doc": "<p>Deferred that resolves when the _initRobot() has been called.\nNote to be confused with dojo/robotx.js, which defines initRobot() without an underscore</p>\n"
   },
   "_primePump": {},
   "_robotInitialized": {},
   "_robotLoaded": {},
   "_runsemaphore": {},
   "_spaceReceived": {},
   "_started": {
    "!doc": "<p>Deferred that resolves when startRobot() has signaled completing by typing on the keyboard,\nwhich in turn calls _run().</p>\n"
   },
   "doc": {},
   "mouseWheelSize": {},
   "window": {},
   "initRobot": {
    "!type": "fn(url: string)",
    "!doc": "<p>Opens the application at the specified URL for testing, redirecting dojo to point to the application\nenvironment instead of the test environment.</p>\n"
   },
   "keyDown": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer)",
    "!doc": "<p>Holds down a single key, like SHIFT or &#39;a&#39;.</p>\n"
   },
   "keyPress": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer, modifiers: ?, asynchronous: bool)",
    "!doc": "<p>Types a key combination, like SHIFT-TAB.</p>\n"
   },
   "keyUp": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer)",
    "!doc": "<p>Releases a single key, like SHIFT or &#39;a&#39;.</p>\n"
   },
   "killRobot": {
    "!type": "fn()"
   },
   "mouseClick": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Convenience function to do a press/release.\nSee robot.mousePress for more info.</p>\n"
   },
   "mouseMove": {
    "!type": "fn(x: number, y: number, delay?: +Integer, duration?: +Integer, absolute: bool)",
    "!doc": "<p>Moves the mouse to the specified x,y offset relative to the viewport.</p>\n"
   },
   "mouseMoveAt": {
    "!type": "fn(node: string|+Node|fn(), delay: ?, duration: ?, offsetX: ?, offsetY: ?)",
    "!doc": "<p>Moves the mouse over the specified node at the specified relative x,y offset.</p>\n"
   },
   "mouseMoveTo": {
    "!type": "fn(point: ?, delay?: +Integer, duration?: +Integer, absolute: bool)",
    "!doc": "<p>Move the mouse from the current position to the specified point.\nDelays reading contents point until queued command starts running.\nSee mouseMove() for details.</p>\n"
   },
   "mousePress": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Presses mouse buttons.</p>\n"
   },
   "mouseRelease": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Releases mouse buttons.</p>\n"
   },
   "mouseWheel": {
    "!type": "fn(wheelAmt: number, delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Spins the mouse wheel.</p>\n"
   },
   "scrollIntoView": {
    "!type": "fn(node: string|+Node|fn(), delay: ?)",
    "!doc": "<p>Scroll the passed node into view, if it is not.</p>\n"
   },
   "sequence": {
    "!type": "fn(f: fn(), delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Defer an action by adding it to the robot&#39;s incrementally delayed queue of actions to execute.</p>\n"
   },
   "setClipboard": {
    "!type": "fn(data: string, format?: string)",
    "!doc": "<p>Set clipboard content.</p>\n"
   },
   "startRobot": {
    "!type": "fn()"
   },
   "typeKeys": {
    "!type": "fn(chars: string|number, delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Types a string of characters in order, or types a dojo.keys.* constant.</p>\n"
   },
   "waitForPageToLoad": {
    "!type": "fn(submitActions: fn())",
    "!doc": "<p>Notifies DOH that the doh.robot is about to make a page change in the application it is driving,\nreturning a doh.Deferred object the user should return in their runTest function as part of a DOH test.</p>\n"
   }
  },
  "dojo/robot._runsemaphore": {
   "lock": {},
   "unlock": {
    "!type": "fn()"
   }
  },
  "doh/_browserRunner": {
   "_currentGroup": {},
   "_currentTest": {},
   "_errorCount": {},
   "_failureCount": {},
   "_groupCount": {},
   "_groups": {},
   "_init": {},
   "_initRobotCalled": {},
   "_line": {},
   "_paused": {},
   "_registerUrl": {},
   "_setupGroupForRun": {},
   "_testCount": {},
   "_testTypes": {},
   "debug": {},
   "error": {},
   "isBrowser": {},
   "registerUrl": {},
   "robot": {},
   "assertEqual": {
    "!type": "fn(expected: ?, actual: ?, hint?: string, doNotThrow: ?)",
    "!doc": "<p>are the passed expected and actual objects/values deeply\nequivalent?</p>\n"
   },
   "assertError": {
    "!type": "fn(expectedError: ?, scope: ?, functionName: string, args: [], hint?: string)",
    "!doc": "<p>Test for a certain error to be thrown by the given function.</p>\n"
   },
   "assertFalse": {
    "!type": "fn(condition: ?, hint?: string)",
    "!doc": "<p>is the passed item &quot;falsey&quot;?</p>\n"
   },
   "assertNotEqual": {
    "!type": "fn(notExpected: ?, actual: ?, hint?: string)",
    "!doc": "<p>are the passed notexpected and actual objects/values deeply\nnot equivalent?</p>\n"
   },
   "assertTrue": {
    "!type": "fn(condition: ?, hint?: string)",
    "!doc": "<p>is the passed item &quot;truthy&quot;?</p>\n"
   },
   "Deferred": {
    "!type": "fn(canceller: ?)"
   },
   "e": {
    "!type": "fn(expectedError: ?, scope: ?, functionName: string, args: [], hint?: string)",
    "!doc": "<p>Test for a certain error to be thrown by the given function.</p>\n"
   },
   "f": {
    "!type": "fn(condition: ?, hint?: string)",
    "!doc": "<p>is the passed item &quot;falsey&quot;?</p>\n"
   },
   "is": {
    "!type": "fn(expected: ?, actual: ?, hint?: string, doNotThrow: ?)",
    "!doc": "<p>are the passed expected and actual objects/values deeply\nequivalent?</p>\n"
   },
   "isNot": {
    "!type": "fn(notExpected: ?, actual: ?, hint?: string)",
    "!doc": "<p>are the passed notexpected and actual objects/values deeply\nnot equivalent?</p>\n"
   },
   "pause": {
    "!type": "fn()",
    "!doc": "<p>halt test run. Can be resumed.</p>\n"
   },
   "register": {
    "!type": "fn(a1: ?, a2: ?, a3: ?, a4: ?, a5: ?)"
   },
   "registerDocTests": {
    "!type": "fn(module: ?)",
    "!doc": "<p>Deprecated.    Won&#39;t work unless you manually load dojox.testing.DocTest, and likely not even then.\nGets all the doctests from the given module and register each of them as a single test case here.</p>\n"
   },
   "registerGroup": {
    "!type": "fn(group: string, tests: []|fn()|?, setUp: fn(), tearDown: fn(), type: string)",
    "!doc": "<p>Deprecated.  Use doh.register(group/type, tests, setUp, tearDown) instead</p>\n"
   },
   "registerTest": {
    "!type": "fn(group: string, test: []|fn()|?, type: string)",
    "!doc": "<p>Deprecated.  Use doh.register(group/type, test) instead</p>\n"
   },
   "registerTestNs": {
    "!type": "fn(group: string, ns: ?)",
    "!doc": "<p>Deprecated.  Use doh.register(group, ns) instead</p>\n"
   },
   "registerTests": {
    "!type": "fn(group: string, testArr: [], type: string)",
    "!doc": "<p>Deprecated.  Use doh.register(group/type, testArr) instead</p>\n"
   },
   "registerTestType": {
    "!type": "fn(name: string, initProc: fn())",
    "!doc": "<p>Adds a test type and associates a function used to initialize each test of the given type</p>\n"
   },
   "run": {
    "!type": "fn()"
   },
   "runGroup": {
    "!type": "fn(groupName: string, idx: +Integer)",
    "!doc": "<p>runs the specified test group</p>\n"
   },
   "runOnLoad": {
    "!type": "fn()"
   },
   "showLogPage": {
    "!type": "fn()"
   },
   "showPerfTestsPage": {
    "!type": "fn()"
   },
   "showTestPage": {
    "!type": "fn()"
   },
   "t": {
    "!type": "fn(condition: ?, hint?: string)",
    "!doc": "<p>is the passed item &quot;truthy&quot;?</p>\n"
   },
   "togglePaused": {
    "!type": "fn()"
   },
   "toggleRunAll": {
    "!type": "fn()"
   }
  },
  "doh/_browserRunner._groups": {},
  "doh/_browserRunner._testTypes": {
   "perf": {
    "!type": "fn(group: ?, tObj: ?, type: ?)"
   }
  },
  "doh/_browserRunner.robot": {
   "_appletDead": {},
   "_loaded": {
    "!doc": "<p>Deferred that resolves when the _initRobot() has been called.\nNote to be confused with dojo/robotx.js, which defines initRobot() without an underscore</p>\n"
   },
   "_primePump": {},
   "_robotInitialized": {},
   "_robotLoaded": {},
   "_runsemaphore": {},
   "_spaceReceived": {},
   "_started": {
    "!doc": "<p>Deferred that resolves when startRobot() has signaled completing by typing on the keyboard,\nwhich in turn calls _run().</p>\n"
   },
   "doc": {},
   "mouseWheelSize": {},
   "window": {},
   "initRobot": {
    "!type": "fn(url: string)",
    "!doc": "<p>Opens the application at the specified URL for testing, redirecting dojo to point to the application\nenvironment instead of the test environment.</p>\n"
   },
   "keyDown": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer)",
    "!doc": "<p>Holds down a single key, like SHIFT or &#39;a&#39;.</p>\n"
   },
   "keyPress": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer, modifiers: ?, asynchronous: bool)",
    "!doc": "<p>Types a key combination, like SHIFT-TAB.</p>\n"
   },
   "keyUp": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer)",
    "!doc": "<p>Releases a single key, like SHIFT or &#39;a&#39;.</p>\n"
   },
   "killRobot": {
    "!type": "fn()"
   },
   "mouseClick": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Convenience function to do a press/release.\nSee robot.mousePress for more info.</p>\n"
   },
   "mouseMove": {
    "!type": "fn(x: number, y: number, delay?: +Integer, duration?: +Integer, absolute: bool)",
    "!doc": "<p>Moves the mouse to the specified x,y offset relative to the viewport.</p>\n"
   },
   "mouseMoveAt": {
    "!type": "fn(node: string|+Node|fn(), delay: ?, duration: ?, offsetX: ?, offsetY: ?)",
    "!doc": "<p>Moves the mouse over the specified node at the specified relative x,y offset.</p>\n"
   },
   "mouseMoveTo": {
    "!type": "fn(point: ?, delay?: +Integer, duration?: +Integer, absolute: bool)",
    "!doc": "<p>Move the mouse from the current position to the specified point.\nDelays reading contents point until queued command starts running.\nSee mouseMove() for details.</p>\n"
   },
   "mousePress": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Presses mouse buttons.</p>\n"
   },
   "mouseRelease": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Releases mouse buttons.</p>\n"
   },
   "mouseWheel": {
    "!type": "fn(wheelAmt: number, delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Spins the mouse wheel.</p>\n"
   },
   "scrollIntoView": {
    "!type": "fn(node: string|+Node|fn(), delay: ?)",
    "!doc": "<p>Scroll the passed node into view, if it is not.</p>\n"
   },
   "sequence": {
    "!type": "fn(f: fn(), delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Defer an action by adding it to the robot&#39;s incrementally delayed queue of actions to execute.</p>\n"
   },
   "setClipboard": {
    "!type": "fn(data: string, format?: string)",
    "!doc": "<p>Set clipboard content.</p>\n"
   },
   "startRobot": {
    "!type": "fn()"
   },
   "typeKeys": {
    "!type": "fn(chars: string|number, delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Types a string of characters in order, or types a dojo.keys.* constant.</p>\n"
   },
   "waitForPageToLoad": {
    "!type": "fn(submitActions: fn())",
    "!doc": "<p>Notifies DOH that the doh.robot is about to make a page change in the application it is driving,\nreturning a doh.Deferred object the user should return in their runTest function as part of a DOH test.</p>\n"
   }
  },
  "doh/runner": {
   "_currentGroup": {},
   "_currentTest": {},
   "_errorCount": {},
   "_failureCount": {},
   "_groupCount": {},
   "_groups": {},
   "_init": {},
   "_initRobotCalled": {},
   "_line": {},
   "_paused": {},
   "_registerUrl": {},
   "_setupGroupForRun": {},
   "_testCount": {},
   "_testTypes": {},
   "debug": {},
   "error": {},
   "isBrowser": {},
   "registerUrl": {},
   "robot": {},
   "assertEqual": {
    "!type": "fn(expected: ?, actual: ?, hint?: string, doNotThrow: ?)",
    "!doc": "<p>are the passed expected and actual objects/values deeply\nequivalent?</p>\n"
   },
   "assertError": {
    "!type": "fn(expectedError: ?, scope: ?, functionName: string, args: [], hint?: string)",
    "!doc": "<p>Test for a certain error to be thrown by the given function.</p>\n"
   },
   "assertFalse": {
    "!type": "fn(condition: ?, hint?: string)",
    "!doc": "<p>is the passed item &quot;falsey&quot;?</p>\n"
   },
   "assertNotEqual": {
    "!type": "fn(notExpected: ?, actual: ?, hint?: string)",
    "!doc": "<p>are the passed notexpected and actual objects/values deeply\nnot equivalent?</p>\n"
   },
   "assertTrue": {
    "!type": "fn(condition: ?, hint?: string)",
    "!doc": "<p>is the passed item &quot;truthy&quot;?</p>\n"
   },
   "Deferred": {
    "!type": "fn(canceller: ?)"
   },
   "e": {
    "!type": "fn(expectedError: ?, scope: ?, functionName: string, args: [], hint?: string)",
    "!doc": "<p>Test for a certain error to be thrown by the given function.</p>\n"
   },
   "f": {
    "!type": "fn(condition: ?, hint?: string)",
    "!doc": "<p>is the passed item &quot;falsey&quot;?</p>\n"
   },
   "is": {
    "!type": "fn(expected: ?, actual: ?, hint?: string, doNotThrow: ?)",
    "!doc": "<p>are the passed expected and actual objects/values deeply\nequivalent?</p>\n"
   },
   "isNot": {
    "!type": "fn(notExpected: ?, actual: ?, hint?: string)",
    "!doc": "<p>are the passed notexpected and actual objects/values deeply\nnot equivalent?</p>\n"
   },
   "pause": {
    "!type": "fn()",
    "!doc": "<p>halt test run. Can be resumed.</p>\n"
   },
   "register": {
    "!type": "fn(a1: ?, a2: ?, a3: ?, a4: ?, a5: ?)"
   },
   "registerDocTests": {
    "!type": "fn(module: ?)",
    "!doc": "<p>Deprecated.    Won&#39;t work unless you manually load dojox.testing.DocTest, and likely not even then.\nGets all the doctests from the given module and register each of them as a single test case here.</p>\n"
   },
   "registerGroup": {
    "!type": "fn(group: string, tests: []|fn()|?, setUp: fn(), tearDown: fn(), type: string)",
    "!doc": "<p>Deprecated.  Use doh.register(group/type, tests, setUp, tearDown) instead</p>\n"
   },
   "registerTest": {
    "!type": "fn(group: string, test: []|fn()|?, type: string)",
    "!doc": "<p>Deprecated.  Use doh.register(group/type, test) instead</p>\n"
   },
   "registerTestNs": {
    "!type": "fn(group: string, ns: ?)",
    "!doc": "<p>Deprecated.  Use doh.register(group, ns) instead</p>\n"
   },
   "registerTests": {
    "!type": "fn(group: string, testArr: [], type: string)",
    "!doc": "<p>Deprecated.  Use doh.register(group/type, testArr) instead</p>\n"
   },
   "registerTestType": {
    "!type": "fn(name: string, initProc: fn())",
    "!doc": "<p>Adds a test type and associates a function used to initialize each test of the given type</p>\n"
   },
   "run": {
    "!type": "fn()"
   },
   "runGroup": {
    "!type": "fn(groupName: string, idx: +Integer)",
    "!doc": "<p>runs the specified test group</p>\n"
   },
   "runOnLoad": {
    "!type": "fn()"
   },
   "showLogPage": {
    "!type": "fn()"
   },
   "showPerfTestsPage": {
    "!type": "fn()"
   },
   "showTestPage": {
    "!type": "fn()"
   },
   "t": {
    "!type": "fn(condition: ?, hint?: string)",
    "!doc": "<p>is the passed item &quot;truthy&quot;?</p>\n"
   },
   "togglePaused": {
    "!type": "fn()"
   },
   "toggleRunAll": {
    "!type": "fn()"
   }
  },
  "doh/runner._groups": {},
  "doh/runner._testTypes": {
   "perf": {
    "!type": "fn(group: ?, tObj: ?, type: ?)"
   }
  },
  "doh/runner.robot": {
   "_appletDead": {},
   "_loaded": {
    "!doc": "<p>Deferred that resolves when the _initRobot() has been called.\nNote to be confused with dojo/robotx.js, which defines initRobot() without an underscore</p>\n"
   },
   "_primePump": {},
   "_robotInitialized": {},
   "_robotLoaded": {},
   "_runsemaphore": {},
   "_spaceReceived": {},
   "_started": {
    "!doc": "<p>Deferred that resolves when startRobot() has signaled completing by typing on the keyboard,\nwhich in turn calls _run().</p>\n"
   },
   "doc": {},
   "mouseWheelSize": {},
   "window": {},
   "initRobot": {
    "!type": "fn(url: string)",
    "!doc": "<p>Opens the application at the specified URL for testing, redirecting dojo to point to the application\nenvironment instead of the test environment.</p>\n"
   },
   "keyDown": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer)",
    "!doc": "<p>Holds down a single key, like SHIFT or &#39;a&#39;.</p>\n"
   },
   "keyPress": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer, modifiers: ?, asynchronous: bool)",
    "!doc": "<p>Types a key combination, like SHIFT-TAB.</p>\n"
   },
   "keyUp": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer)",
    "!doc": "<p>Releases a single key, like SHIFT or &#39;a&#39;.</p>\n"
   },
   "killRobot": {
    "!type": "fn()"
   },
   "mouseClick": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Convenience function to do a press/release.\nSee robot.mousePress for more info.</p>\n"
   },
   "mouseMove": {
    "!type": "fn(x: number, y: number, delay?: +Integer, duration?: +Integer, absolute: bool)",
    "!doc": "<p>Moves the mouse to the specified x,y offset relative to the viewport.</p>\n"
   },
   "mouseMoveAt": {
    "!type": "fn(node: string|+Node|fn(), delay: ?, duration: ?, offsetX: ?, offsetY: ?)",
    "!doc": "<p>Moves the mouse over the specified node at the specified relative x,y offset.</p>\n"
   },
   "mouseMoveTo": {
    "!type": "fn(point: ?, delay?: +Integer, duration?: +Integer, absolute: bool)",
    "!doc": "<p>Move the mouse from the current position to the specified point.\nDelays reading contents point until queued command starts running.\nSee mouseMove() for details.</p>\n"
   },
   "mousePress": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Presses mouse buttons.</p>\n"
   },
   "mouseRelease": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Releases mouse buttons.</p>\n"
   },
   "mouseWheel": {
    "!type": "fn(wheelAmt: number, delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Spins the mouse wheel.</p>\n"
   },
   "scrollIntoView": {
    "!type": "fn(node: string|+Node|fn(), delay: ?)",
    "!doc": "<p>Scroll the passed node into view, if it is not.</p>\n"
   },
   "sequence": {
    "!type": "fn(f: fn(), delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Defer an action by adding it to the robot&#39;s incrementally delayed queue of actions to execute.</p>\n"
   },
   "setClipboard": {
    "!type": "fn(data: string, format?: string)",
    "!doc": "<p>Set clipboard content.</p>\n"
   },
   "startRobot": {
    "!type": "fn()"
   },
   "typeKeys": {
    "!type": "fn(chars: string|number, delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Types a string of characters in order, or types a dojo.keys.* constant.</p>\n"
   },
   "waitForPageToLoad": {
    "!type": "fn(submitActions: fn())",
    "!doc": "<p>Notifies DOH that the doh.robot is about to make a page change in the application it is driving,\nreturning a doh.Deferred object the user should return in their runTest function as part of a DOH test.</p>\n"
   }
  },
  "doh/robot": {
   "_appletDead": {},
   "_loaded": {
    "!doc": "<p>Deferred that resolves when the _initRobot() has been called.\nNote to be confused with dojo/robotx.js, which defines initRobot() without an underscore</p>\n"
   },
   "_primePump": {},
   "_robotInitialized": {},
   "_robotLoaded": {},
   "_runsemaphore": {},
   "_spaceReceived": {},
   "_started": {
    "!doc": "<p>Deferred that resolves when startRobot() has signaled completing by typing on the keyboard,\nwhich in turn calls _run().</p>\n"
   },
   "doc": {},
   "mouseWheelSize": {},
   "window": {},
   "initRobot": {
    "!type": "fn(url: string)",
    "!doc": "<p>Opens the application at the specified URL for testing, redirecting dojo to point to the application\nenvironment instead of the test environment.</p>\n"
   },
   "keyDown": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer)",
    "!doc": "<p>Holds down a single key, like SHIFT or &#39;a&#39;.</p>\n"
   },
   "keyPress": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer, modifiers: ?, asynchronous: bool)",
    "!doc": "<p>Types a key combination, like SHIFT-TAB.</p>\n"
   },
   "keyUp": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer)",
    "!doc": "<p>Releases a single key, like SHIFT or &#39;a&#39;.</p>\n"
   },
   "killRobot": {
    "!type": "fn()"
   },
   "mouseClick": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Convenience function to do a press/release.\nSee robot.mousePress for more info.</p>\n"
   },
   "mouseMove": {
    "!type": "fn(x: number, y: number, delay?: +Integer, duration?: +Integer, absolute: bool)",
    "!doc": "<p>Moves the mouse to the specified x,y offset relative to the viewport.</p>\n"
   },
   "mouseMoveAt": {
    "!type": "fn(node: string|+Node|fn(), delay: ?, duration: ?, offsetX: ?, offsetY: ?)",
    "!doc": "<p>Moves the mouse over the specified node at the specified relative x,y offset.</p>\n"
   },
   "mouseMoveTo": {
    "!type": "fn(point: ?, delay?: +Integer, duration?: +Integer, absolute: bool)",
    "!doc": "<p>Move the mouse from the current position to the specified point.\nDelays reading contents point until queued command starts running.\nSee mouseMove() for details.</p>\n"
   },
   "mousePress": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Presses mouse buttons.</p>\n"
   },
   "mouseRelease": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Releases mouse buttons.</p>\n"
   },
   "mouseWheel": {
    "!type": "fn(wheelAmt: number, delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Spins the mouse wheel.</p>\n"
   },
   "scrollIntoView": {
    "!type": "fn(node: string|+Node|fn(), delay: ?)",
    "!doc": "<p>Scroll the passed node into view, if it is not.</p>\n"
   },
   "sequence": {
    "!type": "fn(f: fn(), delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Defer an action by adding it to the robot&#39;s incrementally delayed queue of actions to execute.</p>\n"
   },
   "setClipboard": {
    "!type": "fn(data: string, format?: string)",
    "!doc": "<p>Set clipboard content.</p>\n"
   },
   "startRobot": {
    "!type": "fn()"
   },
   "typeKeys": {
    "!type": "fn(chars: string|number, delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Types a string of characters in order, or types a dojo.keys.* constant.</p>\n"
   },
   "waitForPageToLoad": {
    "!type": "fn(submitActions: fn())",
    "!doc": "<p>Notifies DOH that the doh.robot is about to make a page change in the application it is driving,\nreturning a doh.Deferred object the user should return in their runTest function as part of a DOH test.</p>\n"
   }
  },
  "doh/robot._runsemaphore": {
   "lock": {},
   "unlock": {
    "!type": "fn()"
   }
  },
  "dojo/robotx": {
   "_appletDead": {},
   "_loaded": {
    "!doc": "<p>Deferred that resolves when the _initRobot() has been called.\nNote to be confused with dojo/robotx.js, which defines initRobot() without an underscore</p>\n"
   },
   "_primePump": {},
   "_robotInitialized": {},
   "_robotLoaded": {},
   "_runsemaphore": {},
   "_spaceReceived": {},
   "_started": {
    "!doc": "<p>Deferred that resolves when startRobot() has signaled completing by typing on the keyboard,\nwhich in turn calls _run().</p>\n"
   },
   "doc": {},
   "mouseWheelSize": {},
   "window": {},
   "initRobot": {
    "!type": "fn(url: string)",
    "!doc": "<p>Opens the application at the specified URL for testing, redirecting dojo to point to the application\nenvironment instead of the test environment.</p>\n"
   },
   "keyDown": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer)",
    "!doc": "<p>Holds down a single key, like SHIFT or &#39;a&#39;.</p>\n"
   },
   "keyPress": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer, modifiers: ?, asynchronous: bool)",
    "!doc": "<p>Types a key combination, like SHIFT-TAB.</p>\n"
   },
   "keyUp": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer)",
    "!doc": "<p>Releases a single key, like SHIFT or &#39;a&#39;.</p>\n"
   },
   "killRobot": {
    "!type": "fn()"
   },
   "mouseClick": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Convenience function to do a press/release.\nSee robot.mousePress for more info.</p>\n"
   },
   "mouseMove": {
    "!type": "fn(x: number, y: number, delay?: +Integer, duration?: +Integer, absolute: bool)",
    "!doc": "<p>Moves the mouse to the specified x,y offset relative to the viewport.</p>\n"
   },
   "mouseMoveAt": {
    "!type": "fn(node: string|+Node|fn(), delay: ?, duration: ?, offsetX: ?, offsetY: ?)",
    "!doc": "<p>Moves the mouse over the specified node at the specified relative x,y offset.</p>\n"
   },
   "mouseMoveTo": {
    "!type": "fn(point: ?, delay?: +Integer, duration?: +Integer, absolute: bool)",
    "!doc": "<p>Move the mouse from the current position to the specified point.\nDelays reading contents point until queued command starts running.\nSee mouseMove() for details.</p>\n"
   },
   "mousePress": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Presses mouse buttons.</p>\n"
   },
   "mouseRelease": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Releases mouse buttons.</p>\n"
   },
   "mouseWheel": {
    "!type": "fn(wheelAmt: number, delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Spins the mouse wheel.</p>\n"
   },
   "scrollIntoView": {
    "!type": "fn(node: string|+Node|fn(), delay: ?)",
    "!doc": "<p>Scroll the passed node into view, if it is not.</p>\n"
   },
   "sequence": {
    "!type": "fn(f: fn(), delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Defer an action by adding it to the robot&#39;s incrementally delayed queue of actions to execute.</p>\n"
   },
   "setClipboard": {
    "!type": "fn(data: string, format?: string)",
    "!doc": "<p>Set clipboard content.</p>\n"
   },
   "startRobot": {
    "!type": "fn()"
   },
   "typeKeys": {
    "!type": "fn(chars: string|number, delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Types a string of characters in order, or types a dojo.keys.* constant.</p>\n"
   },
   "waitForPageToLoad": {
    "!type": "fn(submitActions: fn())",
    "!doc": "<p>Notifies DOH that the doh.robot is about to make a page change in the application it is driving,\nreturning a doh.Deferred object the user should return in their runTest function as part of a DOH test.</p>\n"
   }
  },
  "dojo/robotx._runsemaphore": {
   "lock": {},
   "unlock": {
    "!type": "fn()"
   }
  },
  "doh/main": {
   "_currentGroup": {},
   "_currentTest": {},
   "_errorCount": {},
   "_failureCount": {},
   "_groupCount": {},
   "_groups": {},
   "_init": {},
   "_initRobotCalled": {},
   "_line": {},
   "_paused": {},
   "_registerUrl": {},
   "_setupGroupForRun": {},
   "_testCount": {},
   "_testTypes": {},
   "debug": {},
   "error": {},
   "isBrowser": {},
   "registerUrl": {},
   "robot": {},
   "assertEqual": {
    "!type": "fn(expected: ?, actual: ?, hint?: string, doNotThrow: ?)",
    "!doc": "<p>are the passed expected and actual objects/values deeply\nequivalent?</p>\n"
   },
   "assertError": {
    "!type": "fn(expectedError: ?, scope: ?, functionName: string, args: [], hint?: string)",
    "!doc": "<p>Test for a certain error to be thrown by the given function.</p>\n"
   },
   "assertFalse": {
    "!type": "fn(condition: ?, hint?: string)",
    "!doc": "<p>is the passed item &quot;falsey&quot;?</p>\n"
   },
   "assertNotEqual": {
    "!type": "fn(notExpected: ?, actual: ?, hint?: string)",
    "!doc": "<p>are the passed notexpected and actual objects/values deeply\nnot equivalent?</p>\n"
   },
   "assertTrue": {
    "!type": "fn(condition: ?, hint?: string)",
    "!doc": "<p>is the passed item &quot;truthy&quot;?</p>\n"
   },
   "Deferred": {
    "!type": "fn(canceller: ?)"
   },
   "e": {
    "!type": "fn(expectedError: ?, scope: ?, functionName: string, args: [], hint?: string)",
    "!doc": "<p>Test for a certain error to be thrown by the given function.</p>\n"
   },
   "f": {
    "!type": "fn(condition: ?, hint?: string)",
    "!doc": "<p>is the passed item &quot;falsey&quot;?</p>\n"
   },
   "is": {
    "!type": "fn(expected: ?, actual: ?, hint?: string, doNotThrow: ?)",
    "!doc": "<p>are the passed expected and actual objects/values deeply\nequivalent?</p>\n"
   },
   "isNot": {
    "!type": "fn(notExpected: ?, actual: ?, hint?: string)",
    "!doc": "<p>are the passed notexpected and actual objects/values deeply\nnot equivalent?</p>\n"
   },
   "pause": {
    "!type": "fn()",
    "!doc": "<p>halt test run. Can be resumed.</p>\n"
   },
   "register": {
    "!type": "fn(a1: ?, a2: ?, a3: ?, a4: ?, a5: ?)"
   },
   "registerDocTests": {
    "!type": "fn(module: ?)",
    "!doc": "<p>Deprecated.    Won&#39;t work unless you manually load dojox.testing.DocTest, and likely not even then.\nGets all the doctests from the given module and register each of them as a single test case here.</p>\n"
   },
   "registerGroup": {
    "!type": "fn(group: string, tests: []|fn()|?, setUp: fn(), tearDown: fn(), type: string)",
    "!doc": "<p>Deprecated.  Use doh.register(group/type, tests, setUp, tearDown) instead</p>\n"
   },
   "registerTest": {
    "!type": "fn(group: string, test: []|fn()|?, type: string)",
    "!doc": "<p>Deprecated.  Use doh.register(group/type, test) instead</p>\n"
   },
   "registerTestNs": {
    "!type": "fn(group: string, ns: ?)",
    "!doc": "<p>Deprecated.  Use doh.register(group, ns) instead</p>\n"
   },
   "registerTests": {
    "!type": "fn(group: string, testArr: [], type: string)",
    "!doc": "<p>Deprecated.  Use doh.register(group/type, testArr) instead</p>\n"
   },
   "registerTestType": {
    "!type": "fn(name: string, initProc: fn())",
    "!doc": "<p>Adds a test type and associates a function used to initialize each test of the given type</p>\n"
   },
   "run": {
    "!type": "fn()"
   },
   "runGroup": {
    "!type": "fn(groupName: string, idx: +Integer)",
    "!doc": "<p>runs the specified test group</p>\n"
   },
   "runOnLoad": {
    "!type": "fn()"
   },
   "showLogPage": {
    "!type": "fn()"
   },
   "showPerfTestsPage": {
    "!type": "fn()"
   },
   "showTestPage": {
    "!type": "fn()"
   },
   "t": {
    "!type": "fn(condition: ?, hint?: string)",
    "!doc": "<p>is the passed item &quot;truthy&quot;?</p>\n"
   },
   "togglePaused": {
    "!type": "fn()"
   },
   "toggleRunAll": {
    "!type": "fn()"
   }
  },
  "doh/main._groups": {},
  "doh/main._testTypes": {
   "perf": {
    "!type": "fn(group: ?, tObj: ?, type: ?)"
   }
  },
  "doh/main.robot": {
   "_appletDead": {},
   "_loaded": {
    "!doc": "<p>Deferred that resolves when the _initRobot() has been called.\nNote to be confused with dojo/robotx.js, which defines initRobot() without an underscore</p>\n"
   },
   "_primePump": {},
   "_robotInitialized": {},
   "_robotLoaded": {},
   "_runsemaphore": {},
   "_spaceReceived": {},
   "_started": {
    "!doc": "<p>Deferred that resolves when startRobot() has signaled completing by typing on the keyboard,\nwhich in turn calls _run().</p>\n"
   },
   "doc": {},
   "mouseWheelSize": {},
   "window": {},
   "initRobot": {
    "!type": "fn(url: string)",
    "!doc": "<p>Opens the application at the specified URL for testing, redirecting dojo to point to the application\nenvironment instead of the test environment.</p>\n"
   },
   "keyDown": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer)",
    "!doc": "<p>Holds down a single key, like SHIFT or &#39;a&#39;.</p>\n"
   },
   "keyPress": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer, modifiers: ?, asynchronous: bool)",
    "!doc": "<p>Types a key combination, like SHIFT-TAB.</p>\n"
   },
   "keyUp": {
    "!type": "fn(charOrCode: +Integer, delay?: +Integer)",
    "!doc": "<p>Releases a single key, like SHIFT or &#39;a&#39;.</p>\n"
   },
   "killRobot": {
    "!type": "fn()"
   },
   "mouseClick": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Convenience function to do a press/release.\nSee robot.mousePress for more info.</p>\n"
   },
   "mouseMove": {
    "!type": "fn(x: number, y: number, delay?: +Integer, duration?: +Integer, absolute: bool)",
    "!doc": "<p>Moves the mouse to the specified x,y offset relative to the viewport.</p>\n"
   },
   "mouseMoveAt": {
    "!type": "fn(node: string|+Node|fn(), delay: ?, duration: ?, offsetX: ?, offsetY: ?)",
    "!doc": "<p>Moves the mouse over the specified node at the specified relative x,y offset.</p>\n"
   },
   "mouseMoveTo": {
    "!type": "fn(point: ?, delay?: +Integer, duration?: +Integer, absolute: bool)",
    "!doc": "<p>Move the mouse from the current position to the specified point.\nDelays reading contents point until queued command starts running.\nSee mouseMove() for details.</p>\n"
   },
   "mousePress": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Presses mouse buttons.</p>\n"
   },
   "mouseRelease": {
    "!type": "fn(buttons: ?, delay?: +Integer)",
    "!doc": "<p>Releases mouse buttons.</p>\n"
   },
   "mouseWheel": {
    "!type": "fn(wheelAmt: number, delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Spins the mouse wheel.</p>\n"
   },
   "scrollIntoView": {
    "!type": "fn(node: string|+Node|fn(), delay: ?)",
    "!doc": "<p>Scroll the passed node into view, if it is not.</p>\n"
   },
   "sequence": {
    "!type": "fn(f: fn(), delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Defer an action by adding it to the robot&#39;s incrementally delayed queue of actions to execute.</p>\n"
   },
   "setClipboard": {
    "!type": "fn(data: string, format?: string)",
    "!doc": "<p>Set clipboard content.</p>\n"
   },
   "startRobot": {
    "!type": "fn()"
   },
   "typeKeys": {
    "!type": "fn(chars: string|number, delay?: +Integer, duration?: +Integer)",
    "!doc": "<p>Types a string of characters in order, or types a dojo.keys.* constant.</p>\n"
   },
   "waitForPageToLoad": {
    "!type": "fn(submitActions: fn())",
    "!doc": "<p>Notifies DOH that the doh.robot is about to make a page change in the application it is driving,\nreturning a doh.Deferred object the user should return in their runTest function as part of a DOH test.</p>\n"
   }
  },
  "dojo/router/RouterBase": {
   "!type": "fn()",
   "prototype": {
    "_currentPath": {},
    "_routeIndex": {},
    "_routes": {},
    "_started": {},
    "globMatch": {},
    "globReplacement": {},
    "idMatch": {},
    "idReplacement": {},
    "destroy": {
     "!type": "fn()"
    },
    "go": {
     "!type": "fn(path: ?, replace: ?) -> bool",
     "!doc": "<p>A simple pass-through to make changing the hash easy,\nwithout having to require dojo/hash directly. It also\nsynchronously fires off any routes that match.</p>\n"
    },
    "register": {
     "!type": "fn(route: string|+RegExp, callback: fn())",
     "!doc": "<p>Registers a route to a handling callback</p>\n"
    },
    "registerBefore": {
     "!type": "fn(route: string|+RegExp, callback: fn())",
     "!doc": "<p>Registers a route to a handling callback, except before\nany previously registered callbacks</p>\n"
    },
    "startup": {
     "!type": "fn(defaultPath: ?)",
     "!doc": "<p>This method must be called to activate the router. Until\nstartup is called, no hash changes will trigger route\ncallbacks.</p>\n"
    }
   }
  },
  "dojo/router": {},
  "dojo/rpc/JsonService": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/rpc/RpcService.prototype",
    "bustCache": {},
    "contentType": {},
    "lastSubmissionId": {},
    "serviceUrl": {},
    "strictArgChecks": {},
    "bind": {
     "!type": "fn(method: string, parameters: [], deferredRequestHandler: +deferred, url: ?)",
     "!doc": "<p>JSON-RPC bind method. Takes remote method, parameters,\ndeferred, and a url, calls createRequest to make a JSON-RPC\nenvelope and passes that off with bind.</p>\n"
    },
    "callRemote": {
     "!type": "fn(method: string, params: [])",
     "!doc": "<p>call an arbitrary remote method without requiring it to be\npredefined with SMD</p>\n"
    },
    "createRequest": {
     "!type": "fn(method: string, params: +dojo)",
     "!doc": "<p>create a JSON-RPC envelope for the request</p>\n"
    },
    "errorCallback": {
     "!type": "fn(deferredRequestHandler: +dojo) -> fn()",
     "!doc": "<p>create callback that calls the Deferred errback method</p>\n"
    },
    "generateMethod": {
     "!type": "fn(method: string, parameters: +dojo, url: string)",
     "!doc": "<p>generate the local bind methods for the remote object</p>\n"
    },
    "parseResults": {
     "!type": "fn(obj: ?)",
     "!doc": "<p>parse the result envelope and pass the results back to\nthe callback function</p>\n"
    },
    "processSmd": {
     "!type": "fn(object: ?)",
     "!doc": "<p>callback method for receipt of a smd object.  Parse the smd\nand generate functions based on the description</p>\n"
    },
    "resultCallback": {
     "!type": "fn(deferredRequestHandler: +dojo)",
     "!doc": "<p>create callback that calls the Deferred&#39;s callback method</p>\n"
    }
   }
  },
  "dojo/rpc/RpcService": {
   "!type": "fn()",
   "prototype": {
    "serviceUrl": {},
    "strictArgChecks": {},
    "errorCallback": {
     "!type": "fn(deferredRequestHandler: +dojo) -> fn()",
     "!doc": "<p>create callback that calls the Deferred errback method</p>\n"
    },
    "generateMethod": {
     "!type": "fn(method: string, parameters: +dojo, url: string)",
     "!doc": "<p>generate the local bind methods for the remote object</p>\n"
    },
    "parseResults": {
     "!type": "fn(obj: ?)",
     "!doc": "<p>parse the results coming back from an rpc request.  this\nbase implementation, just returns the full object\nsubclasses should parse and only return the actual results</p>\n"
    },
    "processSmd": {
     "!type": "fn(object: ?)",
     "!doc": "<p>callback method for receipt of a smd object.  Parse the smd\nand generate functions based on the description</p>\n"
    },
    "resultCallback": {
     "!type": "fn(deferredRequestHandler: +dojo)",
     "!doc": "<p>create callback that calls the Deferred&#39;s callback method</p>\n"
    }
   }
  },
  "dojo/rpc/JsonpService": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/rpc/RpcService.prototype",
    "serviceUrl": {},
    "strictArgChecks": {},
    "bind": {
     "!type": "fn(method: string, parameters: +dojo, deferredRequestHandler: +deferred, url: ?)",
     "!doc": "<p>JSONP bind method. Takes remote method, parameters,\ndeferred, and a url, calls createRequest to make a JSON-RPC\nenvelope and passes that off with bind.</p>\n"
    },
    "createRequest": {
     "!type": "fn(parameters: ?)",
     "!doc": "<p>create a JSONP req</p>\n"
    },
    "errorCallback": {
     "!type": "fn(deferredRequestHandler: +dojo) -> fn()",
     "!doc": "<p>create callback that calls the Deferred errback method</p>\n"
    },
    "generateMethod": {
     "!type": "fn(method: string, parameters: +dojo, url: string)",
     "!doc": "<p>generate the local bind methods for the remote object</p>\n"
    },
    "parseResults": {
     "!type": "fn(obj: ?)",
     "!doc": "<p>parse the results coming back from an rpc request.  this\nbase implementation, just returns the full object\nsubclasses should parse and only return the actual results</p>\n"
    },
    "processSmd": {
     "!type": "fn(object: ?)",
     "!doc": "<p>callback method for receipt of a smd object.  Parse the smd\nand generate functions based on the description</p>\n"
    },
    "resultCallback": {
     "!type": "fn(deferredRequestHandler: +dojo)",
     "!doc": "<p>create callback that calls the Deferred&#39;s callback method</p>\n"
    }
   }
  },
  "dojo/selector/acme": {
   "!type": "fn(query: string, root?: string|+Node) -> []",
   "!doc": "<p>Returns nodes which match the given CSS3 selector, searching the\nentire document by default but optionally taking a node to scope\nthe search by. Returns an array.</p>\n",
   "filter": {
    "!type": "fn(nodeList: [+Node], filter: string, root?: string|+Node)",
    "!doc": "<p>function for filtering a NodeList based on a selector, optimized for simple selectors</p>\n"
   }
  },
  "dojo/selector/lite": {
   "!type": "fn(selector: ?, root: ?) -> []",
   "!doc": "<p>A small lightweight query selector engine that implements CSS2.1 selectors\nminus pseudo-classes and the sibling combinator, plus CSS3 attribute selectors</p>\n",
   "match": {}
  },
  "dojo/store/Cache": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/store/api/Store.prototype",
    "idProperty": {
     "!doc": "<p>If the store has a single primary key, this indicates the property to use as the\nidentity property. The values of this property should be unique.</p>\n"
    },
    "queryEngine": {
     "!doc": "<p>If the store can be queried locally (on the client side in JS), this defines\nthe query engine to use for querying the data store.\nThis takes a query and query options and returns a function that can execute\nthe provided query on a JavaScript array. The queryEngine may be replace to\nprovide more sophisticated querying capabilities. For example:\n\n</p>\n<pre><code>var query = store.queryEngine({foo:&quot;bar&quot;}, {count:10});\nquery(someArray) -&gt; filtered array</code></pre>\n<p>The returned query function may have a &quot;matches&quot; property that can be\n\n</p>\n<p>used to determine if an object matches the query. For example:\n\n</p>\n<pre><code>query.matches({id:&quot;some-object&quot;, foo:&quot;bar&quot;}) -&gt; true\nquery.matches({id:&quot;some-object&quot;, foo:&quot;something else&quot;}) -&gt; false</code></pre>\n"
    },
    "add": {
     "!type": "fn(object: ?, directives?: +dojo) -> number",
     "!doc": "<p>Add the given object to the store.</p>\n"
    },
    "evict": {
     "!type": "fn(id: number)",
     "!doc": "<p>Remove the object with the given id from the underlying caching store.</p>\n"
    },
    "get": {
     "!type": "fn(id: number, directives?: ?) -> +dojo",
     "!doc": "<p>Get the object with the specific id.</p>\n"
    },
    "getChildren": {
     "!type": "fn(parent: ?, options?: +dojo) -> +dojo",
     "!doc": "<p>Retrieves the children of an object.</p>\n"
    },
    "getIdentity": {
     "!type": "fn(object: ?) -> string",
     "!doc": "<p>Returns an object&#39;s identity</p>\n"
    },
    "getMetadata": {
     "!type": "fn(object: ?)",
     "!doc": "<p>Returns any metadata about the object. This may include attribution,\ncache directives, history, or version information.</p>\n"
    },
    "put": {
     "!type": "fn(object: ?, directives?: +dojo) -> number",
     "!doc": "<p>Put the object into the store (similar to an HTTP PUT).</p>\n"
    },
    "query": {
     "!type": "fn(query: ?|string, directives?: +dojo) -> +dojo",
     "!doc": "<p>Query the underlying master store and cache any results.</p>\n"
    },
    "remove": {
     "!type": "fn(id: number)",
     "!doc": "<p>Remove the object with the specific id.</p>\n"
    },
    "transaction": {
     "!type": "fn() -> +dojo",
     "!doc": "<p>Starts a new transaction.\nNote that a store user might not call transaction() prior to using put,\ndelete, etc. in which case these operations effectively could be thought of\nas &quot;auto-commit&quot; style actions.</p>\n"
    }
   }
  },
  "dojo/store/api/Store": {
   "!type": "fn()",
   "prototype": {
    "idProperty": {
     "!doc": "<p>If the store has a single primary key, this indicates the property to use as the\nidentity property. The values of this property should be unique.</p>\n"
    },
    "queryEngine": {
     "!doc": "<p>If the store can be queried locally (on the client side in JS), this defines\nthe query engine to use for querying the data store.\nThis takes a query and query options and returns a function that can execute\nthe provided query on a JavaScript array. The queryEngine may be replace to\nprovide more sophisticated querying capabilities. For example:\n\n</p>\n<pre><code>var query = store.queryEngine({foo:&quot;bar&quot;}, {count:10});\nquery(someArray) -&gt; filtered array</code></pre>\n<p>The returned query function may have a &quot;matches&quot; property that can be\n\n</p>\n<p>used to determine if an object matches the query. For example:\n\n</p>\n<pre><code>query.matches({id:&quot;some-object&quot;, foo:&quot;bar&quot;}) -&gt; true\nquery.matches({id:&quot;some-object&quot;, foo:&quot;something else&quot;}) -&gt; false</code></pre>\n"
    },
    "add": {
     "!type": "fn(object: ?, directives?: +dojo) -> number",
     "!doc": "<p>Creates an object, throws an error if the object already exists</p>\n"
    },
    "get": {
     "!type": "fn(id: number)",
     "!doc": "<p>Retrieves an object by its identity</p>\n"
    },
    "getChildren": {
     "!type": "fn(parent: ?, options?: +dojo) -> +dojo",
     "!doc": "<p>Retrieves the children of an object.</p>\n"
    },
    "getIdentity": {
     "!type": "fn(object: ?) -> string",
     "!doc": "<p>Returns an object&#39;s identity</p>\n"
    },
    "getMetadata": {
     "!type": "fn(object: ?)",
     "!doc": "<p>Returns any metadata about the object. This may include attribution,\ncache directives, history, or version information.</p>\n"
    },
    "put": {
     "!type": "fn(object: ?, directives?: +dojo) -> number",
     "!doc": "<p>Stores an object</p>\n"
    },
    "query": {
     "!type": "fn(query: string|?|fn(), options: +dojo) -> +dojo",
     "!doc": "<p>Queries the store for objects. This does not alter the store, but returns a\nset of data from the store.</p>\n"
    },
    "remove": {
     "!type": "fn(id: number)",
     "!doc": "<p>Deletes an object by its identity</p>\n"
    },
    "transaction": {
     "!type": "fn() -> +dojo",
     "!doc": "<p>Starts a new transaction.\nNote that a store user might not call transaction() prior to using put,\ndelete, etc. in which case these operations effectively could be thought of\nas &quot;auto-commit&quot; style actions.</p>\n"
    }
   },
   "PutDirectives": {
    "!type": "fn()"
   },
   "QueryOptions": {
    "!type": "fn()"
   },
   "QueryResults": {
    "!type": "fn()"
   },
   "SortInformation": {
    "!type": "fn()"
   },
   "Transaction": {
    "!type": "fn()"
   }
  },
  "dojo/store/api/Store.PutDirectives": {
   "!type": "fn()",
   "prototype": {
    "before": {
     "!doc": "<p>If the collection of objects in the store has a natural ordering,\nthis indicates that the created or updated object should be placed before the\nobject specified by the value of this property. A value of null indicates that the\nobject should be last.</p>\n"
    },
    "id": {
     "!doc": "<p>Indicates the identity of the object if a new object is created</p>\n"
    },
    "overwrite": {
     "!doc": "<p>If this is provided as a boolean it indicates that the object should or should not\noverwrite an existing object. A value of true indicates that a new object\nshould not be created, the operation should update an existing object. A\nvalue of false indicates that an existing object should not be updated, a new\nobject should be created (which is the same as an add() operation). When\nthis property is not provided, either an update or creation is acceptable.</p>\n"
    },
    "parent": {
     "!doc": "<p>If the store is hierarchical (with single parenting) this property indicates the\nnew parent of the created or updated object.</p>\n"
    }
   }
  },
  "dojo/store/api/Store.SortInformation": {
   "!type": "fn()",
   "prototype": {
    "attribute": {
     "!doc": "<p>The name of the attribute to sort on.</p>\n"
    },
    "descending": {
     "!doc": "<p>The direction of the sort.  Default is false.</p>\n"
    }
   }
  },
  "dojo/store/api/Store.QueryOptions": {
   "!type": "fn()",
   "prototype": {
    "count": {
     "!doc": "<p>The number of how many results should be returned.</p>\n"
    },
    "sort": {
     "!doc": "<p>A list of attributes to sort on, as well as direction\nFor example:\n\n</p>\n<pre><code>[{attribute:&quot;price, descending: true}].</code></pre>\n<p>If the sort parameter is omitted, then the natural order of the store may be\n\n</p>\n<p>applied if there is a natural order.</p>\n"
    },
    "start": {
     "!doc": "<p>The first result to begin iteration on</p>\n"
    }
   }
  },
  "dojo/store/api/Store.QueryResults": {
   "!type": "fn()",
   "prototype": {
    "total": {
     "!doc": "<p>This property should be included in if the query options included the &quot;count&quot;\nproperty limiting the result set. This property indicates the total number of objects\nmatching the query (as if &quot;start&quot; and &quot;count&quot; weren&#39;t present). This may be\na promise if the query is asynchronous.</p>\n"
    },
    "filter": {
     "!type": "fn(callback: ?, thisObject: ?) -> +dojo",
     "!doc": "<p>Filters the query results, based on\n<a href=\"https://developer.mozilla.org/en/Core_JavaScript_1.5_Reference/Objects/Array/filter\">https://developer.mozilla.org/en/Core_JavaScript_1.5_Reference/Objects/Array/filter</a>.\nNote that this may executed asynchronously. The callback may be called\nafter this function returns.</p>\n"
    },
    "forEach": {
     "!type": "fn(callback: ?, thisObject: ?)",
     "!doc": "<p>Iterates over the query results, based on\n<a href=\"https://developer.mozilla.org/en/Core_JavaScript_1.5_Reference/Objects/Array/forEach\">https://developer.mozilla.org/en/Core_JavaScript_1.5_Reference/Objects/Array/forEach</a>.\nNote that this may executed asynchronously. The callback may be called\nafter this function returns.</p>\n"
    },
    "map": {
     "!type": "fn(callback: ?, thisObject: ?) -> +dojo",
     "!doc": "<p>Maps the query results, based on\n<a href=\"https://developer.mozilla.org/en/Core_JavaScript_1.5_Reference/Objects/Array/map\">https://developer.mozilla.org/en/Core_JavaScript_1.5_Reference/Objects/Array/map</a>.\nNote that this may executed asynchronously. The callback may be called\nafter this function returns.</p>\n"
    },
    "observe": {
     "!type": "fn(listener: fn(), includeAllUpdates: ?)",
     "!doc": "<p>This registers a callback for notification of when data is modified in the query results.\nThis is an optional method, and is usually provided by dojo/store/Observable.</p>\n"
    },
    "then": {
     "!type": "fn(callback: ?, errorHandler: ?)",
     "!doc": "<p>This registers a callback for when the query is complete, if the query is asynchronous.\nThis is an optional method, and may not be present for synchronous queries.</p>\n"
    }
   }
  },
  "dojo/store/api/Store.Transaction": {
   "!type": "fn()",
   "prototype": {
    "abort": {
     "!type": "fn(callback: ?, thisObject: ?)",
     "!doc": "<p>Aborts the transaction. This may throw an error if it fails. Of if the operation\nis asynchronous, it may return a promise that represents the eventual success\nor failure of the abort.</p>\n"
    },
    "commit": {
     "!type": "fn()",
     "!doc": "<p>Commits the transaction. This may throw an error if it fails. Of if the operation\nis asynchronous, it may return a promise that represents the eventual success\nor failure of the commit.</p>\n"
    }
   }
  },
  "dojo/store/DataStore": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/store/api/Store.prototype",
    "idProperty": {
     "!doc": "<p>The object property to use to store the identity of the store items.</p>\n"
    },
    "store": {
     "!doc": "<p>The object store to convert to a data store</p>\n"
    },
    "target": {},
    "add": {
     "!type": "fn(object: ?, options?: +dojo) -> number",
     "!doc": "<p>Creates an object, throws an error if the object already exists</p>\n"
    },
    "get": {
     "!type": "fn(id?: ?, options: ?)",
     "!doc": "<p>Retrieves an object by it&#39;s identity. This will trigger a fetchItemByIdentity</p>\n"
    },
    "getChildren": {
     "!type": "fn(parent: ?, options?: +dojo) -> +dojo",
     "!doc": "<p>Retrieves the children of an object.</p>\n"
    },
    "getIdentity": {
     "!type": "fn(object: ?) -> number",
     "!doc": "<p>Fetch the identity for the given object.</p>\n"
    },
    "getMetadata": {
     "!type": "fn(object: ?)",
     "!doc": "<p>Returns any metadata about the object. This may include attribution,\ncache directives, history, or version information.</p>\n"
    },
    "put": {
     "!type": "fn(object: ?, options?: ?)",
     "!doc": "<p>Stores an object by its identity.</p>\n"
    },
    "query": {
     "!type": "fn(query: ?, options?: ?) -> +dojo",
     "!doc": "<p>Queries the store for objects.</p>\n"
    },
    "queryEngine": {
     "!type": "fn(query: ?, options?: +dojo) -> fn()",
     "!doc": "<p>Defines the query engine to use for querying the data store</p>\n"
    },
    "remove": {
     "!type": "fn(id: ?)",
     "!doc": "<p>Deletes an object by its identity.</p>\n"
    },
    "transaction": {
     "!type": "fn() -> +dojo",
     "!doc": "<p>Starts a new transaction.\nNote that a store user might not call transaction() prior to using put,\ndelete, etc. in which case these operations effectively could be thought of\nas &quot;auto-commit&quot; style actions.</p>\n"
    }
   }
  },
  "dojo/store/util/QueryResults": {
   "!type": "fn(results: []|+dojo) -> []",
   "!doc": "<p>A function that wraps the results of a store query with additional\nmethods.</p>\n"
  },
  "dojo/store/util/SimpleQueryEngine": {
   "!type": "fn(query: ?, options?: +dojo) -> fn()",
   "!doc": "<p>Simple query engine that matches using filter functions, named filter\nfunctions or objects by name-value on a query object hash\n</p>\n"
  },
  "dojo/store/JsonRest": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/store/api/Store.prototype",
    "accepts": {
     "!doc": "<p>Defines the Accept header to use on HTTP requests</p>\n"
    },
    "ascendingPrefix": {
     "!doc": "<p>The prefix to apply to sort attribute names that are ascending</p>\n"
    },
    "descendingPrefix": {
     "!doc": "<p>The prefix to apply to sort attribute names that are ascending</p>\n"
    },
    "headers": {
     "!doc": "<p>Additional headers to pass in all requests to the server. These can be overridden\nby passing additional headers to calls to the store.</p>\n"
    },
    "idProperty": {
     "!doc": "<p>Indicates the property to use as the identity property. The values of this\nproperty should be unique.</p>\n"
    },
    "queryEngine": {
     "!doc": "<p>If the store can be queried locally (on the client side in JS), this defines\nthe query engine to use for querying the data store.\nThis takes a query and query options and returns a function that can execute\nthe provided query on a JavaScript array. The queryEngine may be replace to\nprovide more sophisticated querying capabilities. For example:\n\n</p>\n<pre><code>var query = store.queryEngine({foo:&quot;bar&quot;}, {count:10});\nquery(someArray) -&gt; filtered array</code></pre>\n<p>The returned query function may have a &quot;matches&quot; property that can be\n\n</p>\n<p>used to determine if an object matches the query. For example:\n\n</p>\n<pre><code>query.matches({id:&quot;some-object&quot;, foo:&quot;bar&quot;}) -&gt; true\nquery.matches({id:&quot;some-object&quot;, foo:&quot;something else&quot;}) -&gt; false</code></pre>\n"
    },
    "target": {
     "!doc": "<p>The target base URL to use for all requests to the server. This string will be\nprepended to the id to generate the URL (relative or absolute) for requests\nsent to the server</p>\n"
    },
    "add": {
     "!type": "fn(object: ?, options?: ?)",
     "!doc": "<p>Adds an object. This will trigger a PUT request to the server\nif the object has an id, otherwise it will trigger a POST request.</p>\n"
    },
    "get": {
     "!type": "fn(id: number, options?: ?)",
     "!doc": "<p>Retrieves an object by its identity. This will trigger a GET request to the server using\nthe url <code>this.target + id</code>.</p>\n"
    },
    "getChildren": {
     "!type": "fn(parent: ?, options?: +dojo) -> +dojo",
     "!doc": "<p>Retrieves the children of an object.</p>\n"
    },
    "getIdentity": {
     "!type": "fn(object: ?) -> number",
     "!doc": "<p>Returns an object&#39;s identity</p>\n"
    },
    "getMetadata": {
     "!type": "fn(object: ?)",
     "!doc": "<p>Returns any metadata about the object. This may include attribution,\ncache directives, history, or version information.</p>\n"
    },
    "put": {
     "!type": "fn(object: ?, options?: ?) -> +dojo",
     "!doc": "<p>Stores an object. This will trigger a PUT request to the server\nif the object has an id, otherwise it will trigger a POST request.</p>\n"
    },
    "query": {
     "!type": "fn(query: ?, options?: ?) -> +dojo",
     "!doc": "<p>Queries the store for objects. This will trigger a GET request to the server, with the\nquery added as a query string.</p>\n"
    },
    "remove": {
     "!type": "fn(id: number, options?: ?)",
     "!doc": "<p>Deletes an object by its identity. This will trigger a DELETE request to the server.</p>\n"
    },
    "transaction": {
     "!type": "fn() -> +dojo",
     "!doc": "<p>Starts a new transaction.\nNote that a store user might not call transaction() prior to using put,\ndelete, etc. in which case these operations effectively could be thought of\nas &quot;auto-commit&quot; style actions.</p>\n"
    }
   }
  },
  "dojo/store/Memory": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/store/api/Store.prototype",
    "data": {
     "!doc": "<p>The array of all the objects in the memory store</p>\n"
    },
    "idProperty": {
     "!doc": "<p>Indicates the property to use as the identity property. The values of this\nproperty should be unique.</p>\n"
    },
    "index": {
     "!doc": "<p>An index of data indices into the data array by id</p>\n"
    },
    "add": {
     "!type": "fn(object: ?, options?: +dojo) -> number",
     "!doc": "<p>Creates an object, throws an error if the object already exists</p>\n"
    },
    "get": {
     "!type": "fn(id: number)",
     "!doc": "<p>Retrieves an object by its identity</p>\n"
    },
    "getChildren": {
     "!type": "fn(parent: ?, options?: +dojo) -> +dojo",
     "!doc": "<p>Retrieves the children of an object.</p>\n"
    },
    "getIdentity": {
     "!type": "fn(object: ?) -> number",
     "!doc": "<p>Returns an object&#39;s identity</p>\n"
    },
    "getMetadata": {
     "!type": "fn(object: ?)",
     "!doc": "<p>Returns any metadata about the object. This may include attribution,\ncache directives, history, or version information.</p>\n"
    },
    "put": {
     "!type": "fn(object: ?, options?: +dojo) -> number",
     "!doc": "<p>Stores an object</p>\n"
    },
    "query": {
     "!type": "fn(query: ?, options?: +dojo) -> +dojo",
     "!doc": "<p>Queries the store for objects.</p>\n"
    },
    "queryEngine": {
     "!type": "fn(query: ?, options?: +dojo) -> fn()",
     "!doc": "<p>Defines the query engine to use for querying the data store</p>\n"
    },
    "remove": {
     "!type": "fn(id: number) -> bool",
     "!doc": "<p>Deletes an object by its identity</p>\n"
    },
    "setData": {
     "!type": "fn(data: [+Object])",
     "!doc": "<p>Sets the given data as the source for this store, and indexes it</p>\n"
    },
    "transaction": {
     "!type": "fn() -> +dojo",
     "!doc": "<p>Starts a new transaction.\nNote that a store user might not call transaction() prior to using put,\ndelete, etc. in which case these operations effectively could be thought of\nas &quot;auto-commit&quot; style actions.</p>\n"
    }
   }
  },
  "dojo/store/Observable": {
   "!type": "fn(store: +dojo)",
   "!doc": "<p>The Observable store wrapper takes a store and sets an observe method on query()\nresults that can be used to monitor results for changes.\n</p>\n"
  },
  "dojo/uacss": {},
  "dijit/BackgroundIframe": {
   "!type": "fn(node: +Node)",
   "!doc": "<p>For IE/FF z-index shenanigans. id attribute is required.\n</p>\n",
   "prototype": {
    "destroy": {
     "!type": "fn()",
     "!doc": "<p>destroy the iframe</p>\n"
    },
    "resize": {
     "!type": "fn(node: ?)",
     "!doc": "<p>Resize the iframe so it&#39;s the same size as node.\nNeeded on IE6 and IE/quirks because height:100% doesn&#39;t work right.</p>\n"
    }
   }
  },
  "dijit/main": {
   "_activeStack": {
    "!doc": "<p>List of currently active widgets (focused widget and it&#39;s ancestors)</p>\n"
   },
   "_curFocus": {
    "!doc": "<p>Currently focused item on screen</p>\n"
   },
   "_masterTT": {},
   "_prevFocus": {
    "!doc": "<p>Previously focused item on screen</p>\n"
   },
   "_underlay": {},
   "defaultDuration": {
    "!doc": "<p>The default fx.animation speed (in ms) to use for all Dijit\ntransitional fx.animations, unless otherwise specified\non a per-instance basis. Defaults to 200, overrided by\n<code>djConfig.defaultDuration</code></p>\n"
   },
   "place": {
    "!doc": "<p>Code to place a DOMNode relative to another DOMNode.\nLoad using require([&quot;dijit/place&quot;], function(place){ ... }).</p>\n"
   },
   "popup": {},
   "registry": {
    "!doc": "<p>Registry of existing widget on page, plus some utility methods.</p>\n"
   },
   "typematic": {
    "!doc": "<p>These functions are used to repetitively call a user specified callback\nmethod when a specific key or mouse click over a specific DOM node is\nheld down for a specific amount of time.\nOnly 1 such event is allowed to occur on the browser page at 1 time.</p>\n"
   },
   "BackgroundIframe": {
    "!type": "fn(node: +Node)",
    "!doc": "<p>For IE/FF z-index shenanigans. id attribute is required.\n</p>\n"
   },
   "effectiveTabIndex": {
    "!type": "fn(elem: +Element) -> number",
    "!doc": "<p>Returns effective tabIndex of an element, either a number, or undefined if element isn&#39;t focusable.</p>\n"
   },
   "focus": {
    "!type": "fn(node: ?)"
   },
   "getBookmark": {
    "!type": "fn()",
    "!doc": "<p>Retrieves a bookmark that can be used with moveToBookmark to return to the same range</p>\n"
   },
   "getDocumentWindow": {
    "!type": "fn(doc: ?)"
   },
   "getFirstInTabbingOrder": {
    "!type": "fn(root: string|+Node, doc?: +Document)",
    "!doc": "<p>Finds the descendant of the specified root node\nthat is first in the tabbing order</p>\n"
   },
   "getFocus": {
    "!type": "fn(menu?: +dijit._Widget, openedForWindow?: +Window)",
    "!doc": "<p>Called as getFocus(), this returns an Object showing the current focus\nand selected text.\n\n</p>\n<p>Called as getFocus(widget), where widget is a (widget representing) a button\nthat was just pressed, it returns where focus was before that button\nwas pressed.   (Pressing the button may have either shifted focus to the button,\nor removed focus altogether.)   In this case the selected text is not returned,\nsince it can&#39;t be accurately determined.\n</p>\n"
   },
   "getLastInTabbingOrder": {
    "!type": "fn(root: string|+Node, doc?: +Document)",
    "!doc": "<p>Finds the descendant of the specified root node\nthat is last in the tabbing order</p>\n"
   },
   "getPopupAroundAlignment": {
    "!type": "fn(position: [string], leftToRight: bool)",
    "!doc": "<p>Deprecated method, unneeded when using dijit/place directly.\nTransforms the passed array of preferred positions into a format suitable for\npassing as the aroundCorners argument to dijit/place.placeOnScreenAroundElement.</p>\n"
   },
   "getViewport": {
    "!type": "fn()",
    "!doc": "<p>Deprecated method to return the dimensions and scroll position of the viewable area of a browser window.\nNew code should use windowUtils.getBox()</p>\n"
   },
   "getWaiRole": {
    "!type": "fn(elem: +Element)",
    "!doc": "<p>Gets the role for an element (which should be a wai role).</p>\n"
   },
   "getWaiState": {
    "!type": "fn(elem: +Element, state: string) -> string",
    "!doc": "<p>Gets the value of a state on an element.</p>\n"
   },
   "hasDefaultTabStop": {
    "!type": "fn(elem: +Element) -> bool",
    "!doc": "<p>Tests if element is tab-navigable even without an explicit tabIndex setting</p>\n"
   },
   "hasWaiRole": {
    "!type": "fn(elem: +Element, role?: string) -> bool",
    "!doc": "<p>Determines if an element has a particular role.</p>\n"
   },
   "hasWaiState": {
    "!type": "fn(elem: +Element, state: string) -> bool",
    "!doc": "<p>Determines if an element has a given state.</p>\n"
   },
   "hideTooltip": {
    "!type": "fn(aroundNode: ?)",
    "!doc": "<p>Static method to hide the tooltip displayed via showTooltip()</p>\n"
   },
   "isCollapsed": {
    "!type": "fn()",
    "!doc": "<p>Returns true if there is no text selected</p>\n"
   },
   "isFocusable": {
    "!type": "fn(elem: +Element) -> bool",
    "!doc": "<p>Tests if an element is focusable by tabbing to it, or clicking it with the mouse.</p>\n"
   },
   "isTabNavigable": {
    "!type": "fn(elem: +Element) -> bool",
    "!doc": "<p>Tests if an element is tab-navigable</p>\n"
   },
   "moveToBookmark": {
    "!type": "fn(bookmark: ?)",
    "!doc": "<p>Moves current selection to a bookmark</p>\n"
   },
   "placeOnScreen": {
    "!type": "fn(node: +Node, pos: +dijit, corners: [string], padding?: +dijit, layoutNode: ?)",
    "!doc": "<p>Positions node kitty-corner to the rectangle centered at (pos.x, pos.y) with width and height of\npadding.x <em> 2 and padding.y </em> 2, or zero if padding not specified.  Picks first corner in corners[]\nwhere node is fully visible, or the corner where it&#39;s most visible.\n\n</p>\n<p>Node is assumed to be absolutely or relatively positioned.</p>\n"
   },
   "placeOnScreenAroundElement": {
    "!type": "fn(node: ?, aroundNode: ?, aroundCorners: ?, layoutNode: ?)",
    "!doc": "<p>Like dijit.placeOnScreenAroundNode(), except it accepts an arbitrary object\nfor the &quot;around&quot; argument and finds a proper processor to place a node.\nDeprecated, new code should use dijit/place.around() instead.</p>\n"
   },
   "placeOnScreenAroundNode": {
    "!type": "fn(node: ?, aroundNode: ?, aroundCorners: ?, layoutNode: ?)",
    "!doc": "<p>Position node adjacent or kitty-corner to aroundNode\nsuch that it&#39;s fully visible in viewport.\nDeprecated, new code should use dijit/place.around() instead.</p>\n"
   },
   "placeOnScreenAroundRectangle": {
    "!type": "fn(node: ?, aroundRect: ?, aroundCorners: ?, layoutNode: ?)",
    "!doc": "<p>Like dijit.placeOnScreenAroundNode(), except that the &quot;around&quot;\nparameter is an arbitrary rectangle on the screen (x, y, width, height)\ninstead of a dom node.\nDeprecated, new code should use dijit/place.around() instead.</p>\n"
   },
   "registerIframe": {
    "!type": "fn(iframe: +Node)",
    "!doc": "<p>Registers listeners on the specified iframe so that any click\nor focus event on that iframe (or anything in it) is reported\nas a focus/click event on the <code>&lt;iframe&gt;</code> itself.</p>\n"
   },
   "registerWin": {
    "!type": "fn(targetWindow?: +Window, effectiveNode?: +Node)",
    "!doc": "<p>Registers listeners on the specified window (either the main\nwindow or an iframe&#39;s window) to detect when the user has clicked somewhere\nor focused somewhere.</p>\n"
   },
   "removeWaiRole": {
    "!type": "fn(elem: +Element, role: string)",
    "!doc": "<p>Removes the specified role from an element.\nRemoves role attribute if no specific role provided (for backwards compat.)</p>\n"
   },
   "removeWaiState": {
    "!type": "fn(elem: +Element, state: string)",
    "!doc": "<p>Removes a state from an element.</p>\n"
   },
   "scrollIntoView": {
    "!type": "fn(node: +Node, pos?: ?)",
    "!doc": "<p>Scroll the passed node into view, if it is not already.\nDeprecated, use <code>windowUtils.scrollIntoView</code> instead.</p>\n"
   },
   "selectInputText": {
    "!type": "fn(element: +Node, start?: number, stop?: number)",
    "!doc": "<p>Select text in the input element argument, from start (default 0), to stop (default end).</p>\n"
   },
   "setWaiRole": {
    "!type": "fn(elem: +Element, role: string)",
    "!doc": "<p>Sets the role on an element.</p>\n"
   },
   "setWaiState": {
    "!type": "fn(elem: +Element, state: string, value: string)",
    "!doc": "<p>Sets a state on an element.</p>\n"
   },
   "showTooltip": {
    "!type": "fn(innerHTML: string, aroundNode: ?, position?: [string], rtl?: bool, textDir?: string, onMouseEnter?: fn(), onMouseLeave?: fn())",
    "!doc": "<p>Static method to display tooltip w/specified contents in specified position.\nSee description of dijit/Tooltip.defaultPosition for details on position parameter.\nIf position is not specified then dijit/Tooltip.defaultPosition is used.</p>\n"
   },
   "unregisterIframe": {
    "!type": "fn(handle: ?)",
    "!doc": "<p>Unregisters listeners on the specified iframe created by registerIframe.\nAfter calling be sure to delete or null out the handle itself.</p>\n"
   },
   "unregisterWin": {
    "!type": "fn(handle: +Handle)",
    "!doc": "<p>Unregisters listeners on the specified window (either the main\nwindow or an iframe&#39;s window) according to handle returned from registerWin().\nAfter calling be sure to delete or null out the handle itself.</p>\n"
   }
  },
  "dijit/main.registry": {
   "_hash": {},
   "length": {
    "!doc": "<p>Number of registered widgets</p>\n"
   },
   "add": {
    "!type": "fn(widget: +dijit)",
    "!doc": "<p>Add a widget to the registry. If a duplicate ID is detected, a error is thrown.</p>\n"
   },
   "byId": {
    "!type": "fn(id: string|+dijit._Widget) -> string",
    "!doc": "<p>Find a widget by it&#39;s id.\nIf passed a widget then just returns the widget.</p>\n"
   },
   "byNode": {
    "!type": "fn(node: +Node)",
    "!doc": "<p>Returns the widget corresponding to the given DOMNode</p>\n"
   },
   "findWidgets": {
    "!type": "fn(root: +Node, skipNode: +Node) -> []",
    "!doc": "<p>Search subtree under root returning widgets found.\nDoesn&#39;t search for nested widgets (ie, widgets inside other widgets).</p>\n"
   },
   "getEnclosingWidget": {
    "!type": "fn(node: +Node)",
    "!doc": "<p>Returns the widget whose DOM tree contains the specified DOMNode, or null if\nthe node is not contained within the DOM tree of any widget</p>\n"
   },
   "getUniqueId": {
    "!type": "fn(widgetType: string) -> string",
    "!doc": "<p>Generates a unique id for a given widgetType</p>\n"
   },
   "remove": {
    "!type": "fn(id: string)",
    "!doc": "<p>Remove a widget from the registry. Does not destroy the widget; simply\nremoves the reference.</p>\n"
   },
   "toArray": {
    "!type": "fn() -> []",
    "!doc": "<p>Convert registry into a true Array\n</p>\n"
   }
  },
  "dijit/main.place": {
   "__Position": {},
   "__Rectangle": {},
   "around": {
    "!type": "fn(node: +Node, anchor: +Node|+dijit, positions: [string], leftToRight: bool, layoutNode?: fn())",
    "!doc": "<p>Position node adjacent or kitty-corner to anchor\nsuch that it&#39;s fully visible in viewport.</p>\n"
   },
   "at": {
    "!type": "fn(node: +Node, pos: +dijit, corners: [string], padding?: +dijit, layoutNode: ?)",
    "!doc": "<p>Positions node kitty-corner to the rectangle centered at (pos.x, pos.y) with width and height of\npadding.x <em> 2 and padding.y </em> 2, or zero if padding not specified.  Picks first corner in corners[]\nwhere node is fully visible, or the corner where it&#39;s most visible.\n\n</p>\n<p>Node is assumed to be absolutely or relatively positioned.</p>\n"
   }
  },
  "dijit/main.typematic": {
   "addKeyListener": {
    "!type": "fn(node: +Node, keyObject: ?, _this: ?, callback: fn(), subsequentDelay: number, initialDelay: number, minDelay?: number)",
    "!doc": "<p>Start listening for a specific typematic key.\nSee also the trigger method for other parameters.</p>\n"
   },
   "addListener": {
    "!type": "fn(mouseNode: +Node, keyNode: +Node, keyObject: ?, _this: ?, callback: fn(), subsequentDelay: number, initialDelay: number, minDelay?: number)",
    "!doc": "<p>Start listening for a specific typematic key and mouseclick.\nThis is a thin wrapper to addKeyListener and addMouseListener.\nSee the addMouseListener and addKeyListener methods for other parameters.</p>\n"
   },
   "addMouseListener": {
    "!type": "fn(node: +Node, _this: ?, callback: fn(), subsequentDelay: number, initialDelay: number, minDelay?: number)",
    "!doc": "<p>Start listening for a typematic mouse click.\nSee the trigger method for other parameters.</p>\n"
   },
   "stop": {
    "!type": "fn()",
    "!doc": "<p>Stop an ongoing timed, repeating callback sequence.</p>\n"
   },
   "trigger": {
    "!type": "fn(evt: +Event, _this: ?, node: +Node, callback: fn(), obj: ?, subsequentDelay?: number, initialDelay?: number, minDelay?: number)",
    "!doc": "<p>Start a timed, repeating callback sequence.\nIf already started, the function call is ignored.\nThis method is not normally called by the user but can be\nwhen the normal listener code is insufficient.</p>\n"
   }
  },
  "dijit/main._Calendar": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/CalendarLite.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setSummaryAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {},
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {},
    "currentFocus": {
     "!doc": "<p>Date object containing the currently focused date, or the date which would be focused\nif the calendar itself was focused.   Also indicates which year and month to display,\ni.e. the current &quot;page&quot; the calendar is on.</p>\n"
    },
    "datePackage": {
     "!doc": "<p>JavaScript namespace to find calendar routines.  If unspecified, uses Gregorian calendar routines\nat dojo/date and dojo/date/locale.</p>\n"
    },
    "dateTemplateString": {},
    "dayWidth": {
     "!doc": "<p>How to represent the days of the week in the calendar header. See locale</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "dowTemplateString": {},
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "searchContainerNode": {},
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Order fields are traversed when user hits the tab key</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "value": {
     "!doc": "<p>The currently selected Date, initially set to invalid date to indicate no selection.</p>\n"
    },
    "weekTemplateString": {},
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Focus the calendar by focusing one of the calendar cells</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getClassForDate": {
     "!type": "fn(dateObject: +Date, locale?: string) -> string",
     "!doc": "<p>May be overridden to return CSS classes to associate with the date entry for the given dateObject,\nfor example to indicate a holiday in specified locale.</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "goToToday": {
     "!type": "fn()",
     "!doc": "<p>Sets calendar&#39;s value to today&#39;s date</p>\n"
    },
    "handleKey": {
     "!type": "fn(evt: +Event) -> bool",
     "!doc": "<p>Provides keyboard navigation of calendar.</p>\n"
    },
    "isDisabledDate": {
     "!type": "fn(dateObject: +Date, locale?: string) -> bool",
     "!doc": "<p>May be overridden to disable certain dates in the calendar e.g. <code>isDisabledDate=dojo.date.locale.isWeekend</code></p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())",
     "!doc": "<p>Call specified function when event occurs, ex: myWidget.on(&quot;click&quot;, function(){ ... }).</p>\n"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setValue": {
     "!type": "fn(value: +Date)",
     "!doc": "<p>Deprecated.   Use set(&#39;value&#39;, ...) instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/Calendar": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/CalendarLite.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setSummaryAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {},
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {},
    "currentFocus": {
     "!doc": "<p>Date object containing the currently focused date, or the date which would be focused\nif the calendar itself was focused.   Also indicates which year and month to display,\ni.e. the current &quot;page&quot; the calendar is on.</p>\n"
    },
    "datePackage": {
     "!doc": "<p>JavaScript namespace to find calendar routines.  If unspecified, uses Gregorian calendar routines\nat dojo/date and dojo/date/locale.</p>\n"
    },
    "dateTemplateString": {},
    "dayWidth": {
     "!doc": "<p>How to represent the days of the week in the calendar header. See locale</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "dowTemplateString": {},
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "searchContainerNode": {},
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Order fields are traversed when user hits the tab key</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "value": {
     "!doc": "<p>The currently selected Date, initially set to invalid date to indicate no selection.</p>\n"
    },
    "weekTemplateString": {},
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Focus the calendar by focusing one of the calendar cells</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getClassForDate": {
     "!type": "fn(dateObject: +Date, locale?: string) -> string",
     "!doc": "<p>May be overridden to return CSS classes to associate with the date entry for the given dateObject,\nfor example to indicate a holiday in specified locale.</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "goToToday": {
     "!type": "fn()",
     "!doc": "<p>Sets calendar&#39;s value to today&#39;s date</p>\n"
    },
    "handleKey": {
     "!type": "fn(evt: +Event) -> bool",
     "!doc": "<p>Provides keyboard navigation of calendar.</p>\n"
    },
    "isDisabledDate": {
     "!type": "fn(dateObject: +Date, locale?: string) -> bool",
     "!doc": "<p>May be overridden to disable certain dates in the calendar e.g. <code>isDisabledDate=dojo.date.locale.isWeekend</code></p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())",
     "!doc": "<p>Call specified function when event occurs, ex: myWidget.on(&quot;click&quot;, function(){ ... }).</p>\n"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setValue": {
     "!type": "fn(value: +Date)",
     "!doc": "<p>Deprecated.   Use set(&#39;value&#39;, ...) instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/Calendar._MonthDropDownButton": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/form/DropDownButton.prototype",
    "_aroundNode": {
     "!doc": "<p>The node to display the popup around.\nCan be set via a data-dojo-attach-point assignment.\nIf missing, then domNode will be used.</p>\n"
    },
    "_arrowWrapperNode": {
     "!doc": "<p>Will set CSS class dijitUpArrow, dijitDownArrow, dijitRightArrow etc. on this node depending\non where the drop down is set to be positioned.\nCan be set via a data-dojo-attach-point assignment.\nIf missing, then _buttonNode will be used.</p>\n"
    },
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_buttonNode": {
     "!doc": "<p>The button/icon/node to click to display the drop down.\nCan be set via a data-dojo-attach-point assignment.\nIf missing, then either focusNode or domNode (if focusNode is also missing) will be used.</p>\n"
    },
    "_focusManager": {},
    "_onChangeActive": {
     "!doc": "<p>Indicates that changes to the value should call onChange() callback.\nThis is false during widget initialization, to avoid calling onChange()\nwhen the initial value is set.</p>\n"
    },
    "_popupStateNode": {
     "!doc": "<p>The node to set the aria-expanded class on.\nAlso sets popupActive class but that will be removed in 2.0.\nCan be set via a data-dojo-attach-point assignment.\nIf missing, then focusNode or _buttonNode (if focusNode is missing) will be used.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIconClassAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTabIndexAttr": {},
    "_setTypeAttr": {},
    "_setValueAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "_stopClickEvents": {
     "!doc": "<p>When set to false, the click events will not be stopped, in\ncase you want to use them in your subclass</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "alt": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "aria-label": {},
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "autoWidth": {
     "!doc": "<p>Set to true to make the drop down at least as wide as this\nwidget.  Set to false if the drop down should just be its\ndefault width.</p>\n"
    },
    "baseClass": {},
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "disabled": {
     "!doc": "<p>Should this widget respond to user input?\nIn markup, this is specified as &quot;disabled=&#39;disabled&#39;&quot;, or just &quot;disabled&quot;.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "dropDown": {
     "!doc": "<p>The widget to display as a popup.  This widget <em>must</em> be\ndefined before the startup function is called.</p>\n"
    },
    "dropDownPosition": {
     "!doc": "<p>This variable controls the position of the drop down.\nIt&#39;s an array of strings with the following values:\n\n</p>\n<ul>\n<li>before: places drop down to the left of the target node/widget, or to the right in\nthe case of RTL scripts like Hebrew and Arabic</li>\n<li>after: places drop down to the right of the target node/widget, or to the left in\nthe case of RTL scripts like Hebrew and Arabic</li>\n<li>above: drop down goes above target node</li>\n<li>below: drop down goes below target node</li>\n</ul>\n<p>The list is positions is tried, in order, until a position is found where the drop down fits\nwithin the viewport.\n</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "forceWidth": {
     "!doc": "<p>Set to true to make the drop down exactly as wide as this\nwidget.  Overrides autoWidth.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "iconClass": {
     "!doc": "<p>Class to apply to DOMNode in button to make it display an icon</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "intermediateChanges": {
     "!doc": "<p>Fires onChange for each value change or only on demand</p>\n"
    },
    "label": {
     "!doc": "<p>Content to display in button.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "maxHeight": {
     "!doc": "<p>The max height for our dropdown.\nAny dropdown taller than this will have scrollbars.\nSet to 0 for no max height, or -1 to limit height to available space in viewport</p>\n"
    },
    "name": {
     "!doc": "<p>Name used when submitting form; same as &quot;name&quot; attribute or plain HTML elements</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "scrollOnFocus": {
     "!doc": "<p>On focus, should this widget scroll into view?</p>\n"
    },
    "searchContainerNode": {},
    "showLabel": {
     "!doc": "<p>Set this to true to hide the label text and display only the icon.\n(If showLabel=false then iconClass must be specified.)\nEspecially useful for toolbars.\nIf showLabel=true, the label will become the title (a.k.a. tooltip/hint) of the icon.\n\n</p>\n<p>The exception case is for computers in high-contrast mode, where the label\nwill still be displayed, since the icon doesn&#39;t appear.</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Order fields are traversed when user hits the tab key</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "type": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "value": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "addChild": {
     "!type": "fn(widget: +dijit, insertIndex?: number)",
     "!doc": "<p>Makes the given widget a child of this widget.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "closeDropDown": {
     "!type": "fn(focus: bool)",
     "!doc": "<p>Closes the drop down on this widget</p>\n"
    },
    "compare": {
     "!type": "fn(val1: +anything, val2: +anything) -> number",
     "!doc": "<p>Compare 2 values (as returned by get(&#39;value&#39;) for this widget).</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Put focus on this widget</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getIndexOfChild": {
     "!type": "fn(child: +dijit)",
     "!doc": "<p>Gets the index of the child in this container or -1 if not found</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "getValue": {
     "!type": "fn()",
     "!doc": "<p>Deprecated.  Use get(&#39;value&#39;) instead.</p>\n"
    },
    "hasChildren": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if widget has child widgets, i.e. if this.containerNode contains widgets.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "isLoaded": {
     "!type": "fn() -> bool"
    },
    "loadAndOpenDropDown": {
     "!type": "fn() -> +Deferred",
     "!doc": "<p>Creates the drop down if it doesn&#39;t exist, loads the data\nif there&#39;s an href and it hasn&#39;t been loaded yet, and\nthen opens the drop down.  This is basically a callback when the\nuser presses the down arrow button to open the drop down.</p>\n"
    },
    "loadDropDown": {
     "!type": "fn(callback: fn())"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "openDropDown": {
     "!type": "fn()",
     "!doc": "<p>Opens the dropdown for this widget.   To be called only when this.dropDown\nhas been created and is ready to display (ie, it&#39;s data is loaded).</p>\n"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()"
    },
    "removeChild": {
     "!type": "fn(widget: +dijit._Widget|number)",
     "!doc": "<p>Removes the passed widget instance from this widget but does\nnot destroy it.  You can also pass in an integer indicating\nthe index within the container to remove (ie, removeChild(5) removes the sixth widget).</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setDisabled": {
     "!type": "fn(disabled: bool)",
     "!doc": "<p>Deprecated.  Use set(&#39;disabled&#39;, ...) instead.</p>\n"
    },
    "setLabel": {
     "!type": "fn(content: string)",
     "!doc": "<p>Deprecated.  Use set(&#39;label&#39;, ...) instead.</p>\n"
    },
    "setValue": {
     "!type": "fn(value: string)",
     "!doc": "<p>Deprecated.  Use set(&#39;value&#39;, ...) instead.</p>\n"
    },
    "startup": {
     "!type": "fn()"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toggleDropDown": {
     "!type": "fn()",
     "!doc": "<p>Callback when the user presses the down arrow button or presses\nthe down arrow key to open/close the drop down.\nToggle the drop-down widget; if it is up, close it, if not, open it</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/Calendar._MonthDropDown": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_Widget.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {},
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "months": {
     "!doc": "<p>List of names of months, possibly w/some undefined entries for Hebrew leap months\n(ex: [&quot;January&quot;, &quot;February&quot;, undefined, &quot;April&quot;, ...])</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "searchContainerNode": {},
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/CalendarLite": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_WidgetBase.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setSummaryAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {},
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "currentFocus": {
     "!doc": "<p>Date object containing the currently focused date, or the date which would be focused\nif the calendar itself was focused.   Also indicates which year and month to display,\ni.e. the current &quot;page&quot; the calendar is on.</p>\n"
    },
    "datePackage": {
     "!doc": "<p>JavaScript namespace to find calendar routines.  If unspecified, uses Gregorian calendar routines\nat dojo/date and dojo/date/locale.</p>\n"
    },
    "dateTemplateString": {},
    "dayWidth": {
     "!doc": "<p>How to represent the days of the week in the calendar header. See locale</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "dowTemplateString": {},
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "searchContainerNode": {},
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Order fields are traversed when user hits the tab key</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "value": {
     "!doc": "<p>The currently selected Date, initially set to invalid date to indicate no selection.</p>\n"
    },
    "weekTemplateString": {},
    "buildRendering": {
     "!type": "fn()"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Focus the calendar by focusing one of the calendar cells</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getClassForDate": {
     "!type": "fn(dateObject: +Date, locale?: string) -> string",
     "!doc": "<p>May be overridden to return CSS classes to associate with the date entry for the given dateObject,\nfor example to indicate a holiday in specified locale.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "goToToday": {
     "!type": "fn()",
     "!doc": "<p>Sets calendar&#39;s value to today&#39;s date</p>\n"
    },
    "isDisabledDate": {
     "!type": "fn(dateObject: +Date, locale?: string) -> bool",
     "!doc": "<p>May be overridden to disable certain dates in the calendar e.g. <code>isDisabledDate=dojo.date.locale.isWeekend</code></p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())",
     "!doc": "<p>Call specified function when event occurs, ex: myWidget.on(&quot;click&quot;, function(){ ... }).</p>\n"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/CalendarLite._MonthWidget": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_WidgetBase.prototype",
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {
     "!doc": "<p>Root CSS class of the widget (ex: dijitTextBox), used to construct CSS classes to indicate\nwidget state.</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())",
     "!doc": "<p>Call specified function when event occurs, ex: myWidget.on(&quot;click&quot;, function(){ ... }).</p>\n"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is created</p>\n"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/_WidgetBase": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/Stateful.prototype",
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {
     "!doc": "<p>Root CSS class of the widget (ex: dijitTextBox), used to construct CSS classes to indicate\nwidget state.</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "isValid": {
     "!type": "fn()"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())",
     "!doc": "<p>Call specified function when event occurs, ex: myWidget.on(&quot;click&quot;, function(){ ... }).</p>\n"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is created</p>\n"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/Destroyable": {
   "!type": "fn()",
   "prototype": {
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this class, releasing any resources registered via own().</p>\n"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    }
   }
  },
  "dijit/registry": {
   "_hash": {},
   "length": {
    "!doc": "<p>Number of registered widgets</p>\n"
   },
   "add": {
    "!type": "fn(widget: +dijit)",
    "!doc": "<p>Add a widget to the registry. If a duplicate ID is detected, a error is thrown.</p>\n"
   },
   "byId": {
    "!type": "fn(id: string|+dijit._Widget) -> string",
    "!doc": "<p>Find a widget by it&#39;s id.\nIf passed a widget then just returns the widget.</p>\n"
   },
   "byNode": {
    "!type": "fn(node: +Node)",
    "!doc": "<p>Returns the widget corresponding to the given DOMNode</p>\n"
   },
   "findWidgets": {
    "!type": "fn(root: +Node, skipNode: +Node) -> []",
    "!doc": "<p>Search subtree under root returning widgets found.\nDoesn&#39;t search for nested widgets (ie, widgets inside other widgets).</p>\n"
   },
   "getEnclosingWidget": {
    "!type": "fn(node: +Node)",
    "!doc": "<p>Returns the widget whose DOM tree contains the specified DOMNode, or null if\nthe node is not contained within the DOM tree of any widget</p>\n"
   },
   "getUniqueId": {
    "!type": "fn(widgetType: string) -> string",
    "!doc": "<p>Generates a unique id for a given widgetType</p>\n"
   },
   "remove": {
    "!type": "fn(id: string)",
    "!doc": "<p>Remove a widget from the registry. Does not destroy the widget; simply\nremoves the reference.</p>\n"
   },
   "toArray": {
    "!type": "fn() -> []",
    "!doc": "<p>Convert registry into a true Array\n</p>\n"
   }
  },
  "dijit/registry._hash": {},
  "dijit/_TemplatedMixin": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_AttachMixin.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_skipNodeCache": {},
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "searchContainerNode": {},
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {
     "!doc": "<p>A string that represents the widget template.\nUse in conjunction with dojo.cache() to load from a file.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget from a template, setting this.domNode.</p>\n"
    },
    "destroyRendering": {
     "!type": "fn()"
    }
   },
   "_templateCache": {},
   "getCachedTemplate": {
    "!type": "fn(templateString: string, alwaysUseString: bool, doc?: +Document) -> +Mixed",
    "!doc": "<p>Static method to get a template based on the templatePath or\ntemplateString key</p>\n"
   }
  },
  "dijit/_TemplatedMixin._templateCache": {},
  "dijit/_AttachMixin": {
   "!type": "fn()",
   "prototype": {
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "searchContainerNode": {
     "!doc": "<p>Search descendants of this.containerNode for data-dojo-attach-point and data-dojo-attach-event.\nShould generally be left false (the default value) both for performance and to avoid failures when\nthis.containerNode holds other _AttachMixin instances with their own attach points and events.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Attach to DOM nodes marked with special attributes.</p>\n"
    },
    "destroyRendering": {
     "!type": "fn()"
    }
   }
  },
  "dijit/a11yclick": {
   "click": {
    "!type": "fn(node: ?, listener: ?)",
    "!doc": "<p>Logical click operation for mouse, touch, or keyboard (space/enter key)</p>\n"
   },
   "move": {
    "!type": "fn(node: ?, listener: ?)",
    "!doc": "<p>Mouse cursor or a finger is dragged over the given node.</p>\n"
   },
   "press": {
    "!type": "fn(node: ?, listener: ?)",
    "!doc": "<p>Mousedown (left button), touchstart, or keydown (space or enter) corresponding to logical click operation.</p>\n"
   },
   "release": {
    "!type": "fn(node: ?, listener: ?)",
    "!doc": "<p>Mouseup (left button), touchend, or keyup (space or enter) corresponding to logical click operation.</p>\n"
   }
  },
  "dijit/hccss": {
   "!type": "fn()",
   "!doc": "<p>Test if computer is in high contrast mode, and sets <code>dijit_a11y</code> flag on <code>&lt;body&gt;</code> if it is.\nDeprecated, use <code>dojo/hccss</code> instead.</p>\n"
  },
  "dijit/_Widget": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_WidgetBase.prototype",
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {
     "!doc": "<p>Root CSS class of the widget (ex: dijitTextBox), used to construct CSS classes to indicate\nwidget state.</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "dndType": {
     "!doc": "<p>Defines a type of widget.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "observer": {},
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "slideFrom": {
     "!doc": "<p>A parameter needed by RadioGroupSlide only. An optional parameter to force\nthe ContentPane to slide in from a set direction. Defaults\nto &quot;random&quot;, or specify one of &quot;top&quot;, &quot;left&quot;, &quot;right&quot;, &quot;bottom&quot;\nto slideFrom top, left, right, or bottom.</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/_OnDijitClickMixin": {
   "!type": "fn()",
   "prototype": {
    "connect": {
     "!type": "fn(obj: ?, event: ?, method: ?)"
    }
   },
   "a11yclick": {
    "!doc": "<p>Custom press, release, and click synthetic events\nwhich trigger on a left mouse click, touch, or space/enter keyup.</p>\n"
   }
  },
  "dijit/_OnDijitClickMixin.a11yclick": {
   "click": {
    "!type": "fn(node: ?, listener: ?)",
    "!doc": "<p>Logical click operation for mouse, touch, or keyboard (space/enter key)</p>\n"
   },
   "move": {
    "!type": "fn(node: ?, listener: ?)",
    "!doc": "<p>Mouse cursor or a finger is dragged over the given node.</p>\n"
   },
   "press": {
    "!type": "fn(node: ?, listener: ?)",
    "!doc": "<p>Mousedown (left button), touchstart, or keydown (space or enter) corresponding to logical click operation.</p>\n"
   },
   "release": {
    "!type": "fn(node: ?, listener: ?)",
    "!doc": "<p>Mouseup (left button), touchend, or keyup (space or enter) corresponding to logical click operation.</p>\n"
   }
  },
  "dijit/_FocusMixin": {
   "!type": "fn()",
   "prototype": {
    "_focusManager": {}
   }
  },
  "dijit/focus": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/Stateful.prototype",
    "_attrPairNames": {
     "!doc": "<p>Used across all instances a hash to cache attribute names and their getter \nand setter names.</p>\n"
    },
    "activeStack": {
     "!doc": "<p>List of currently active widgets (focused widget and it&#39;s ancestors)</p>\n"
    },
    "curNode": {
     "!doc": "<p>Currently focused item on screen</p>\n"
    },
    "emit": {
     "!type": "fn(type: ?, event: ?)"
    },
    "focus": {
     "!type": "fn(node: ?)",
     "!doc": "<p>Focus the specified node, suppressing errors if they occur</p>\n"
    },
    "get": {
     "!type": "fn(name: string)",
     "!doc": "<p>Get a property on a Stateful instance.</p>\n"
    },
    "on": {
     "!type": "fn(type: ?, listener: ?)"
    },
    "postscript": {
     "!type": "fn(params?: ?)"
    },
    "registerIframe": {
     "!type": "fn(iframe: +Node)",
     "!doc": "<p>Registers listeners on the specified iframe so that any click\nor focus event on that iframe (or anything in it) is reported\nas a focus/click event on the <code>&lt;iframe&gt;</code> itself.</p>\n"
    },
    "registerWin": {
     "!type": "fn(targetWindow?: +Window, effectiveNode?: +Node)",
     "!doc": "<p>Registers listeners on the specified window (either the main\nwindow or an iframe&#39;s window) to detect when the user has clicked somewhere\nor focused somewhere.</p>\n"
    },
    "set": {
     "!type": "fn(name: string, value: ?) -> fn()",
     "!doc": "<p>Set a property on a Stateful instance</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/a11y": {
   "effectiveTabIndex": {
    "!type": "fn(elem: +Element) -> number",
    "!doc": "<p>Returns effective tabIndex of an element, either a number, or undefined if element isn&#39;t focusable.</p>\n"
   },
   "getFirstInTabbingOrder": {
    "!type": "fn(root: string|+Node, doc?: +Document)",
    "!doc": "<p>Finds the descendant of the specified root node\nthat is first in the tabbing order</p>\n"
   },
   "getLastInTabbingOrder": {
    "!type": "fn(root: string|+Node, doc?: +Document)",
    "!doc": "<p>Finds the descendant of the specified root node\nthat is last in the tabbing order</p>\n"
   },
   "hasDefaultTabStop": {
    "!type": "fn(elem: +Element) -> bool",
    "!doc": "<p>Tests if element is tab-navigable even without an explicit tabIndex setting</p>\n"
   },
   "isFocusable": {
    "!type": "fn(elem: +Element) -> bool",
    "!doc": "<p>Tests if an element is focusable by tabbing to it, or clicking it with the mouse.</p>\n"
   },
   "isTabNavigable": {
    "!type": "fn(elem: +Element) -> bool",
    "!doc": "<p>Tests if an element is tab-navigable</p>\n"
   }
  },
  "dijit/_CssStateMixin": {
   "!type": "fn()",
   "prototype": {
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    }
   }
  },
  "dijit/form/DropDownButton": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/form/Button.prototype",
    "_aroundNode": {
     "!doc": "<p>The node to display the popup around.\nCan be set via a data-dojo-attach-point assignment.\nIf missing, then domNode will be used.</p>\n"
    },
    "_arrowWrapperNode": {
     "!doc": "<p>Will set CSS class dijitUpArrow, dijitDownArrow, dijitRightArrow etc. on this node depending\non where the drop down is set to be positioned.\nCan be set via a data-dojo-attach-point assignment.\nIf missing, then _buttonNode will be used.</p>\n"
    },
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_buttonNode": {
     "!doc": "<p>The button/icon/node to click to display the drop down.\nCan be set via a data-dojo-attach-point assignment.\nIf missing, then either focusNode or domNode (if focusNode is also missing) will be used.</p>\n"
    },
    "_focusManager": {},
    "_onChangeActive": {
     "!doc": "<p>Indicates that changes to the value should call onChange() callback.\nThis is false during widget initialization, to avoid calling onChange()\nwhen the initial value is set.</p>\n"
    },
    "_popupStateNode": {
     "!doc": "<p>The node to set the aria-expanded class on.\nAlso sets popupActive class but that will be removed in 2.0.\nCan be set via a data-dojo-attach-point assignment.\nIf missing, then focusNode or _buttonNode (if focusNode is missing) will be used.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIconClassAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTabIndexAttr": {},
    "_setTypeAttr": {},
    "_setValueAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "_stopClickEvents": {
     "!doc": "<p>When set to false, the click events will not be stopped, in\ncase you want to use them in your subclass</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "alt": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "aria-label": {},
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "autoWidth": {
     "!doc": "<p>Set to true to make the drop down at least as wide as this\nwidget.  Set to false if the drop down should just be its\ndefault width.</p>\n"
    },
    "baseClass": {},
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "disabled": {
     "!doc": "<p>Should this widget respond to user input?\nIn markup, this is specified as &quot;disabled=&#39;disabled&#39;&quot;, or just &quot;disabled&quot;.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "dropDown": {
     "!doc": "<p>The widget to display as a popup.  This widget <em>must</em> be\ndefined before the startup function is called.</p>\n"
    },
    "dropDownPosition": {
     "!doc": "<p>This variable controls the position of the drop down.\nIt&#39;s an array of strings with the following values:\n\n</p>\n<ul>\n<li>before: places drop down to the left of the target node/widget, or to the right in\nthe case of RTL scripts like Hebrew and Arabic</li>\n<li>after: places drop down to the right of the target node/widget, or to the left in\nthe case of RTL scripts like Hebrew and Arabic</li>\n<li>above: drop down goes above target node</li>\n<li>below: drop down goes below target node</li>\n</ul>\n<p>The list is positions is tried, in order, until a position is found where the drop down fits\nwithin the viewport.\n</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "forceWidth": {
     "!doc": "<p>Set to true to make the drop down exactly as wide as this\nwidget.  Overrides autoWidth.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "iconClass": {
     "!doc": "<p>Class to apply to DOMNode in button to make it display an icon</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "intermediateChanges": {
     "!doc": "<p>Fires onChange for each value change or only on demand</p>\n"
    },
    "label": {
     "!doc": "<p>Content to display in button.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "maxHeight": {
     "!doc": "<p>The max height for our dropdown.\nAny dropdown taller than this will have scrollbars.\nSet to 0 for no max height, or -1 to limit height to available space in viewport</p>\n"
    },
    "name": {
     "!doc": "<p>Name used when submitting form; same as &quot;name&quot; attribute or plain HTML elements</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "scrollOnFocus": {
     "!doc": "<p>On focus, should this widget scroll into view?</p>\n"
    },
    "searchContainerNode": {},
    "showLabel": {
     "!doc": "<p>Set this to true to hide the label text and display only the icon.\n(If showLabel=false then iconClass must be specified.)\nEspecially useful for toolbars.\nIf showLabel=true, the label will become the title (a.k.a. tooltip/hint) of the icon.\n\n</p>\n<p>The exception case is for computers in high-contrast mode, where the label\nwill still be displayed, since the icon doesn&#39;t appear.</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Order fields are traversed when user hits the tab key</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "type": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "value": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "addChild": {
     "!type": "fn(widget: +dijit, insertIndex?: number)",
     "!doc": "<p>Makes the given widget a child of this widget.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "closeDropDown": {
     "!type": "fn(focus: bool)",
     "!doc": "<p>Closes the drop down on this widget</p>\n"
    },
    "compare": {
     "!type": "fn(val1: +anything, val2: +anything) -> number",
     "!doc": "<p>Compare 2 values (as returned by get(&#39;value&#39;) for this widget).</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Put focus on this widget</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getIndexOfChild": {
     "!type": "fn(child: +dijit)",
     "!doc": "<p>Gets the index of the child in this container or -1 if not found</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "getValue": {
     "!type": "fn()",
     "!doc": "<p>Deprecated.  Use get(&#39;value&#39;) instead.</p>\n"
    },
    "hasChildren": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if widget has child widgets, i.e. if this.containerNode contains widgets.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "isLoaded": {
     "!type": "fn() -> bool"
    },
    "loadAndOpenDropDown": {
     "!type": "fn() -> +Deferred",
     "!doc": "<p>Creates the drop down if it doesn&#39;t exist, loads the data\nif there&#39;s an href and it hasn&#39;t been loaded yet, and\nthen opens the drop down.  This is basically a callback when the\nuser presses the down arrow button to open the drop down.</p>\n"
    },
    "loadDropDown": {
     "!type": "fn(callback: fn())"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "openDropDown": {
     "!type": "fn()",
     "!doc": "<p>Opens the dropdown for this widget.   To be called only when this.dropDown\nhas been created and is ready to display (ie, it&#39;s data is loaded).</p>\n"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()"
    },
    "removeChild": {
     "!type": "fn(widget: +dijit._Widget|number)",
     "!doc": "<p>Removes the passed widget instance from this widget but does\nnot destroy it.  You can also pass in an integer indicating\nthe index within the container to remove (ie, removeChild(5) removes the sixth widget).</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setDisabled": {
     "!type": "fn(disabled: bool)",
     "!doc": "<p>Deprecated.  Use set(&#39;disabled&#39;, ...) instead.</p>\n"
    },
    "setLabel": {
     "!type": "fn(content: string)",
     "!doc": "<p>Deprecated.  Use set(&#39;label&#39;, ...) instead.</p>\n"
    },
    "setValue": {
     "!type": "fn(value: string)",
     "!doc": "<p>Deprecated.  Use set(&#39;value&#39;, ...) instead.</p>\n"
    },
    "startup": {
     "!type": "fn()"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toggleDropDown": {
     "!type": "fn()",
     "!doc": "<p>Callback when the user presses the down arrow button or presses\nthe down arrow key to open/close the drop down.\nToggle the drop-down widget; if it is up, close it, if not, open it</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/popup": {
   "!type": "fn()",
   "prototype": {
    "_beginZIndex": {
     "!doc": "<p>Z-index of the first popup.   (If first popup opens other\npopups they get a higher z-index.)</p>\n"
    },
    "_idGen": {},
    "_stack": {
     "!doc": "<p>Stack of currently popped up widgets.\n(someone opened _stack[0], and then it opened _stack[1], etc.)</p>\n"
    },
    "close": {
     "!type": "fn(popup?: +dijit._Widget)",
     "!doc": "<p>Close specified popup and any popups that it parented.\nIf no popup is specified, closes all popups.</p>\n"
    },
    "getTopPopup": {
     "!type": "fn()",
     "!doc": "<p>Compute the closest ancestor popup that&#39;s <em>not</em> a child of another popup.\nEx: For a TooltipDialog with a button that spawns a tree of menus, find the popup of the button.</p>\n"
    },
    "hide": {
     "!type": "fn(widget: +dijit._Widget)",
     "!doc": "<p>Hide this popup widget (until it is ready to be shown).\nInitialization for widgets that will be used as popups\n\n</p>\n<p>Also puts widget inside a wrapper DIV (if not already in one)\n\n</p>\n<p>If popup widget needs to layout it should\ndo so when it is made visible, and popup._onShow() is called.</p>\n"
    },
    "moveOffScreen": {
     "!type": "fn(widget: +dijit._Widget)",
     "!doc": "<p>Moves the popup widget off-screen.\nDo not use this method to hide popups when not in use, because\nthat will create an accessibility issue: the offscreen popup is\nstill in the tabbing order.</p>\n"
    },
    "open": {
     "!type": "fn(args: ?)",
     "!doc": "<p>Popup the widget at the specified position\n</p>\n"
    }
   }
  },
  "dijit/place": {
   "__Position": {},
   "__Rectangle": {},
   "around": {
    "!type": "fn(node: +Node, anchor: +Node|+dijit, positions: [string], leftToRight: bool, layoutNode?: fn())",
    "!doc": "<p>Position node adjacent or kitty-corner to anchor\nsuch that it&#39;s fully visible in viewport.</p>\n"
   },
   "at": {
    "!type": "fn(node: +Node, pos: +dijit, corners: [string], padding?: +dijit, layoutNode: ?)",
    "!doc": "<p>Positions node kitty-corner to the rectangle centered at (pos.x, pos.y) with width and height of\npadding.x <em> 2 and padding.y </em> 2, or zero if padding not specified.  Picks first corner in corners[]\nwhere node is fully visible, or the corner where it&#39;s most visible.\n\n</p>\n<p>Node is assumed to be absolutely or relatively positioned.</p>\n"
   }
  },
  "dijit/place.__Position": {
   "x": {
    "!doc": "<p>horizontal coordinate in pixels, relative to document body</p>\n"
   },
   "y": {
    "!doc": "<p>vertical coordinate in pixels, relative to document body</p>\n"
   }
  },
  "dijit/place.__Rectangle": {
   "h": {
    "!doc": "<p>height in pixels.   Can also be specified as &quot;height&quot; for backwards-compatibility.</p>\n"
   },
   "w": {
    "!doc": "<p>width in pixels.   Can also be specified as &quot;width&quot; for backwards-compatibility.</p>\n"
   },
   "x": {
    "!doc": "<p>horizontal offset in pixels, relative to document body</p>\n"
   },
   "y": {
    "!doc": "<p>vertical offset in pixels, relative to document body</p>\n"
   }
  },
  "dijit/Viewport": {},
  "dijit/form/Button": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/form/_FormWidget.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_onChangeActive": {
     "!doc": "<p>Indicates that changes to the value should call onChange() callback.\nThis is false during widget initialization, to avoid calling onChange()\nwhen the initial value is set.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIconClassAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTabIndexAttr": {},
    "_setTypeAttr": {},
    "_setValueAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "alt": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "aria-label": {},
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {},
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "disabled": {
     "!doc": "<p>Should this widget respond to user input?\nIn markup, this is specified as &quot;disabled=&#39;disabled&#39;&quot;, or just &quot;disabled&quot;.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "iconClass": {
     "!doc": "<p>Class to apply to DOMNode in button to make it display an icon</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "intermediateChanges": {
     "!doc": "<p>Fires onChange for each value change or only on demand</p>\n"
    },
    "label": {
     "!doc": "<p>Content to display in button.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "name": {
     "!doc": "<p>Name used when submitting form; same as &quot;name&quot; attribute or plain HTML elements</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "scrollOnFocus": {
     "!doc": "<p>On focus, should this widget scroll into view?</p>\n"
    },
    "searchContainerNode": {},
    "showLabel": {
     "!doc": "<p>Set this to true to hide the label text and display only the icon.\n(If showLabel=false then iconClass must be specified.)\nEspecially useful for toolbars.\nIf showLabel=true, the label will become the title (a.k.a. tooltip/hint) of the icon.\n\n</p>\n<p>The exception case is for computers in high-contrast mode, where the label\nwill still be displayed, since the icon doesn&#39;t appear.</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Order fields are traversed when user hits the tab key</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "type": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "value": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "compare": {
     "!type": "fn(val1: +anything, val2: +anything) -> number",
     "!doc": "<p>Compare 2 values (as returned by get(&#39;value&#39;) for this widget).</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Put focus on this widget</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "getValue": {
     "!type": "fn()",
     "!doc": "<p>Deprecated.  Use get(&#39;value&#39;) instead.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setDisabled": {
     "!type": "fn(disabled: bool)",
     "!doc": "<p>Deprecated.  Use set(&#39;disabled&#39;, ...) instead.</p>\n"
    },
    "setLabel": {
     "!type": "fn(content: string)",
     "!doc": "<p>Deprecated.  Use set(&#39;label&#39;, ...) instead.</p>\n"
    },
    "setValue": {
     "!type": "fn(value: string)",
     "!doc": "<p>Deprecated.  Use set(&#39;value&#39;, ...) instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/form/_FormWidget": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_Widget.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_onChangeActive": {
     "!doc": "<p>Indicates that changes to the value should call onChange() callback.\nThis is false during widget initialization, to avoid calling onChange()\nwhen the initial value is set.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTabIndexAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "alt": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "aria-label": {},
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {
     "!doc": "<p>Root CSS class of the widget (ex: dijitTextBox), used to construct CSS classes to indicate\nwidget state.</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "disabled": {
     "!doc": "<p>Should this widget respond to user input?\nIn markup, this is specified as &quot;disabled=&#39;disabled&#39;&quot;, or just &quot;disabled&quot;.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "intermediateChanges": {
     "!doc": "<p>Fires onChange for each value change or only on demand</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "name": {
     "!doc": "<p>Name used when submitting form; same as &quot;name&quot; attribute or plain HTML elements</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "scrollOnFocus": {
     "!doc": "<p>On focus, should this widget scroll into view?</p>\n"
    },
    "searchContainerNode": {},
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Order fields are traversed when user hits the tab key</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {
     "!doc": "<p>A string that represents the widget template.\nUse in conjunction with dojo.cache() to load from a file.</p>\n"
    },
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "type": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "value": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "compare": {
     "!type": "fn(val1: +anything, val2: +anything) -> number",
     "!doc": "<p>Compare 2 values (as returned by get(&#39;value&#39;) for this widget).</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Put focus on this widget</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "getValue": {
     "!type": "fn()",
     "!doc": "<p>Deprecated.  Use get(&#39;value&#39;) instead.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setDisabled": {
     "!type": "fn(disabled: bool)",
     "!doc": "<p>Deprecated.  Use set(&#39;disabled&#39;, ...) instead.</p>\n"
    },
    "setValue": {
     "!type": "fn(value: string)",
     "!doc": "<p>Deprecated.  Use set(&#39;value&#39;, ...) instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/form/_FormWidgetMixin": {
   "!type": "fn()",
   "prototype": {
    "_onChangeActive": {
     "!doc": "<p>Indicates that changes to the value should call onChange() callback.\nThis is false during widget initialization, to avoid calling onChange()\nwhen the initial value is set.</p>\n"
    },
    "_setIdAttr": {},
    "_setTabIndexAttr": {},
    "alt": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "aria-label": {},
    "disabled": {
     "!doc": "<p>Should this widget respond to user input?\nIn markup, this is specified as &quot;disabled=&#39;disabled&#39;&quot;, or just &quot;disabled&quot;.</p>\n"
    },
    "intermediateChanges": {
     "!doc": "<p>Fires onChange for each value change or only on demand</p>\n"
    },
    "name": {
     "!doc": "<p>Name used when submitting form; same as &quot;name&quot; attribute or plain HTML elements</p>\n"
    },
    "scrollOnFocus": {
     "!doc": "<p>On focus, should this widget scroll into view?</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Order fields are traversed when user hits the tab key</p>\n"
    },
    "type": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "value": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "compare": {
     "!type": "fn(val1: +anything, val2: +anything) -> number",
     "!doc": "<p>Compare 2 values (as returned by get(&#39;value&#39;) for this widget).</p>\n"
    },
    "create": {
     "!type": "fn()"
    },
    "destroy": {
     "!type": "fn()"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Put focus on this widget</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Tells if this widget is focusable or not.  Used internally by dijit.</p>\n"
    }
   }
  },
  "dijit/form/_ButtonMixin": {
   "!type": "fn()",
   "prototype": {
    "label": {
     "!doc": "<p>Content to display in button.</p>\n"
    },
    "type": {
     "!doc": "<p>Type of button (submit, reset, button, checkbox, radio)</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    }
   }
  },
  "dijit/_Container": {
   "!type": "fn()",
   "prototype": {
    "addChild": {
     "!type": "fn(widget: +dijit, insertIndex?: number)",
     "!doc": "<p>Makes the given widget a child of this widget.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "getIndexOfChild": {
     "!type": "fn(child: +dijit)",
     "!doc": "<p>Gets the index of the child in this container or -1 if not found</p>\n"
    },
    "hasChildren": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if widget has child widgets, i.e. if this.containerNode contains widgets.</p>\n"
    },
    "removeChild": {
     "!type": "fn(widget: +dijit._Widget|number)",
     "!doc": "<p>Removes the passed widget instance from this widget but does\nnot destroy it.  You can also pass in an integer indicating\nthe index within the container to remove (ie, removeChild(5) removes the sixth widget).</p>\n"
    }
   }
  },
  "dijit/_HasDropDown": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_FocusMixin.prototype",
    "_aroundNode": {
     "!doc": "<p>The node to display the popup around.\nCan be set via a data-dojo-attach-point assignment.\nIf missing, then domNode will be used.</p>\n"
    },
    "_arrowWrapperNode": {
     "!doc": "<p>Will set CSS class dijitUpArrow, dijitDownArrow, dijitRightArrow etc. on this node depending\non where the drop down is set to be positioned.\nCan be set via a data-dojo-attach-point assignment.\nIf missing, then _buttonNode will be used.</p>\n"
    },
    "_buttonNode": {
     "!doc": "<p>The button/icon/node to click to display the drop down.\nCan be set via a data-dojo-attach-point assignment.\nIf missing, then either focusNode or domNode (if focusNode is also missing) will be used.</p>\n"
    },
    "_focusManager": {},
    "_popupStateNode": {
     "!doc": "<p>The node to set the aria-expanded class on.\nAlso sets popupActive class but that will be removed in 2.0.\nCan be set via a data-dojo-attach-point assignment.\nIf missing, then focusNode or _buttonNode (if focusNode is missing) will be used.</p>\n"
    },
    "_stopClickEvents": {
     "!doc": "<p>When set to false, the click events will not be stopped, in\ncase you want to use them in your subclass</p>\n"
    },
    "autoWidth": {
     "!doc": "<p>Set to true to make the drop down at least as wide as this\nwidget.  Set to false if the drop down should just be its\ndefault width.</p>\n"
    },
    "dropDown": {
     "!doc": "<p>The widget to display as a popup.  This widget <em>must</em> be\ndefined before the startup function is called.</p>\n"
    },
    "dropDownPosition": {
     "!doc": "<p>This variable controls the position of the drop down.\nIt&#39;s an array of strings with the following values:\n\n</p>\n<ul>\n<li>before: places drop down to the left of the target node/widget, or to the right in\nthe case of RTL scripts like Hebrew and Arabic</li>\n<li>after: places drop down to the right of the target node/widget, or to the left in\nthe case of RTL scripts like Hebrew and Arabic</li>\n<li>above: drop down goes above target node</li>\n<li>below: drop down goes below target node</li>\n</ul>\n<p>The list is positions is tried, in order, until a position is found where the drop down fits\nwithin the viewport.\n</p>\n"
    },
    "forceWidth": {
     "!doc": "<p>Set to true to make the drop down exactly as wide as this\nwidget.  Overrides autoWidth.</p>\n"
    },
    "maxHeight": {
     "!doc": "<p>The max height for our dropdown.\nAny dropdown taller than this will have scrollbars.\nSet to 0 for no max height, or -1 to limit height to available space in viewport</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "closeDropDown": {
     "!type": "fn(focus: bool)",
     "!doc": "<p>Closes the drop down on this widget</p>\n"
    },
    "destroy": {
     "!type": "fn()"
    },
    "isLoaded": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if the dropdown exists and it&#39;s data is loaded.  This can\nbe overridden in order to force a call to loadDropDown().</p>\n"
    },
    "loadAndOpenDropDown": {
     "!type": "fn() -> +Deferred",
     "!doc": "<p>Creates the drop down if it doesn&#39;t exist, loads the data\nif there&#39;s an href and it hasn&#39;t been loaded yet, and\nthen opens the drop down.  This is basically a callback when the\nuser presses the down arrow button to open the drop down.</p>\n"
    },
    "loadDropDown": {
     "!type": "fn(loadCallback: fn())",
     "!doc": "<p>Creates the drop down if it doesn&#39;t exist, loads the data\nif there&#39;s an href and it hasn&#39;t been loaded yet, and then calls\nthe given callback.</p>\n"
    },
    "openDropDown": {
     "!type": "fn()",
     "!doc": "<p>Opens the dropdown for this widget.   To be called only when this.dropDown\nhas been created and is ready to display (ie, it&#39;s data is loaded).</p>\n"
    },
    "postCreate": {
     "!type": "fn()",
     "!doc": "<p>set up nodes and connect our mouse and keyboard events</p>\n"
    },
    "toggleDropDown": {
     "!type": "fn()",
     "!doc": "<p>Callback when the user presses the down arrow button or presses\nthe down arrow key to open/close the drop down.\nToggle the drop-down widget; if it is up, close it, if not, open it</p>\n"
    }
   }
  },
  "dijit/CheckedMenuItem": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/MenuItem.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIconClassAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "accelKey": {
     "!doc": "<p>Text for the accelerator (shortcut) key combination, a control, alt, etc. modified keystroke meant to\nexecute the menu item regardless of where the focus is on the page.\n\n</p>\n<p>Note that although Menu can display accelerator keys, there is no infrastructure to actually catch and\nexecute those accelerators.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {},
    "checked": {
     "!doc": "<p>Our checked state</p>\n"
    },
    "checkedChar": {
     "!doc": "<p>Character (or string) used in place of checkbox icon when display in high contrast mode</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "disabled": {
     "!doc": "<p>If true, the menu item is disabled.\nIf false, the menu item is enabled.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "iconClass": {},
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "label": {
     "!doc": "<p>Menu text as HTML</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "role": {},
    "searchContainerNode": {},
    "shortcutKey": {
     "!doc": "<p>Single character (underlined when the parent Menu is focused) used to navigate directly to this widget,\nalso known as <a href=\"http://en.wikipedia.org/wiki/Mnemonics_(keyboard%29\">a mnemonic</a>.\nThis is denoted in the label by surrounding the single character with {}.\nFor example, if label=&quot;{F}ile&quot;, then shortcutKey=&quot;F&quot;.</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Focus on this MenuItem</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getIndexInParent": {
     "!type": "fn() -> number",
     "!doc": "<p>Returns the index of this widget within its container parent.\nIt returns -1 if the parent does not exist, or if the parent\nis not a dijit/_Container</p>\n"
    },
    "getNextSibling": {
     "!type": "fn()",
     "!doc": "<p>Returns null if this is the last child of the parent,\notherwise returns the next element sibling to the &quot;right&quot;.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "getPreviousSibling": {
     "!type": "fn()",
     "!doc": "<p>Returns null if this is the first child of the parent,\notherwise returns the next element sibling to the &quot;left&quot;.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setDisabled": {
     "!type": "fn(disabled: bool)",
     "!doc": "<p>Deprecated.   Use set(&#39;disabled&#39;, bool) instead.</p>\n"
    },
    "setLabel": {
     "!type": "fn(content: string)",
     "!doc": "<p>Deprecated.   Use set(&#39;label&#39;, ...) instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/MenuItem": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_Widget.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIconClassAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "accelKey": {
     "!doc": "<p>Text for the accelerator (shortcut) key combination, a control, alt, etc. modified keystroke meant to\nexecute the menu item regardless of where the focus is on the page.\n\n</p>\n<p>Note that although Menu can display accelerator keys, there is no infrastructure to actually catch and\nexecute those accelerators.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {},
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "disabled": {
     "!doc": "<p>If true, the menu item is disabled.\nIf false, the menu item is enabled.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "iconClass": {
     "!doc": "<p>Class to apply to DOMNode to make it display an icon.</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "label": {
     "!doc": "<p>Menu text as HTML</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "searchContainerNode": {},
    "shortcutKey": {
     "!doc": "<p>Single character (underlined when the parent Menu is focused) used to navigate directly to this widget,\nalso known as <a href=\"http://en.wikipedia.org/wiki/Mnemonics_(keyboard%29\">a mnemonic</a>.\nThis is denoted in the label by surrounding the single character with {}.\nFor example, if label=&quot;{F}ile&quot;, then shortcutKey=&quot;F&quot;.</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Focus on this MenuItem</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getIndexInParent": {
     "!type": "fn() -> number",
     "!doc": "<p>Returns the index of this widget within its container parent.\nIt returns -1 if the parent does not exist, or if the parent\nis not a dijit/_Container</p>\n"
    },
    "getNextSibling": {
     "!type": "fn()",
     "!doc": "<p>Returns null if this is the last child of the parent,\notherwise returns the next element sibling to the &quot;right&quot;.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "getPreviousSibling": {
     "!type": "fn()",
     "!doc": "<p>Returns null if this is the first child of the parent,\notherwise returns the next element sibling to the &quot;left&quot;.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setDisabled": {
     "!type": "fn(disabled: bool)",
     "!doc": "<p>Deprecated.   Use set(&#39;disabled&#39;, bool) instead.</p>\n"
    },
    "setLabel": {
     "!type": "fn(content: string)",
     "!doc": "<p>Deprecated.   Use set(&#39;label&#39;, ...) instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/_Contained": {
   "!type": "fn()",
   "prototype": {
    "getIndexInParent": {
     "!type": "fn() -> number",
     "!doc": "<p>Returns the index of this widget within its container parent.\nIt returns -1 if the parent does not exist, or if the parent\nis not a dijit/_Container</p>\n"
    },
    "getNextSibling": {
     "!type": "fn()",
     "!doc": "<p>Returns null if this is the last child of the parent,\notherwise returns the next element sibling to the &quot;right&quot;.</p>\n"
    },
    "getPreviousSibling": {
     "!type": "fn()",
     "!doc": "<p>Returns null if this is the first child of the parent,\notherwise returns the next element sibling to the &quot;left&quot;.</p>\n"
    }
   }
  },
  "dijit/ColorPalette": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_Widget.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_currentFocus": {
     "!doc": "<p>The currently focused cell (if the palette itself has focus), or otherwise\nthe cell to be focused when the palette itself gets focus.\nDifferent from value, which represents the selected (i.e. clicked) cell.</p>\n"
    },
    "_focusManager": {},
    "_palettes": {
     "!doc": "<p>This represents the value of the colors.\nThe first level is a hashmap of the different palettes available.\nThe next two dimensions represent the columns and rows of colors.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_selectedCell": {
     "!doc": "<p>Index of the currently selected cell. Initially, none selected</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "_xDim": {
     "!doc": "<p>This is the number of cells horizontally across.</p>\n"
    },
    "_yDim": {
     "!doc": "<p>This is the number of cells vertically down.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {},
    "cellClass": {
     "!doc": "<p>CSS class applied to each cell in the palette</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "defaultTimeout": {
     "!doc": "<p>Number of milliseconds before a held key or button becomes typematic</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "dyeClass": {
     "!doc": "<p>Constructor for Object created for each cell of the palette.\ndyeClass should implement the dijit/_PaletteMixin.__Dye interface.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "palette": {
     "!doc": "<p>Size of grid, either &quot;7x10&quot; or &quot;3x4&quot;.</p>\n"
    },
    "searchContainerNode": {},
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Widget tab index.</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {
     "!doc": "<p>The template of this widget.</p>\n"
    },
    "timeoutChangeRate": {
     "!doc": "<p>Fraction of time used to change the typematic timer between events\n1.0 means that each typematic event fires at defaultTimeout intervals\nLess than 1.0 means that each typematic event fires at an increasing faster rate</p>\n"
    },
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "value": {
     "!doc": "<p>Currently selected color/emoticon/etc.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Focus this widget.  Puts focus on the most recently focused cell.</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/ColorPalette._Color": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dojo/_base/Color.prototype",
    "_imagePaths": {
     "!doc": "<p>This is stores the path to the palette images used for high-contrast mode display</p>\n"
    },
    "a": {},
    "b": {},
    "g": {},
    "hcTemplate": {},
    "r": {},
    "template": {},
    "fillCell": {
     "!type": "fn(cell: +Node, blankGif: string)"
    },
    "getValue": {
     "!type": "fn()",
     "!doc": "<p>Note that although dijit._Color is initialized with a value like &quot;white&quot; getValue() always\nreturns a hex value</p>\n"
    },
    "sanitize": {
     "!type": "fn()",
     "!doc": "<p>makes sure that the object has correct attributes</p>\n"
    },
    "setColor": {
     "!type": "fn(color: []|string|?) -> fn()",
     "!doc": "<p>Takes a named string, hex string, array of rgb or rgba values,\nan object with r, g, b, and a properties, or another <code>Color</code> object\nand sets this color instance to that value.\n</p>\n"
    },
    "toCss": {
     "!type": "fn(includeAlpha?: bool) -> string",
     "!doc": "<p>Returns a css color string in rgb(a) representation</p>\n"
    },
    "toHex": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a CSS color string in hexadecimal representation</p>\n"
    },
    "toRgb": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns 3 component array of rgb values</p>\n"
    },
    "toRgba": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns a 4 component array of rgba values from the color\nrepresented by this object.</p>\n"
    },
    "toString": {
     "!type": "fn()",
     "!doc": "<p>Returns a visual representation of the color</p>\n"
    }
   }
  },
  "dijit/_PaletteMixin": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_CssStateMixin.prototype",
    "_currentFocus": {
     "!doc": "<p>The currently focused cell (if the palette itself has focus), or otherwise\nthe cell to be focused when the palette itself gets focus.\nDifferent from value, which represents the selected (i.e. clicked) cell.</p>\n"
    },
    "_selectedCell": {
     "!doc": "<p>Index of the currently selected cell. Initially, none selected</p>\n"
    },
    "_xDim": {
     "!doc": "<p>This is the number of cells horizontally across.</p>\n"
    },
    "_yDim": {
     "!doc": "<p>This is the number of cells vertically down.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "cellClass": {
     "!doc": "<p>CSS class applied to each cell in the palette</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "defaultTimeout": {
     "!doc": "<p>Number of milliseconds before a held key or button becomes typematic</p>\n"
    },
    "dyeClass": {
     "!doc": "<p>Constructor for Object created for each cell of the palette.\ndyeClass should implement the dijit/_PaletteMixin.__Dye interface.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Widget tab index.</p>\n"
    },
    "timeoutChangeRate": {
     "!doc": "<p>Fraction of time used to change the typematic timer between events\n1.0 means that each typematic event fires at defaultTimeout intervals\nLess than 1.0 means that each typematic event fires at an increasing faster rate</p>\n"
    },
    "value": {
     "!doc": "<p>Currently selected color/emoticon/etc.</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Focus this widget.  Puts focus on the most recently focused cell.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    }
   }
  },
  "dijit/_PaletteMixin.__Dye": {
   "!type": "fn()",
   "prototype": {
    "fillCell": {
     "!type": "fn(cell: +Node, blankGif: string)",
     "!doc": "<p>Add cell DOMNode inner structure</p>\n"
    },
    "getValue": {
     "!type": "fn()",
     "!doc": "<p>Return &quot;value&quot; of cell; meaning of &quot;value&quot; varies by subclass.</p>\n"
    }
   }
  },
  "dijit/typematic": {
   "addKeyListener": {
    "!type": "fn(node: +Node, keyObject: ?, _this: ?, callback: fn(), subsequentDelay: number, initialDelay: number, minDelay?: number)",
    "!doc": "<p>Start listening for a specific typematic key.\nSee also the trigger method for other parameters.</p>\n"
   },
   "addListener": {
    "!type": "fn(mouseNode: +Node, keyNode: +Node, keyObject: ?, _this: ?, callback: fn(), subsequentDelay: number, initialDelay: number, minDelay?: number)",
    "!doc": "<p>Start listening for a specific typematic key and mouseclick.\nThis is a thin wrapper to addKeyListener and addMouseListener.\nSee the addMouseListener and addKeyListener methods for other parameters.</p>\n"
   },
   "addMouseListener": {
    "!type": "fn(node: +Node, _this: ?, callback: fn(), subsequentDelay: number, initialDelay: number, minDelay?: number)",
    "!doc": "<p>Start listening for a typematic mouse click.\nSee the trigger method for other parameters.</p>\n"
   },
   "stop": {
    "!type": "fn()",
    "!doc": "<p>Stop an ongoing timed, repeating callback sequence.</p>\n"
   },
   "trigger": {
    "!type": "fn(evt: +Event, _this: ?, node: +Node, callback: fn(), obj: ?, subsequentDelay?: number, initialDelay?: number, minDelay?: number)",
    "!doc": "<p>Start a timed, repeating callback sequence.\nIf already started, the function call is ignored.\nThis method is not normally called by the user but can be\nwhen the normal listener code is insufficient.</p>\n"
   }
  },
  "dijit/ConfirmDialog": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/Dialog.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_earlyTemplatedStartup": {
     "!doc": "<p>A fallback to preserve the 1.0 - 1.3 behavior of children in\ntemplates having their startup called before the parent widget\nfires postCreate. Defaults to &#39;false&#39;, causing child widgets to\nhave their .startup() called immediately before a parent widget\n.startup(), but always after the parent .postCreate(). Set to\n&#39;true&#39; to re-enable to previous, arguably broken, behavior.</p>\n"
    },
    "_firstFocusItem": {
     "!doc": "<p>The pointer to the first focusable node in the dialog.\nSet by <code>dijit/_DialogMixin._getFocusItems()</code>.</p>\n"
    },
    "_focusManager": {},
    "_lastFocusItem": {
     "!doc": "<p>The pointer to which node has focus prior to our dialog.\nSet by <code>dijit/_DialogMixin._getFocusItems()</code>.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setButtonCancelAttr": {},
    "_setButtonOkAttr": {},
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTitleAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "actionBarTemplate": {
     "!doc": "<p>HTML snippet to show the action bar (gray bar with OK/cancel buttons).\nBlank by default, but used by ConfirmDialog/ConfirmTooltipDialog subclasses.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "autofocus": {
     "!doc": "<p>A Toggle to modify the default focus behavior of a Dialog, which\nis to focus on the first dialog element after opening the dialog.\nFalse will disable autofocusing. Default: true</p>\n"
    },
    "baseClass": {},
    "buttonCancel": {
     "!doc": "<p>Label of cancel button</p>\n"
    },
    "buttonOk": {
     "!doc": "<p>Label of OK button</p>\n"
    },
    "class": {},
    "closable": {
     "!doc": "<p>Dialog show [x] icon to close itself, and ESC key will close the dialog.</p>\n"
    },
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "content": {
     "!doc": "<p>The innerHTML of the ContentPane.\nNote that the initialization parameter / argument to set(&quot;content&quot;, ...)\ncan be a String, DomNode, Nodelist, or _Widget.</p>\n"
    },
    "contextRequire": {
     "!doc": "<p>Used to provide a context require to the dojo/parser in order to be\nable to use relative MIDs (e.g. <code>./Widget</code>) in the widget&#39;s template.</p>\n"
    },
    "cssStateNodes": {},
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "doLayout": {
     "!doc": "<ul>\n<li>false - don&#39;t adjust size of children</li>\n<li>true - if there is a single visible child widget, set it&#39;s size to however big the ContentPane is</li>\n</ul>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "draggable": {
     "!doc": "<p>Toggles the movable aspect of the Dialog. If true, Dialog\ncan be dragged by it&#39;s title. If false it will remain centered\nin the viewport.</p>\n"
    },
    "duration": {
     "!doc": "<p>The time in milliseconds it takes the dialog to fade in and out</p>\n"
    },
    "errorMessage": {
     "!doc": "<p>Message that shows if an error occurs</p>\n"
    },
    "extractContent": {
     "!doc": "<p>Extract visible content from inside of <code>&lt;body&gt; .... &lt;/body&gt;</code>.\nI.e., strip <code>&lt;html&gt;</code> and <code>&lt;head&gt;</code> (and it&#39;s contents) from the href</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "href": {
     "!doc": "<p>The href of the content that displays now.\nSet this at construction if you want to load data externally when the\npane is shown.  (Set preload=true to load it immediately.)\nChanging href after creation doesn&#39;t have any effect; Use set(&#39;href&#39;, ...);</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "ioArgs": {
     "!doc": "<p>Parameters to pass to xhrGet() request, for example:\n\n</p>\n<pre><code>&lt;div data-dojo-type=&quot;dijit/layout/ContentPane&quot; data-dojo-props=&quot;href: &#39;./bar&#39;, ioArgs: {timeout: 500}&quot;&gt;</code></pre>\n"
    },
    "isLayoutContainer": {
     "!doc": "<p>Indicates that this widget will call resize() on it&#39;s child widgets\nwhen they become visible.</p>\n"
    },
    "isLoaded": {
     "!doc": "<p>True if the ContentPane has data in it, either specified\nduring initialization (via href or inline content), or set\nvia set(&#39;content&#39;, ...) / set(&#39;href&#39;, ...)\n\n</p>\n<p>False if it doesn&#39;t have any content, or if ContentPane is\nstill in the process of downloading href.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "loadingMessage": {
     "!doc": "<p>Message that shows while downloading</p>\n"
    },
    "maxRatio": {
     "!doc": "<p>Maximum size to allow the dialog to expand to, relative to viewport size</p>\n"
    },
    "onLoadDeferred": {
     "!doc": "<p>This is the <code>dojo.Deferred</code> returned by set(&#39;href&#39;, ...) and refresh().\nCalling onLoadDeferred.then() registers your\ncallback to be called only once, when the prior set(&#39;href&#39;, ...) call or\nthe initial href parameter to the constructor finishes loading.\n\n</p>\n<p>This is different than an onLoad() handler which gets called any time any href\nor content is loaded.</p>\n"
    },
    "open": {
     "!doc": "<p>True if Dialog is currently displayed on screen.</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "parseOnLoad": {
     "!doc": "<p>Parse content and create the widgets, if any.</p>\n"
    },
    "parserScope": {
     "!doc": "<p>Flag passed to parser.  Root for attribute names to search for.   If scopeName is dojo,\nwill search for data-dojo-type (or dojoType).  For backwards compatibility\nreasons defaults to dojo._scopeName (which is &quot;dojo&quot; except when\nmulti-version support is used, when it will be something like dojo16, dojo20, etc.)</p>\n"
    },
    "preload": {
     "!doc": "<p>Force load of data on initialization even if pane is hidden.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Prevent caching of data from href&#39;s by appending a timestamp to the href.</p>\n"
    },
    "refocus": {
     "!doc": "<p>A Toggle to modify the default focus behavior of a Dialog, which\nis to re-focus the element which had focus before being opened.\nFalse will disable refocusing. Default: true</p>\n"
    },
    "refreshOnShow": {
     "!doc": "<p>Refresh (re-download) content when pane goes from hidden to shown</p>\n"
    },
    "searchContainerNode": {},
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "state": {
     "!doc": "<p>Will be &quot;Error&quot; if one or more of the child widgets has an invalid value,\n&quot;Incomplete&quot; if not all of the required child widgets are filled in.  Otherwise, &quot;&quot;,\nwhich indicates that the form is ready to be submitted.</p>\n"
    },
    "stopParser": {},
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "template": {
     "!doc": "<p>Flag from the parser that this ContentPane is inside a template\nso the contents are pre-parsed.</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "widgetsInTemplate": {
     "!doc": "<p>Should we parse the template to find widgets that might be\ndeclared in markup inside it?  (Remove for 2.0 and assume true)</p>\n"
    },
    "addChild": {
     "!type": "fn(widget: +dijit, insertIndex?: number)",
     "!doc": "<p>Makes the given widget a child of this widget.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "cancel": {
     "!type": "fn()",
     "!doc": "<p>Cancels an in-flight download of content</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "connectChildren": {
     "!type": "fn(inStartup: bool)",
     "!doc": "<p>You can call this function directly, ex. in the event that you\nprogrammatically add a widget to the form <em>after</em> the form has been\ninitialized.</p>\n"
    },
    "create": {
     "!type": "fn(params: ?, srcNodeRef: ?)"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn()"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy all the widgets inside the ContentPane and empty containerNode</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy the ContentPane and its contents</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "disconnectChildren": {
     "!type": "fn()",
     "!doc": "<p>Deprecated method.   Applications no longer need to call this.   Remove for 2.0.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "execute": {
     "!type": "fn(formContents: ?)",
     "!doc": "<p>Callback when the user hits the submit button.\nOverride this method to handle Dialog execution.</p>\n"
    },
    "focus": {
     "!type": "fn()"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getIndexOfChild": {
     "!type": "fn(child: +dijit)",
     "!doc": "<p>Gets the index of the child in this container or -1 if not found</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "getValues": {
     "!type": "fn()"
    },
    "hasChildren": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if widget has child widgets, i.e. if this.containerNode contains widgets.</p>\n"
    },
    "hide": {
     "!type": "fn() -> +dojo",
     "!doc": "<p>Hide the dialog</p>\n"
    },
    "ioMethod": {
     "!type": "fn(args: ?)",
     "!doc": "<p>Function that should grab the content specified via href.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "isValid": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if all of the widgets are valid.\nDeprecated, will be removed in 2.0.  Use get(&quot;state&quot;) instead.</p>\n"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, ctor: ?)"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()"
    },
    "refresh": {
     "!type": "fn()",
     "!doc": "<p>[Re]download contents of href and display</p>\n"
    },
    "removeChild": {
     "!type": "fn(widget: +dijit._Widget|number)",
     "!doc": "<p>Removes the passed widget instance from this widget but does\nnot destroy it.  You can also pass in an integer indicating\nthe index within the container to remove (ie, removeChild(5) removes the sixth widget).</p>\n"
    },
    "reset": {
     "!type": "fn()"
    },
    "resize": {
     "!type": "fn(changeSize: ?, resultSize: ?)",
     "!doc": "<p>See <code>dijit/layout/_LayoutWidget.resize()</code> for description.\nAlthough ContentPane doesn&#39;t extend _LayoutWidget, it does implement\nthe same API.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setContent": {
     "!type": "fn(data: string|+Node|+Nodelist)",
     "!doc": "<p>Deprecated.   Use set(&#39;content&#39;, ...) instead.</p>\n"
    },
    "setHref": {
     "!type": "fn(href: string|+Uri)",
     "!doc": "<p>Deprecated.   Use set(&#39;href&#39;, ...) instead.</p>\n"
    },
    "setValues": {
     "!type": "fn(val: ?)"
    },
    "show": {
     "!type": "fn() -> +dojo",
     "!doc": "<p>Display the dialog</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Call startup() on all children including non _Widget ones like dojo/dnd/Source objects</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "validate": {
     "!type": "fn()",
     "!doc": "<p>returns if the form is valid - same as isValid - but\nprovides a few additional (ui-specific) features:\n\n</p>\n<ol>\n<li>it will highlight any sub-widgets that are not valid</li>\n<li>it will call focus() on the first invalid sub-widget</li>\n</ol>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/Dialog": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/layout/ContentPane.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_firstFocusItem": {
     "!doc": "<p>The pointer to the first focusable node in the dialog.\nSet by <code>dijit/_DialogMixin._getFocusItems()</code>.</p>\n"
    },
    "_focusManager": {},
    "_lastFocusItem": {
     "!doc": "<p>The pointer to which node has focus prior to our dialog.\nSet by <code>dijit/_DialogMixin._getFocusItems()</code>.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTitleAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "actionBarTemplate": {
     "!doc": "<p>HTML snippet to show the action bar (gray bar with OK/cancel buttons).\nBlank by default, but used by ConfirmDialog/ConfirmTooltipDialog subclasses.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "autofocus": {
     "!doc": "<p>A Toggle to modify the default focus behavior of a Dialog, which\nis to focus on the first dialog element after opening the dialog.\nFalse will disable autofocusing. Default: true</p>\n"
    },
    "baseClass": {},
    "class": {},
    "closable": {
     "!doc": "<p>Dialog show [x] icon to close itself, and ESC key will close the dialog.</p>\n"
    },
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "content": {
     "!doc": "<p>The innerHTML of the ContentPane.\nNote that the initialization parameter / argument to set(&quot;content&quot;, ...)\ncan be a String, DomNode, Nodelist, or _Widget.</p>\n"
    },
    "cssStateNodes": {},
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "doLayout": {
     "!doc": "<ul>\n<li>false - don&#39;t adjust size of children</li>\n<li>true - if there is a single visible child widget, set it&#39;s size to however big the ContentPane is</li>\n</ul>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "draggable": {
     "!doc": "<p>Toggles the movable aspect of the Dialog. If true, Dialog\ncan be dragged by it&#39;s title. If false it will remain centered\nin the viewport.</p>\n"
    },
    "duration": {
     "!doc": "<p>The time in milliseconds it takes the dialog to fade in and out</p>\n"
    },
    "errorMessage": {
     "!doc": "<p>Message that shows if an error occurs</p>\n"
    },
    "extractContent": {
     "!doc": "<p>Extract visible content from inside of <code>&lt;body&gt; .... &lt;/body&gt;</code>.\nI.e., strip <code>&lt;html&gt;</code> and <code>&lt;head&gt;</code> (and it&#39;s contents) from the href</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "href": {
     "!doc": "<p>The href of the content that displays now.\nSet this at construction if you want to load data externally when the\npane is shown.  (Set preload=true to load it immediately.)\nChanging href after creation doesn&#39;t have any effect; Use set(&#39;href&#39;, ...);</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "ioArgs": {
     "!doc": "<p>Parameters to pass to xhrGet() request, for example:\n\n</p>\n<pre><code>&lt;div data-dojo-type=&quot;dijit/layout/ContentPane&quot; data-dojo-props=&quot;href: &#39;./bar&#39;, ioArgs: {timeout: 500}&quot;&gt;</code></pre>\n"
    },
    "isLayoutContainer": {
     "!doc": "<p>Indicates that this widget will call resize() on it&#39;s child widgets\nwhen they become visible.</p>\n"
    },
    "isLoaded": {
     "!doc": "<p>True if the ContentPane has data in it, either specified\nduring initialization (via href or inline content), or set\nvia set(&#39;content&#39;, ...) / set(&#39;href&#39;, ...)\n\n</p>\n<p>False if it doesn&#39;t have any content, or if ContentPane is\nstill in the process of downloading href.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "loadingMessage": {
     "!doc": "<p>Message that shows while downloading</p>\n"
    },
    "maxRatio": {
     "!doc": "<p>Maximum size to allow the dialog to expand to, relative to viewport size</p>\n"
    },
    "onLoadDeferred": {
     "!doc": "<p>This is the <code>dojo.Deferred</code> returned by set(&#39;href&#39;, ...) and refresh().\nCalling onLoadDeferred.then() registers your\ncallback to be called only once, when the prior set(&#39;href&#39;, ...) call or\nthe initial href parameter to the constructor finishes loading.\n\n</p>\n<p>This is different than an onLoad() handler which gets called any time any href\nor content is loaded.</p>\n"
    },
    "open": {
     "!doc": "<p>True if Dialog is currently displayed on screen.</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "parseOnLoad": {
     "!doc": "<p>Parse content and create the widgets, if any.</p>\n"
    },
    "parserScope": {
     "!doc": "<p>Flag passed to parser.  Root for attribute names to search for.   If scopeName is dojo,\nwill search for data-dojo-type (or dojoType).  For backwards compatibility\nreasons defaults to dojo._scopeName (which is &quot;dojo&quot; except when\nmulti-version support is used, when it will be something like dojo16, dojo20, etc.)</p>\n"
    },
    "preload": {
     "!doc": "<p>Force load of data on initialization even if pane is hidden.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Prevent caching of data from href&#39;s by appending a timestamp to the href.</p>\n"
    },
    "refocus": {
     "!doc": "<p>A Toggle to modify the default focus behavior of a Dialog, which\nis to re-focus the element which had focus before being opened.\nFalse will disable refocusing. Default: true</p>\n"
    },
    "refreshOnShow": {
     "!doc": "<p>Refresh (re-download) content when pane goes from hidden to shown</p>\n"
    },
    "searchContainerNode": {},
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "state": {
     "!doc": "<p>Will be &quot;Error&quot; if one or more of the child widgets has an invalid value,\n&quot;Incomplete&quot; if not all of the required child widgets are filled in.  Otherwise, &quot;&quot;,\nwhich indicates that the form is ready to be submitted.</p>\n"
    },
    "stopParser": {},
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "template": {
     "!doc": "<p>Flag from the parser that this ContentPane is inside a template\nso the contents are pre-parsed.</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "addChild": {
     "!type": "fn(widget: +dijit, insertIndex?: number)",
     "!doc": "<p>Makes the given widget a child of this widget.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "cancel": {
     "!type": "fn()",
     "!doc": "<p>Cancels an in-flight download of content</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "connectChildren": {
     "!type": "fn(inStartup: bool)",
     "!doc": "<p>You can call this function directly, ex. in the event that you\nprogrammatically add a widget to the form <em>after</em> the form has been\ninitialized.</p>\n"
    },
    "create": {
     "!type": "fn(params: ?, srcNodeRef: ?)"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn()"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy all the widgets inside the ContentPane and empty containerNode</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy the ContentPane and its contents</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "disconnectChildren": {
     "!type": "fn()",
     "!doc": "<p>Deprecated method.   Applications no longer need to call this.   Remove for 2.0.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "execute": {
     "!type": "fn(formContents: ?)",
     "!doc": "<p>Callback when the user hits the submit button.\nOverride this method to handle Dialog execution.</p>\n"
    },
    "focus": {
     "!type": "fn()"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getIndexOfChild": {
     "!type": "fn(child: +dijit)",
     "!doc": "<p>Gets the index of the child in this container or -1 if not found</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "getValues": {
     "!type": "fn()"
    },
    "hasChildren": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if widget has child widgets, i.e. if this.containerNode contains widgets.</p>\n"
    },
    "hide": {
     "!type": "fn() -> +dojo",
     "!doc": "<p>Hide the dialog</p>\n"
    },
    "ioMethod": {
     "!type": "fn(args: ?)",
     "!doc": "<p>Function that should grab the content specified via href.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "isValid": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if all of the widgets are valid.\nDeprecated, will be removed in 2.0.  Use get(&quot;state&quot;) instead.</p>\n"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, ctor: ?)"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()"
    },
    "refresh": {
     "!type": "fn()",
     "!doc": "<p>[Re]download contents of href and display</p>\n"
    },
    "removeChild": {
     "!type": "fn(widget: +dijit._Widget|number)",
     "!doc": "<p>Removes the passed widget instance from this widget but does\nnot destroy it.  You can also pass in an integer indicating\nthe index within the container to remove (ie, removeChild(5) removes the sixth widget).</p>\n"
    },
    "reset": {
     "!type": "fn()"
    },
    "resize": {
     "!type": "fn(changeSize: ?, resultSize: ?)",
     "!doc": "<p>See <code>dijit/layout/_LayoutWidget.resize()</code> for description.\nAlthough ContentPane doesn&#39;t extend _LayoutWidget, it does implement\nthe same API.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setContent": {
     "!type": "fn(data: string|+Node|+Nodelist)",
     "!doc": "<p>Deprecated.   Use set(&#39;content&#39;, ...) instead.</p>\n"
    },
    "setHref": {
     "!type": "fn(href: string|+Uri)",
     "!doc": "<p>Deprecated.   Use set(&#39;href&#39;, ...) instead.</p>\n"
    },
    "setValues": {
     "!type": "fn(val: ?)"
    },
    "show": {
     "!type": "fn() -> +dojo",
     "!doc": "<p>Display the dialog</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Call startup() on all children including non _Widget ones like dojo/dnd/Source objects</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "validate": {
     "!type": "fn()",
     "!doc": "<p>returns if the form is valid - same as isValid - but\nprovides a few additional (ui-specific) features:\n\n</p>\n<ol>\n<li>it will highlight any sub-widgets that are not valid</li>\n<li>it will call focus() on the first invalid sub-widget</li>\n</ol>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   },
   "_DialogLevelManager": {
    "!doc": "<p>Controls the various active &quot;levels&quot; on the page, starting with the\nstuff initially visible on the page (at z-index 0), and then having an entry for\neach Dialog shown.</p>\n"
   },
   "_dialogStack": {}
  },
  "dijit/Dialog._DialogBase": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_TemplatedMixin.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_firstFocusItem": {
     "!doc": "<p>The pointer to the first focusable node in the dialog.\nSet by <code>dijit/_DialogMixin._getFocusItems()</code>.</p>\n"
    },
    "_lastFocusItem": {
     "!doc": "<p>The pointer to which node has focus prior to our dialog.\nSet by <code>dijit/_DialogMixin._getFocusItems()</code>.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_skipNodeCache": {},
    "actionBarTemplate": {
     "!doc": "<p>HTML snippet to show the action bar (gray bar with OK/cancel buttons).\nBlank by default, but used by ConfirmDialog/ConfirmTooltipDialog subclasses.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "autofocus": {
     "!doc": "<p>A Toggle to modify the default focus behavior of a Dialog, which\nis to focus on the first dialog element after opening the dialog.\nFalse will disable autofocusing. Default: true</p>\n"
    },
    "baseClass": {},
    "closable": {
     "!doc": "<p>Dialog show [x] icon to close itself, and ESC key will close the dialog.</p>\n"
    },
    "cssStateNodes": {},
    "draggable": {
     "!doc": "<p>Toggles the movable aspect of the Dialog. If true, Dialog\ncan be dragged by it&#39;s title. If false it will remain centered\nin the viewport.</p>\n"
    },
    "duration": {
     "!doc": "<p>The time in milliseconds it takes the dialog to fade in and out</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "maxRatio": {
     "!doc": "<p>Maximum size to allow the dialog to expand to, relative to viewport size</p>\n"
    },
    "open": {
     "!doc": "<p>True if Dialog is currently displayed on screen.</p>\n"
    },
    "refocus": {
     "!doc": "<p>A Toggle to modify the default focus behavior of a Dialog, which\nis to re-focus the element which had focus before being opened.\nFalse will disable refocusing. Default: true</p>\n"
    },
    "searchContainerNode": {},
    "state": {
     "!doc": "<p>Will be &quot;Error&quot; if one or more of the child widgets has an invalid value,\n&quot;Incomplete&quot; if not all of the required child widgets are filled in.  Otherwise, &quot;&quot;,\nwhich indicates that the form is ready to be submitted.</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget from a template, setting this.domNode.</p>\n"
    },
    "connectChildren": {
     "!type": "fn(inStartup: bool)",
     "!doc": "<p>You can call this function directly, ex. in the event that you\nprogrammatically add a widget to the form <em>after</em> the form has been\ninitialized.</p>\n"
    },
    "destroy": {
     "!type": "fn()"
    },
    "destroyRendering": {
     "!type": "fn()"
    },
    "disconnectChildren": {
     "!type": "fn()",
     "!doc": "<p>Deprecated method.   Applications no longer need to call this.   Remove for 2.0.</p>\n"
    },
    "execute": {
     "!type": "fn(formContents: ?)",
     "!doc": "<p>Callback when the user hits the submit button.\nOverride this method to handle Dialog execution.</p>\n"
    },
    "focus": {
     "!type": "fn()"
    },
    "getValues": {
     "!type": "fn()"
    },
    "hide": {
     "!type": "fn() -> +dojo",
     "!doc": "<p>Hide the dialog</p>\n"
    },
    "isValid": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if all of the widgets are valid.\nDeprecated, will be removed in 2.0.  Use get(&quot;state&quot;) instead.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()"
    },
    "reset": {
     "!type": "fn()"
    },
    "resize": {
     "!type": "fn(dim?: ?)",
     "!doc": "<p>Called with no argument when viewport scrolled or viewport size changed.  Adjusts Dialog as\nnecessary to keep it visible.\n\n</p>\n<p>Can also be called with an argument (by dojox/layout/ResizeHandle etc.) to explicitly set the\nsize of the dialog.</p>\n"
    },
    "setValues": {
     "!type": "fn(val: ?)"
    },
    "show": {
     "!type": "fn() -> +dojo",
     "!doc": "<p>Display the dialog</p>\n"
    },
    "startup": {
     "!type": "fn()"
    },
    "validate": {
     "!type": "fn()",
     "!doc": "<p>returns if the form is valid - same as isValid - but\nprovides a few additional (ui-specific) features:\n\n</p>\n<ol>\n<li>it will highlight any sub-widgets that are not valid</li>\n<li>it will call focus() on the first invalid sub-widget</li>\n</ol>\n"
    }
   }
  },
  "dijit/Dialog._DialogLevelManager": {
   "_beginZIndex": {},
   "hide": {
    "!type": "fn(dialog: +dijit)",
    "!doc": "<p>Called when the specified dialog is hidden/destroyed, after the fade-out\nanimation ends, in order to reset page focus, fix the underlay, etc.\nIf the specified dialog isn&#39;t open then does nothing.\n\n</p>\n<p>Caller is responsible for either setting display:none on the dialog domNode,\nor calling dijit/popup.hide(), or removing it from the page DOM.</p>\n"
   },
   "isTop": {
    "!type": "fn(dialog: +dijit) -> bool",
    "!doc": "<p>Returns true if specified Dialog is the top in the task</p>\n"
   },
   "show": {
    "!type": "fn(dialog: +dijit, underlayAttrs: ?)",
    "!doc": "<p>Call right before fade-in animation for new dialog.\nSaves current focus, displays/adjusts underlay for new dialog,\nand sets the z-index of the dialog itself.\n\n</p>\n<p>New dialog will be displayed on top of all currently displayed dialogs.\n\n</p>\n<p>Caller is responsible for setting focus in new dialog after the fade-in\nanimation completes.</p>\n"
   }
  },
  "dijit/_base/manager": {
   "defaultDuration": {
    "!doc": "<p>The default fx.animation speed (in ms) to use for all Dijit\ntransitional fx.animations, unless otherwise specified\non a per-instance basis. Defaults to 200, overrided by\n<code>djConfig.defaultDuration</code></p>\n"
   }
  },
  "dijit/form/_FormMixin": {
   "!type": "fn()",
   "prototype": {
    "state": {
     "!doc": "<p>Will be &quot;Error&quot; if one or more of the child widgets has an invalid value,\n&quot;Incomplete&quot; if not all of the required child widgets are filled in.  Otherwise, &quot;&quot;,\nwhich indicates that the form is ready to be submitted.</p>\n"
    },
    "connectChildren": {
     "!type": "fn(inStartup: bool)",
     "!doc": "<p>You can call this function directly, ex. in the event that you\nprogrammatically add a widget to the form <em>after</em> the form has been\ninitialized.</p>\n"
    },
    "destroy": {
     "!type": "fn()"
    },
    "disconnectChildren": {
     "!type": "fn()",
     "!doc": "<p>Deprecated method.   Applications no longer need to call this.   Remove for 2.0.</p>\n"
    },
    "getValues": {
     "!type": "fn()"
    },
    "isValid": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if all of the widgets are valid.\nDeprecated, will be removed in 2.0.  Use get(&quot;state&quot;) instead.</p>\n"
    },
    "reset": {
     "!type": "fn()"
    },
    "setValues": {
     "!type": "fn(val: ?)"
    },
    "startup": {
     "!type": "fn()"
    },
    "validate": {
     "!type": "fn()",
     "!doc": "<p>returns if the form is valid - same as isValid - but\nprovides a few additional (ui-specific) features:\n\n</p>\n<ol>\n<li>it will highlight any sub-widgets that are not valid</li>\n<li>it will call focus() on the first invalid sub-widget</li>\n</ol>\n"
    }
   }
  },
  "dijit/_DialogMixin": {
   "!type": "fn()",
   "prototype": {
    "actionBarTemplate": {
     "!doc": "<p>HTML snippet to show the action bar (gray bar with OK/cancel buttons).\nBlank by default, but used by ConfirmDialog/ConfirmTooltipDialog subclasses.</p>\n"
    },
    "execute": {
     "!type": "fn(formContents: ?)",
     "!doc": "<p>Callback when the user hits the submit button.\nOverride this method to handle Dialog execution.</p>\n"
    }
   }
  },
  "dijit/DialogUnderlay": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_Widget.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_modalConnects": {},
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {
     "!doc": "<p>Root CSS class of the widget (ex: dijitTextBox), used to construct CSS classes to indicate\nwidget state.</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "dialogId": {
     "!doc": "<p>Id of the dialog.... DialogUnderlay&#39;s id is based on this id</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "searchContainerNode": {},
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn()"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "hide": {
     "!type": "fn()",
     "!doc": "<p>Hides the dialog underlay</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "show": {
     "!type": "fn()",
     "!doc": "<p>Show the dialog underlay</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   },
   "_singleton": {},
   "hide": {
    "!type": "fn()",
    "!doc": "<p>Hide the underlay.</p>\n"
   },
   "show": {
    "!type": "fn(attrs: ?, zIndex: number)",
    "!doc": "<p>Display the underlay with the given attributes set.  If the underlay is already displayed,\nthen adjust it&#39;s attributes as specified.</p>\n"
   }
  },
  "dijit/layout/ContentPane": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_Widget.prototype",
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTitleAttr": {},
    "_setTypeAttr": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {},
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "content": {
     "!doc": "<p>The innerHTML of the ContentPane.\nNote that the initialization parameter / argument to set(&quot;content&quot;, ...)\ncan be a String, DomNode, Nodelist, or _Widget.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "doLayout": {
     "!doc": "<ul>\n<li>false - don&#39;t adjust size of children</li>\n<li>true - if there is a single visible child widget, set it&#39;s size to however big the ContentPane is</li>\n</ul>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "errorMessage": {
     "!doc": "<p>Message that shows if an error occurs</p>\n"
    },
    "extractContent": {
     "!doc": "<p>Extract visible content from inside of <code>&lt;body&gt; .... &lt;/body&gt;</code>.\nI.e., strip <code>&lt;html&gt;</code> and <code>&lt;head&gt;</code> (and it&#39;s contents) from the href</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "href": {
     "!doc": "<p>The href of the content that displays now.\nSet this at construction if you want to load data externally when the\npane is shown.  (Set preload=true to load it immediately.)\nChanging href after creation doesn&#39;t have any effect; Use set(&#39;href&#39;, ...);</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "ioArgs": {
     "!doc": "<p>Parameters to pass to xhrGet() request, for example:\n\n</p>\n<pre><code>&lt;div data-dojo-type=&quot;dijit/layout/ContentPane&quot; data-dojo-props=&quot;href: &#39;./bar&#39;, ioArgs: {timeout: 500}&quot;&gt;</code></pre>\n"
    },
    "isLayoutContainer": {
     "!doc": "<p>Indicates that this widget will call resize() on it&#39;s child widgets\nwhen they become visible.</p>\n"
    },
    "isLoaded": {
     "!doc": "<p>True if the ContentPane has data in it, either specified\nduring initialization (via href or inline content), or set\nvia set(&#39;content&#39;, ...) / set(&#39;href&#39;, ...)\n\n</p>\n<p>False if it doesn&#39;t have any content, or if ContentPane is\nstill in the process of downloading href.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "loadingMessage": {
     "!doc": "<p>Message that shows while downloading</p>\n"
    },
    "onLoadDeferred": {
     "!doc": "<p>This is the <code>dojo.Deferred</code> returned by set(&#39;href&#39;, ...) and refresh().\nCalling onLoadDeferred.then() registers your\ncallback to be called only once, when the prior set(&#39;href&#39;, ...) call or\nthe initial href parameter to the constructor finishes loading.\n\n</p>\n<p>This is different than an onLoad() handler which gets called any time any href\nor content is loaded.</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "parseOnLoad": {
     "!doc": "<p>Parse content and create the widgets, if any.</p>\n"
    },
    "parserScope": {
     "!doc": "<p>Flag passed to parser.  Root for attribute names to search for.   If scopeName is dojo,\nwill search for data-dojo-type (or dojoType).  For backwards compatibility\nreasons defaults to dojo._scopeName (which is &quot;dojo&quot; except when\nmulti-version support is used, when it will be something like dojo16, dojo20, etc.)</p>\n"
    },
    "preload": {
     "!doc": "<p>Force load of data on initialization even if pane is hidden.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Prevent caching of data from href&#39;s by appending a timestamp to the href.</p>\n"
    },
    "refreshOnShow": {
     "!doc": "<p>Refresh (re-download) content when pane goes from hidden to shown</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "stopParser": {},
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "template": {
     "!doc": "<p>Flag from the parser that this ContentPane is inside a template\nso the contents are pre-parsed.</p>\n"
    },
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "addChild": {
     "!type": "fn(widget: +dijit, insertIndex?: number)",
     "!doc": "<p>Makes the given widget a child of this widget.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "cancel": {
     "!type": "fn()",
     "!doc": "<p>Cancels an in-flight download of content</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "create": {
     "!type": "fn(params: ?, srcNodeRef: ?)"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn()"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy all the widgets inside the ContentPane and empty containerNode</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy the ContentPane and its contents</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getIndexOfChild": {
     "!type": "fn(child: +dijit)",
     "!doc": "<p>Gets the index of the child in this container or -1 if not found</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "hasChildren": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if widget has child widgets, i.e. if this.containerNode contains widgets.</p>\n"
    },
    "ioMethod": {
     "!type": "fn(args: ?)",
     "!doc": "<p>Function that should grab the content specified via href.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, ctor: ?)"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()"
    },
    "refresh": {
     "!type": "fn()",
     "!doc": "<p>[Re]download contents of href and display</p>\n"
    },
    "removeChild": {
     "!type": "fn(widget: +dijit._Widget|number)",
     "!doc": "<p>Removes the passed widget instance from this widget but does\nnot destroy it.  You can also pass in an integer indicating\nthe index within the container to remove (ie, removeChild(5) removes the sixth widget).</p>\n"
    },
    "resize": {
     "!type": "fn(changeSize: ?, resultSize: ?)",
     "!doc": "<p>See <code>dijit/layout/_LayoutWidget.resize()</code> for description.\nAlthough ContentPane doesn&#39;t extend _LayoutWidget, it does implement\nthe same API.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setContent": {
     "!type": "fn(data: string|+Node|+Nodelist)",
     "!doc": "<p>Deprecated.   Use set(&#39;content&#39;, ...) instead.</p>\n"
    },
    "setHref": {
     "!type": "fn(href: string|+Uri)",
     "!doc": "<p>Deprecated.   Use set(&#39;href&#39;, ...) instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Call startup() on all children including non _Widget ones like dojo/dnd/Source objects</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/layout/_ContentPaneResizeMixin": {
   "!type": "fn()",
   "prototype": {
    "doLayout": {
     "!doc": "<ul>\n<li>false - don&#39;t adjust size of children</li>\n<li>true - if there is a single visible child widget, set it&#39;s size to however big the ContentPane is</li>\n</ul>\n"
    },
    "isLayoutContainer": {
     "!doc": "<p>Indicates that this widget will call resize() on it&#39;s child widgets\nwhen they become visible.</p>\n"
    },
    "resize": {
     "!type": "fn(changeSize: ?, resultSize: ?)",
     "!doc": "<p>See <code>dijit/layout/_LayoutWidget.resize()</code> for description.\nAlthough ContentPane doesn&#39;t extend _LayoutWidget, it does implement\nthe same API.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>See <code>dijit/layout/_LayoutWidget.startup()</code> for description.\nAlthough ContentPane doesn&#39;t extend _LayoutWidget, it does implement\nthe same API.</p>\n"
    }
   }
  },
  "dijit/layout/utils": {
   "layoutChildren": {
    "!type": "fn(container: +Node, dim: ?, children: [+Widget], changedRegionId?: string, changedRegionSize?: number)",
    "!doc": "<p>Layout a bunch of child dom nodes within a parent dom node</p>\n"
   },
   "marginBox2contentBox": {
    "!type": "fn(node: +Node, mb: ?)",
    "!doc": "<p>Given the margin-box size of a node, return its content box size.\nFunctions like domGeometry.contentBox() but is more reliable since it doesn&#39;t have\nto wait for the browser to compute sizes.</p>\n"
   }
  },
  "dijit/_ConfirmDialogMixin": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_WidgetsInTemplateMixin.prototype",
    "_earlyTemplatedStartup": {
     "!doc": "<p>A fallback to preserve the 1.0 - 1.3 behavior of children in\ntemplates having their startup called before the parent widget\nfires postCreate. Defaults to &#39;false&#39;, causing child widgets to\nhave their .startup() called immediately before a parent widget\n.startup(), but always after the parent .postCreate(). Set to\n&#39;true&#39; to re-enable to previous, arguably broken, behavior.</p>\n"
    },
    "_setButtonCancelAttr": {},
    "_setButtonOkAttr": {},
    "actionBarTemplate": {},
    "buttonCancel": {
     "!doc": "<p>Label of cancel button</p>\n"
    },
    "buttonOk": {
     "!doc": "<p>Label of OK button</p>\n"
    },
    "contextRequire": {
     "!doc": "<p>Used to provide a context require to the dojo/parser in order to be\nable to use relative MIDs (e.g. <code>./Widget</code>) in the widget&#39;s template.</p>\n"
    },
    "widgetsInTemplate": {
     "!doc": "<p>Should we parse the template to find widgets that might be\ndeclared in markup inside it?  (Remove for 2.0 and assume true)</p>\n"
    },
    "startup": {
     "!type": "fn()"
    }
   }
  },
  "dijit/_WidgetsInTemplateMixin": {
   "!type": "fn()",
   "prototype": {
    "_earlyTemplatedStartup": {
     "!doc": "<p>A fallback to preserve the 1.0 - 1.3 behavior of children in\ntemplates having their startup called before the parent widget\nfires postCreate. Defaults to &#39;false&#39;, causing child widgets to\nhave their .startup() called immediately before a parent widget\n.startup(), but always after the parent .postCreate(). Set to\n&#39;true&#39; to re-enable to previous, arguably broken, behavior.</p>\n"
    },
    "contextRequire": {
     "!doc": "<p>Used to provide a context require to the dojo/parser in order to be\nable to use relative MIDs (e.g. <code>./Widget</code>) in the widget&#39;s template.</p>\n"
    },
    "widgetsInTemplate": {
     "!doc": "<p>Should we parse the template to find widgets that might be\ndeclared in markup inside it?  (Remove for 2.0 and assume true)</p>\n"
    },
    "startup": {
     "!type": "fn()"
    }
   }
  },
  "dijit/ConfirmTooltipDialog": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/TooltipDialog.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_earlyTemplatedStartup": {
     "!doc": "<p>A fallback to preserve the 1.0 - 1.3 behavior of children in\ntemplates having their startup called before the parent widget\nfires postCreate. Defaults to &#39;false&#39;, causing child widgets to\nhave their .startup() called immediately before a parent widget\n.startup(), but always after the parent .postCreate(). Set to\n&#39;true&#39; to re-enable to previous, arguably broken, behavior.</p>\n"
    },
    "_firstFocusItem": {
     "!doc": "<p>The pointer to the first focusable node in the dialog.\nSet by <code>dijit/_DialogMixin._getFocusItems()</code>.</p>\n"
    },
    "_focusManager": {},
    "_lastFocusItem": {
     "!doc": "<p>The pointer to which node has focus prior to our dialog.\nSet by <code>dijit/_DialogMixin._getFocusItems()</code>.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setButtonCancelAttr": {},
    "_setButtonOkAttr": {},
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTitleAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "actionBarTemplate": {
     "!doc": "<p>HTML snippet to show the action bar (gray bar with OK/cancel buttons).\nBlank by default, but used by ConfirmDialog/ConfirmTooltipDialog subclasses.</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "autofocus": {
     "!doc": "<p>A Toggle to modify the default focus behavior of a Dialog, which\nis to focus on the first dialog element after opening the dialog.\nFalse will disable autofocusing.  Default: true.</p>\n"
    },
    "baseClass": {
     "!doc": "<p>The root className to use for the various states of this widget</p>\n"
    },
    "buttonCancel": {
     "!doc": "<p>Label of cancel button</p>\n"
    },
    "buttonOk": {
     "!doc": "<p>Label of OK button</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "content": {
     "!doc": "<p>The innerHTML of the ContentPane.\nNote that the initialization parameter / argument to set(&quot;content&quot;, ...)\ncan be a String, DomNode, Nodelist, or _Widget.</p>\n"
    },
    "contextRequire": {
     "!doc": "<p>Used to provide a context require to the dojo/parser in order to be\nable to use relative MIDs (e.g. <code>./Widget</code>) in the widget&#39;s template.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "doLayout": {
     "!doc": "<p>Don&#39;t change this parameter from the default value.\nThis ContentPane parameter doesn&#39;t make sense for TooltipDialog, since TooltipDialog\nis never a child of a layout container, nor can you specify the size of\nTooltipDialog in order to control the size of an inner widget.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "errorMessage": {
     "!doc": "<p>Message that shows if an error occurs</p>\n"
    },
    "extractContent": {
     "!doc": "<p>Extract visible content from inside of <code>&lt;body&gt; .... &lt;/body&gt;</code>.\nI.e., strip <code>&lt;html&gt;</code> and <code>&lt;head&gt;</code> (and it&#39;s contents) from the href</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "href": {
     "!doc": "<p>The href of the content that displays now.\nSet this at construction if you want to load data externally when the\npane is shown.  (Set preload=true to load it immediately.)\nChanging href after creation doesn&#39;t have any effect; Use set(&#39;href&#39;, ...);</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "ioArgs": {
     "!doc": "<p>Parameters to pass to xhrGet() request, for example:\n\n</p>\n<pre><code>&lt;div data-dojo-type=&quot;dijit/layout/ContentPane&quot; data-dojo-props=&quot;href: &#39;./bar&#39;, ioArgs: {timeout: 500}&quot;&gt;</code></pre>\n"
    },
    "isLayoutContainer": {
     "!doc": "<p>Indicates that this widget will call resize() on it&#39;s child widgets\nwhen they become visible.</p>\n"
    },
    "isLoaded": {
     "!doc": "<p>True if the ContentPane has data in it, either specified\nduring initialization (via href or inline content), or set\nvia set(&#39;content&#39;, ...) / set(&#39;href&#39;, ...)\n\n</p>\n<p>False if it doesn&#39;t have any content, or if ContentPane is\nstill in the process of downloading href.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "loadingMessage": {
     "!doc": "<p>Message that shows while downloading</p>\n"
    },
    "onLoadDeferred": {
     "!doc": "<p>This is the <code>dojo.Deferred</code> returned by set(&#39;href&#39;, ...) and refresh().\nCalling onLoadDeferred.then() registers your\ncallback to be called only once, when the prior set(&#39;href&#39;, ...) call or\nthe initial href parameter to the constructor finishes loading.\n\n</p>\n<p>This is different than an onLoad() handler which gets called any time any href\nor content is loaded.</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "parseOnLoad": {
     "!doc": "<p>Parse content and create the widgets, if any.</p>\n"
    },
    "parserScope": {
     "!doc": "<p>Flag passed to parser.  Root for attribute names to search for.   If scopeName is dojo,\nwill search for data-dojo-type (or dojoType).  For backwards compatibility\nreasons defaults to dojo._scopeName (which is &quot;dojo&quot; except when\nmulti-version support is used, when it will be something like dojo16, dojo20, etc.)</p>\n"
    },
    "preload": {
     "!doc": "<p>Force load of data on initialization even if pane is hidden.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Prevent caching of data from href&#39;s by appending a timestamp to the href.</p>\n"
    },
    "refreshOnShow": {
     "!doc": "<p>Refresh (re-download) content when pane goes from hidden to shown</p>\n"
    },
    "searchContainerNode": {},
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "state": {
     "!doc": "<p>Will be &quot;Error&quot; if one or more of the child widgets has an invalid value,\n&quot;Incomplete&quot; if not all of the required child widgets are filled in.  Otherwise, &quot;&quot;,\nwhich indicates that the form is ready to be submitted.</p>\n"
    },
    "stopParser": {},
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "template": {
     "!doc": "<p>Flag from the parser that this ContentPane is inside a template\nso the contents are pre-parsed.</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>Description of tooltip dialog (required for a11y)</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "widgetsInTemplate": {
     "!doc": "<p>Should we parse the template to find widgets that might be\ndeclared in markup inside it?  (Remove for 2.0 and assume true)</p>\n"
    },
    "addChild": {
     "!type": "fn(widget: +dijit, insertIndex?: number)",
     "!doc": "<p>Makes the given widget a child of this widget.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "cancel": {
     "!type": "fn()",
     "!doc": "<p>Cancels an in-flight download of content</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "connectChildren": {
     "!type": "fn(inStartup: bool)",
     "!doc": "<p>You can call this function directly, ex. in the event that you\nprogrammatically add a widget to the form <em>after</em> the form has been\ninitialized.</p>\n"
    },
    "create": {
     "!type": "fn(params: ?, srcNodeRef: ?)"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn()"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy all the widgets inside the ContentPane and empty containerNode</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy the ContentPane and its contents</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "disconnectChildren": {
     "!type": "fn()",
     "!doc": "<p>Deprecated method.   Applications no longer need to call this.   Remove for 2.0.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "execute": {
     "!type": "fn(formContents: ?)",
     "!doc": "<p>Callback when the user hits the submit button.\nOverride this method to handle Dialog execution.</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Focus on first field</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getIndexOfChild": {
     "!type": "fn(child: +dijit)",
     "!doc": "<p>Gets the index of the child in this container or -1 if not found</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "getValues": {
     "!type": "fn()"
    },
    "hasChildren": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if widget has child widgets, i.e. if this.containerNode contains widgets.</p>\n"
    },
    "ioMethod": {
     "!type": "fn(args: ?)",
     "!doc": "<p>Function that should grab the content specified via href.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "isValid": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if all of the widgets are valid.\nDeprecated, will be removed in 2.0.  Use get(&quot;state&quot;) instead.</p>\n"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, ctor: ?)"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "orient": {
     "!type": "fn(node: +Node, aroundCorner: string, tooltipCorner: string)",
     "!doc": "<p>Configure widget to be displayed in given position relative to the button.\nThis is called from the dijit.popup code, and should not be called\ndirectly.</p>\n"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()"
    },
    "refresh": {
     "!type": "fn()",
     "!doc": "<p>[Re]download contents of href and display</p>\n"
    },
    "removeChild": {
     "!type": "fn(widget: +dijit._Widget|number)",
     "!doc": "<p>Removes the passed widget instance from this widget but does\nnot destroy it.  You can also pass in an integer indicating\nthe index within the container to remove (ie, removeChild(5) removes the sixth widget).</p>\n"
    },
    "reset": {
     "!type": "fn()"
    },
    "resize": {
     "!type": "fn(changeSize: ?, resultSize: ?)",
     "!doc": "<p>See <code>dijit/layout/_LayoutWidget.resize()</code> for description.\nAlthough ContentPane doesn&#39;t extend _LayoutWidget, it does implement\nthe same API.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setContent": {
     "!type": "fn(data: string|+Node|+Nodelist)",
     "!doc": "<p>Deprecated.   Use set(&#39;content&#39;, ...) instead.</p>\n"
    },
    "setHref": {
     "!type": "fn(href: string|+Uri)",
     "!doc": "<p>Deprecated.   Use set(&#39;href&#39;, ...) instead.</p>\n"
    },
    "setValues": {
     "!type": "fn(val: ?)"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Call startup() on all children including non _Widget ones like dojo/dnd/Source objects</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "validate": {
     "!type": "fn()",
     "!doc": "<p>returns if the form is valid - same as isValid - but\nprovides a few additional (ui-specific) features:\n\n</p>\n<ol>\n<li>it will highlight any sub-widgets that are not valid</li>\n<li>it will call focus() on the first invalid sub-widget</li>\n</ol>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/TooltipDialog": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/layout/ContentPane.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_firstFocusItem": {
     "!doc": "<p>The pointer to the first focusable node in the dialog.\nSet by <code>dijit/_DialogMixin._getFocusItems()</code>.</p>\n"
    },
    "_focusManager": {},
    "_lastFocusItem": {
     "!doc": "<p>The pointer to which node has focus prior to our dialog.\nSet by <code>dijit/_DialogMixin._getFocusItems()</code>.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTitleAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "actionBarTemplate": {
     "!doc": "<p>HTML snippet to show the action bar (gray bar with OK/cancel buttons).\nBlank by default, but used by ConfirmDialog/ConfirmTooltipDialog subclasses.</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "autofocus": {
     "!doc": "<p>A Toggle to modify the default focus behavior of a Dialog, which\nis to focus on the first dialog element after opening the dialog.\nFalse will disable autofocusing.  Default: true.</p>\n"
    },
    "baseClass": {
     "!doc": "<p>The root className to use for the various states of this widget</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "content": {
     "!doc": "<p>The innerHTML of the ContentPane.\nNote that the initialization parameter / argument to set(&quot;content&quot;, ...)\ncan be a String, DomNode, Nodelist, or _Widget.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "doLayout": {
     "!doc": "<p>Don&#39;t change this parameter from the default value.\nThis ContentPane parameter doesn&#39;t make sense for TooltipDialog, since TooltipDialog\nis never a child of a layout container, nor can you specify the size of\nTooltipDialog in order to control the size of an inner widget.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "errorMessage": {
     "!doc": "<p>Message that shows if an error occurs</p>\n"
    },
    "extractContent": {
     "!doc": "<p>Extract visible content from inside of <code>&lt;body&gt; .... &lt;/body&gt;</code>.\nI.e., strip <code>&lt;html&gt;</code> and <code>&lt;head&gt;</code> (and it&#39;s contents) from the href</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "href": {
     "!doc": "<p>The href of the content that displays now.\nSet this at construction if you want to load data externally when the\npane is shown.  (Set preload=true to load it immediately.)\nChanging href after creation doesn&#39;t have any effect; Use set(&#39;href&#39;, ...);</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "ioArgs": {
     "!doc": "<p>Parameters to pass to xhrGet() request, for example:\n\n</p>\n<pre><code>&lt;div data-dojo-type=&quot;dijit/layout/ContentPane&quot; data-dojo-props=&quot;href: &#39;./bar&#39;, ioArgs: {timeout: 500}&quot;&gt;</code></pre>\n"
    },
    "isLayoutContainer": {
     "!doc": "<p>Indicates that this widget will call resize() on it&#39;s child widgets\nwhen they become visible.</p>\n"
    },
    "isLoaded": {
     "!doc": "<p>True if the ContentPane has data in it, either specified\nduring initialization (via href or inline content), or set\nvia set(&#39;content&#39;, ...) / set(&#39;href&#39;, ...)\n\n</p>\n<p>False if it doesn&#39;t have any content, or if ContentPane is\nstill in the process of downloading href.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "loadingMessage": {
     "!doc": "<p>Message that shows while downloading</p>\n"
    },
    "onLoadDeferred": {
     "!doc": "<p>This is the <code>dojo.Deferred</code> returned by set(&#39;href&#39;, ...) and refresh().\nCalling onLoadDeferred.then() registers your\ncallback to be called only once, when the prior set(&#39;href&#39;, ...) call or\nthe initial href parameter to the constructor finishes loading.\n\n</p>\n<p>This is different than an onLoad() handler which gets called any time any href\nor content is loaded.</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "parseOnLoad": {
     "!doc": "<p>Parse content and create the widgets, if any.</p>\n"
    },
    "parserScope": {
     "!doc": "<p>Flag passed to parser.  Root for attribute names to search for.   If scopeName is dojo,\nwill search for data-dojo-type (or dojoType).  For backwards compatibility\nreasons defaults to dojo._scopeName (which is &quot;dojo&quot; except when\nmulti-version support is used, when it will be something like dojo16, dojo20, etc.)</p>\n"
    },
    "preload": {
     "!doc": "<p>Force load of data on initialization even if pane is hidden.</p>\n"
    },
    "preventCache": {
     "!doc": "<p>Prevent caching of data from href&#39;s by appending a timestamp to the href.</p>\n"
    },
    "refreshOnShow": {
     "!doc": "<p>Refresh (re-download) content when pane goes from hidden to shown</p>\n"
    },
    "searchContainerNode": {},
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "state": {
     "!doc": "<p>Will be &quot;Error&quot; if one or more of the child widgets has an invalid value,\n&quot;Incomplete&quot; if not all of the required child widgets are filled in.  Otherwise, &quot;&quot;,\nwhich indicates that the form is ready to be submitted.</p>\n"
    },
    "stopParser": {},
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "template": {
     "!doc": "<p>Flag from the parser that this ContentPane is inside a template\nso the contents are pre-parsed.</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>Description of tooltip dialog (required for a11y)</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "addChild": {
     "!type": "fn(widget: +dijit, insertIndex?: number)",
     "!doc": "<p>Makes the given widget a child of this widget.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "cancel": {
     "!type": "fn()",
     "!doc": "<p>Cancels an in-flight download of content</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "connectChildren": {
     "!type": "fn(inStartup: bool)",
     "!doc": "<p>You can call this function directly, ex. in the event that you\nprogrammatically add a widget to the form <em>after</em> the form has been\ninitialized.</p>\n"
    },
    "create": {
     "!type": "fn(params: ?, srcNodeRef: ?)"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn()"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy all the widgets inside the ContentPane and empty containerNode</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy the ContentPane and its contents</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "disconnectChildren": {
     "!type": "fn()",
     "!doc": "<p>Deprecated method.   Applications no longer need to call this.   Remove for 2.0.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "execute": {
     "!type": "fn(formContents: ?)",
     "!doc": "<p>Callback when the user hits the submit button.\nOverride this method to handle Dialog execution.</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Focus on first field</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getIndexOfChild": {
     "!type": "fn(child: +dijit)",
     "!doc": "<p>Gets the index of the child in this container or -1 if not found</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "getValues": {
     "!type": "fn()"
    },
    "hasChildren": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if widget has child widgets, i.e. if this.containerNode contains widgets.</p>\n"
    },
    "ioMethod": {
     "!type": "fn(args: ?)",
     "!doc": "<p>Function that should grab the content specified via href.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "isValid": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if all of the widgets are valid.\nDeprecated, will be removed in 2.0.  Use get(&quot;state&quot;) instead.</p>\n"
    },
    "markupFactory": {
     "!type": "fn(params: ?, node: ?, ctor: ?)"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "orient": {
     "!type": "fn(node: +Node, aroundCorner: string, tooltipCorner: string)",
     "!doc": "<p>Configure widget to be displayed in given position relative to the button.\nThis is called from the dijit.popup code, and should not be called\ndirectly.</p>\n"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()"
    },
    "refresh": {
     "!type": "fn()",
     "!doc": "<p>[Re]download contents of href and display</p>\n"
    },
    "removeChild": {
     "!type": "fn(widget: +dijit._Widget|number)",
     "!doc": "<p>Removes the passed widget instance from this widget but does\nnot destroy it.  You can also pass in an integer indicating\nthe index within the container to remove (ie, removeChild(5) removes the sixth widget).</p>\n"
    },
    "reset": {
     "!type": "fn()"
    },
    "resize": {
     "!type": "fn(changeSize: ?, resultSize: ?)",
     "!doc": "<p>See <code>dijit/layout/_LayoutWidget.resize()</code> for description.\nAlthough ContentPane doesn&#39;t extend _LayoutWidget, it does implement\nthe same API.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setContent": {
     "!type": "fn(data: string|+Node|+Nodelist)",
     "!doc": "<p>Deprecated.   Use set(&#39;content&#39;, ...) instead.</p>\n"
    },
    "setHref": {
     "!type": "fn(href: string|+Uri)",
     "!doc": "<p>Deprecated.   Use set(&#39;href&#39;, ...) instead.</p>\n"
    },
    "setValues": {
     "!type": "fn(val: ?)"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Call startup() on all children including non _Widget ones like dojo/dnd/Source objects</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "validate": {
     "!type": "fn()",
     "!doc": "<p>returns if the form is valid - same as isValid - but\nprovides a few additional (ui-specific) features:\n\n</p>\n<ol>\n<li>it will highlight any sub-widgets that are not valid</li>\n<li>it will call focus() on the first invalid sub-widget</li>\n</ol>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/Declaration": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_Widget.prototype",
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_noScript": {
     "!doc": "<p>Flag to parser to leave alone the script tags contained inside of me</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {
     "!doc": "<p>Root CSS class of the widget (ex: dijitTextBox), used to construct CSS classes to indicate\nwidget state.</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "defaults": {},
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "mixins": {
     "!doc": "<p>List containing the prototype for this widget, and also any mixins,\nex: [&quot;dijit._Widget&quot;, &quot;dijit._Container&quot;]</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "stopParser": {
     "!doc": "<p>Flag to parser to not try and parse widgets declared inside of me</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "widgetClass": {
     "!doc": "<p>Name of class being declared, ex: &quot;acme.myWidget&quot;</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/DropDownMenu": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_MenuBase.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_keyNavCodes": {
     "!doc": "<p>Hash mapping key code (arrow keys and home/end key) to functions to handle those keys.\nUsually not used directly, as subclasses can instead override _onLeftArrow() etc.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_searchString": {},
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "activated": {
     "!doc": "<p>This Menu has been clicked (mouse or via space/arrow key) or opened as a submenu,\nso mere mouseover will open submenus.  Focusing a menu via TAB does NOT automatically make it active\nsince TAB is a navigation operation and not a selection one.\nFor Windows apps, pressing the ALT key focuses the menubar menus (similar to TAB navigation) but the\nmenu is not active (ie no dropdown) until an item is clicked.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "autoFocus": {
     "!doc": "<p>A toggle to control whether or not a Menu gets focused when opened as a drop down from a MenuBar\nor DropDownButton/ComboButton.   Note though that it always get focused when opened via the keyboard.</p>\n"
    },
    "baseClass": {},
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "focusedChild": {
     "!doc": "<p>The currently focused child widget, or null if there isn&#39;t one</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "multiCharSearchDuration": {
     "!doc": "<p>If multiple characters are typed where each keystroke happens within\nmultiCharSearchDuration of the previous keystroke,\nsearch for nodes matching all the keystrokes.\n\n</p>\n<p>For example, typing &quot;ab&quot; will search for entries starting with\n&quot;ab&quot; unless the delay between &quot;a&quot; and &quot;b&quot; is greater than multiCharSearchDuration.</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "parentMenu": {
     "!doc": "<p>pointer to menu that displayed me</p>\n"
    },
    "passivePopupDelay": {
     "!doc": "<p>For a passive (unclicked) Menu, number of milliseconds before hovering (without clicking) will cause\nthe popup to open.  Default is Infinity, meaning you need to click the menu to open it.</p>\n"
    },
    "popupDelay": {
     "!doc": "<p>After a menu has been activated (by clicking on it etc.), number of milliseconds before hovering\n(without clicking) another MenuItem causes that MenuItem&#39;s popup to automatically open.</p>\n"
    },
    "searchContainerNode": {},
    "selected": {
     "!doc": "<p>Currently selected (a.k.a. highlighted) MenuItem, or null if no MenuItem is selected.\nIf a submenu is open, will be set to MenuItem that displayed the submenu.   OTOH, if\nthis Menu is in passive mode (i.e. hasn&#39;t been clicked yet), will be null, because\n&quot;selected&quot; is not merely &quot;hovered&quot;.</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Tab index of the container; same as HTML tabIndex attribute.\nNote then when user tabs into the container, focus is immediately\nmoved to the first item in the container.</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "addChild": {
     "!type": "fn(widget: +dijit, insertIndex?: number)"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "childSelector": {
     "!type": "fn(node: +Node)",
     "!doc": "<p>Selector (passed to on.selector()) used to identify MenuItem child widgets, but exclude inert children\nlike MenuSeparator.  If subclass overrides to a string (ex: &quot;&gt; *&quot;), the subclass must require dojo/query.</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "connectKeyNavHandlers": {
     "!type": "fn(prevKeyCodes: +dojo, nextKeyCodes: +dojo)",
     "!doc": "<p>Deprecated.  You can call this in postCreate() to attach the keyboard handlers to the container,\nbut the preferred method is to override _onLeftArrow() and _onRightArrow(), or\n_onUpArrow() and _onDownArrow(), to call focusPrev() and focusNext().</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Default focus() implementation: focus the first child.</p>\n"
    },
    "focusChild": {
     "!type": "fn(widget: +dijit, last: bool)",
     "!doc": "<p>Focus specified child widget.</p>\n"
    },
    "focusFirstChild": {
     "!type": "fn()",
     "!doc": "<p>Focus the first focusable child in the container.</p>\n"
    },
    "focusLastChild": {
     "!type": "fn()",
     "!doc": "<p>Focus the last focusable child in the container.</p>\n"
    },
    "focusNext": {
     "!type": "fn()",
     "!doc": "<p>Focus the next widget</p>\n"
    },
    "focusPrev": {
     "!type": "fn()",
     "!doc": "<p>Focus the last focusable node in the previous widget\n(ex: go to the ComboButton icon section rather than button section)</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getIndexOfChild": {
     "!type": "fn(child: +dijit)",
     "!doc": "<p>Gets the index of the child in this container or -1 if not found</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "hasChildren": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if widget has child widgets, i.e. if this.containerNode contains widgets.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "removeChild": {
     "!type": "fn(widget: +dijit._Widget|number)",
     "!doc": "<p>Removes the passed widget instance from this widget but does\nnot destroy it.  You can also pass in an integer indicating\nthe index within the container to remove (ie, removeChild(5) removes the sixth widget).</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "startupKeyNavChildren": {
     "!type": "fn()"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/_MenuBase": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_Widget.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_keyNavCodes": {
     "!doc": "<p>Hash mapping key code (arrow keys and home/end key) to functions to handle those keys.\nUsually not used directly, as subclasses can instead override _onLeftArrow() etc.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_searchString": {},
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "activated": {
     "!doc": "<p>This Menu has been clicked (mouse or via space/arrow key) or opened as a submenu,\nso mere mouseover will open submenus.  Focusing a menu via TAB does NOT automatically make it active\nsince TAB is a navigation operation and not a selection one.\nFor Windows apps, pressing the ALT key focuses the menubar menus (similar to TAB navigation) but the\nmenu is not active (ie no dropdown) until an item is clicked.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "autoFocus": {
     "!doc": "<p>A toggle to control whether or not a Menu gets focused when opened as a drop down from a MenuBar\nor DropDownButton/ComboButton.   Note though that it always get focused when opened via the keyboard.</p>\n"
    },
    "baseClass": {
     "!doc": "<p>Root CSS class of the widget (ex: dijitTextBox), used to construct CSS classes to indicate\nwidget state.</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "focusedChild": {
     "!doc": "<p>The currently focused child widget, or null if there isn&#39;t one</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "multiCharSearchDuration": {
     "!doc": "<p>If multiple characters are typed where each keystroke happens within\nmultiCharSearchDuration of the previous keystroke,\nsearch for nodes matching all the keystrokes.\n\n</p>\n<p>For example, typing &quot;ab&quot; will search for entries starting with\n&quot;ab&quot; unless the delay between &quot;a&quot; and &quot;b&quot; is greater than multiCharSearchDuration.</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "parentMenu": {
     "!doc": "<p>pointer to menu that displayed me</p>\n"
    },
    "passivePopupDelay": {
     "!doc": "<p>For a passive (unclicked) Menu, number of milliseconds before hovering (without clicking) will cause\nthe popup to open.  Default is Infinity, meaning you need to click the menu to open it.</p>\n"
    },
    "popupDelay": {
     "!doc": "<p>After a menu has been activated (by clicking on it etc.), number of milliseconds before hovering\n(without clicking) another MenuItem causes that MenuItem&#39;s popup to automatically open.</p>\n"
    },
    "searchContainerNode": {},
    "selected": {
     "!doc": "<p>Currently selected (a.k.a. highlighted) MenuItem, or null if no MenuItem is selected.\nIf a submenu is open, will be set to MenuItem that displayed the submenu.   OTOH, if\nthis Menu is in passive mode (i.e. hasn&#39;t been clicked yet), will be null, because\n&quot;selected&quot; is not merely &quot;hovered&quot;.</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Tab index of the container; same as HTML tabIndex attribute.\nNote then when user tabs into the container, focus is immediately\nmoved to the first item in the container.</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {
     "!doc": "<p>A string that represents the widget template.\nUse in conjunction with dojo.cache() to load from a file.</p>\n"
    },
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "addChild": {
     "!type": "fn(widget: +dijit, insertIndex?: number)"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "childSelector": {
     "!type": "fn(node: +Node)",
     "!doc": "<p>Selector (passed to on.selector()) used to identify MenuItem child widgets, but exclude inert children\nlike MenuSeparator.  If subclass overrides to a string (ex: &quot;&gt; *&quot;), the subclass must require dojo/query.</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "connectKeyNavHandlers": {
     "!type": "fn(prevKeyCodes: +dojo, nextKeyCodes: +dojo)",
     "!doc": "<p>Deprecated.  You can call this in postCreate() to attach the keyboard handlers to the container,\nbut the preferred method is to override _onLeftArrow() and _onRightArrow(), or\n_onUpArrow() and _onDownArrow(), to call focusPrev() and focusNext().</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Default focus() implementation: focus the first child.</p>\n"
    },
    "focusChild": {
     "!type": "fn(widget: +dijit, last: bool)",
     "!doc": "<p>Focus specified child widget.</p>\n"
    },
    "focusFirstChild": {
     "!type": "fn()",
     "!doc": "<p>Focus the first focusable child in the container.</p>\n"
    },
    "focusLastChild": {
     "!type": "fn()",
     "!doc": "<p>Focus the last focusable child in the container.</p>\n"
    },
    "focusNext": {
     "!type": "fn()",
     "!doc": "<p>Focus the next widget</p>\n"
    },
    "focusPrev": {
     "!type": "fn()",
     "!doc": "<p>Focus the last focusable node in the previous widget\n(ex: go to the ComboButton icon section rather than button section)</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getIndexOfChild": {
     "!type": "fn(child: +dijit)",
     "!doc": "<p>Gets the index of the child in this container or -1 if not found</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "hasChildren": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if widget has child widgets, i.e. if this.containerNode contains widgets.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "removeChild": {
     "!type": "fn(widget: +dijit._Widget|number)",
     "!doc": "<p>Removes the passed widget instance from this widget but does\nnot destroy it.  You can also pass in an integer indicating\nthe index within the container to remove (ie, removeChild(5) removes the sixth widget).</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "startupKeyNavChildren": {
     "!type": "fn()"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/_KeyNavContainer": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_FocusMixin.prototype",
    "_focusManager": {},
    "_keyNavCodes": {
     "!doc": "<p>Hash mapping key code (arrow keys and home/end key) to functions to handle those keys.\nUsually not used directly, as subclasses can instead override _onLeftArrow() etc.</p>\n"
    },
    "_searchString": {},
    "focusedChild": {
     "!doc": "<p>The currently focused child widget, or null if there isn&#39;t one</p>\n"
    },
    "multiCharSearchDuration": {
     "!doc": "<p>If multiple characters are typed where each keystroke happens within\nmultiCharSearchDuration of the previous keystroke,\nsearch for nodes matching all the keystrokes.\n\n</p>\n<p>For example, typing &quot;ab&quot; will search for entries starting with\n&quot;ab&quot; unless the delay between &quot;a&quot; and &quot;b&quot; is greater than multiCharSearchDuration.</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Tab index of the container; same as HTML tabIndex attribute.\nNote then when user tabs into the container, focus is immediately\nmoved to the first item in the container.</p>\n"
    },
    "addChild": {
     "!type": "fn(widget: +dijit, insertIndex?: number)"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "childSelector": {
     "!type": "fn(node: +Node)"
    },
    "connectKeyNavHandlers": {
     "!type": "fn(prevKeyCodes: +dojo, nextKeyCodes: +dojo)",
     "!doc": "<p>Deprecated.  You can call this in postCreate() to attach the keyboard handlers to the container,\nbut the preferred method is to override _onLeftArrow() and _onRightArrow(), or\n_onUpArrow() and _onDownArrow(), to call focusPrev() and focusNext().</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Default focus() implementation: focus the first child.</p>\n"
    },
    "focusChild": {
     "!type": "fn(widget: +dijit, last: bool)",
     "!doc": "<p>Focus specified child widget.</p>\n"
    },
    "focusFirstChild": {
     "!type": "fn()",
     "!doc": "<p>Focus the first focusable child in the container.</p>\n"
    },
    "focusLastChild": {
     "!type": "fn()",
     "!doc": "<p>Focus the last focusable child in the container.</p>\n"
    },
    "focusNext": {
     "!type": "fn()",
     "!doc": "<p>Focus the next widget</p>\n"
    },
    "focusPrev": {
     "!type": "fn()",
     "!doc": "<p>Focus the last focusable node in the previous widget\n(ex: go to the ComboButton icon section rather than button section)</p>\n"
    },
    "getIndexOfChild": {
     "!type": "fn(child: +dijit)",
     "!doc": "<p>Gets the index of the child in this container or -1 if not found</p>\n"
    },
    "hasChildren": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if widget has child widgets, i.e. if this.containerNode contains widgets.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "removeChild": {
     "!type": "fn(widget: +dijit._Widget|number)",
     "!doc": "<p>Removes the passed widget instance from this widget but does\nnot destroy it.  You can also pass in an integer indicating\nthe index within the container to remove (ie, removeChild(5) removes the sixth widget).</p>\n"
    },
    "startup": {
     "!type": "fn()"
    },
    "startupKeyNavChildren": {
     "!type": "fn()"
    }
   }
  },
  "dijit/_KeyNavMixin": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_FocusMixin.prototype",
    "_focusManager": {},
    "_keyNavCodes": {
     "!doc": "<p>Hash mapping key code (arrow keys and home/end key) to functions to handle those keys.\nUsually not used directly, as subclasses can instead override _onLeftArrow() etc.</p>\n"
    },
    "_searchString": {},
    "childSelector": {
     "!doc": "<p>Selector (passed to on.selector()) used to identify what to treat as a child widget.   Used to monitor\nfocus events and set this.focusedChild.   Must be set by implementing class.   If this is a string\n(ex: &quot;&gt; *&quot;) then the implementing class must require dojo/query.</p>\n"
    },
    "focusedChild": {
     "!doc": "<p>The currently focused child widget, or null if there isn&#39;t one</p>\n"
    },
    "multiCharSearchDuration": {
     "!doc": "<p>If multiple characters are typed where each keystroke happens within\nmultiCharSearchDuration of the previous keystroke,\nsearch for nodes matching all the keystrokes.\n\n</p>\n<p>For example, typing &quot;ab&quot; will search for entries starting with\n&quot;ab&quot; unless the delay between &quot;a&quot; and &quot;b&quot; is greater than multiCharSearchDuration.</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Tab index of the container; same as HTML tabIndex attribute.\nNote then when user tabs into the container, focus is immediately\nmoved to the first item in the container.</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Default focus() implementation: focus the first child.</p>\n"
    },
    "focusChild": {
     "!type": "fn(widget: +dijit, last: bool)",
     "!doc": "<p>Focus specified child widget.</p>\n"
    },
    "focusFirstChild": {
     "!type": "fn()",
     "!doc": "<p>Focus the first focusable child in the container.</p>\n"
    },
    "focusLastChild": {
     "!type": "fn()",
     "!doc": "<p>Focus the last focusable child in the container.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    }
   }
  },
  "dijit/Editor": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_editor/RichText.prototype",
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_editorCommandsLocalized": {},
    "_focusManager": {},
    "_local2NativeFormatNames": {},
    "_mozSettingProps": {},
    "_NAME_CONTENT_SEP": {
     "!doc": "<p>USed to separate name from content.  Just a colon isn&#39;t safe.</p>\n"
    },
    "_native2LocalFormatNames": {},
    "_qcaCache": {},
    "_SEPARATOR": {
     "!doc": "<p>Used to concat contents from multiple editors into a single string,\nso they can be saved into a single <code>&lt;textarea&gt;</code> node.  See &quot;name&quot; attribute.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "_steps": {},
    "_undoedSteps": {},
    "_updateTimer": {},
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {},
    "captureEvents": {
     "!doc": "<p>Events which should be connected to the underlying editing\narea, events in this array will be addListener with\ncapture=true.</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "customUndo": {
     "!doc": "<p>Whether we shall use custom undo/redo support instead of the native\nbrowser support. By default, we now use custom undo.  It works better\nthan native browser support and provides a consistent behavior across\nbrowsers with a minimal performance hit.  We already had the hit on\nthe slowest browser, IE, anyway.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "disabled": {
     "!doc": "<p>The editor is disabled; the text cannot be changed.</p>\n"
    },
    "disableSpellCheck": {
     "!doc": "<p>When true, disables the browser&#39;s native spell checking, if supported.\nWorks only in Firefox.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "editActionInterval": {
     "!doc": "<p>When using customUndo, not every keystroke will be saved as a step.\nInstead typing (including delete) will be grouped together: after\na user stops typing for editActionInterval seconds, a step will be\nsaved; if a user resume typing within editActionInterval seconds,\nthe timeout will be restarted. By default, editActionInterval is 3\nseconds.</p>\n"
    },
    "events": {
     "!doc": "<p>events which should be connected to the underlying editing area</p>\n"
    },
    "extraPlugins": {
     "!doc": "<p>A list of extra plugin names which will be appended to plugins array</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "focusOnLoad": {
     "!doc": "<p>Focus into this widget when the page is loaded</p>\n"
    },
    "height": {
     "!doc": "<p>Set height to fix the editor at a specific height, with scrolling.\nBy default, this is 300px.  If you want to have the editor always\nresizes to accommodate the content, use AlwaysShowToolbar plugin\nand set height=&quot;&quot;.  If this editor is used within a layout widget,\nset height=&quot;100%&quot;.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "inheritWidth": {
     "!doc": "<p>whether to inherit the parent&#39;s width or simply use 100%</p>\n"
    },
    "isClosed": {},
    "isLoaded": {},
    "isTabIndent": {
     "!doc": "<p>Make tab key and shift-tab indent and outdent rather than navigating.\nCaution: sing this makes web pages inaccessible to users unable to use a mouse.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "minHeight": {
     "!doc": "<p>The minimum height that the editor should have.</p>\n"
    },
    "name": {
     "!doc": "<p>Specifies the name of a (hidden) <code>&lt;textarea&gt;</code> node on the page that&#39;s used to save\nthe editor content on page leave.   Used to restore editor contents after navigating\nto a new page and then hitting the back button.</p>\n"
    },
    "onLoadDeferred": {
     "!doc": "<p>Deferred which is fired when the editor finishes loading.\nCall myEditor.onLoadDeferred.then(callback) it to be informed\nwhen the rich-text area initialization is finalized.</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "plugins": {
     "!doc": "<p>A list of plugin names (as strings) or instances (as objects)\nfor this widget.\n\n</p>\n<p>When declared in markup, it might look like:\n\n</p>\n<pre><code>plugins=&quot;[&#39;bold&#39;,{name:&#39;dijit._editor.plugins.FontChoice&#39;, command:&#39;fontName&#39;, generic:true}]&quot;</code></pre>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "styleSheets": {
     "!doc": "<p>semicolon (&quot;;&quot;) separated list of css files for the editing area</p>\n"
    },
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "updateInterval": {},
    "addKeyHandler": {
     "!type": "fn(key: string|number, ctrl: bool, shift: bool, handler: fn())",
     "!doc": "<p>Add a handler for a keyboard shortcut</p>\n"
    },
    "addPlugin": {
     "!type": "fn(plugin: string|?|fn(), index?: +Integer)",
     "!doc": "<p>takes a plugin name as a string or a plugin instance and\nadds it to the toolbar and associates it with this editor\ninstance. The resulting plugin is added to the Editor&#39;s\nplugins array. If index is passed, it&#39;s placed in the plugins\narray at that index. No big magic, but a nice helper for\npassing in plugin names via markup.</p>\n"
    },
    "addStyleSheet": {
     "!type": "fn(uri: +dojo)",
     "!doc": "<p>add an external stylesheet for the editing area</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "blur": {
     "!type": "fn()",
     "!doc": "<p>Remove focus from this instance.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn()"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "execCommand": {
     "!type": "fn(cmd: ?)",
     "!doc": "<p>Main handler for executing any commands to the editor, like paste, bold, etc.\nCalled by plugins, but not meant to be called by end users.</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Move focus to this editor</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getFooterHeight": {
     "!type": "fn()",
     "!doc": "<p>A function for obtaining the height of the footer node</p>\n"
    },
    "getHeaderHeight": {
     "!type": "fn()",
     "!doc": "<p>A function for obtaining the height of the header node</p>\n"
    },
    "getNodeChildrenHtml": {
     "!type": "fn(dom: +Node)",
     "!doc": "<p>Deprecated.   Use dijit/_editor/html::getChildrenHtml() instead.</p>\n"
    },
    "getNodeHtml": {
     "!type": "fn(node: +Node)",
     "!doc": "<p>Deprecated.   Use dijit/_editor/html::_getNodeHtml() instead.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "layout": {
     "!type": "fn()",
     "!doc": "<p>Called from <code>dijit/layout/_LayoutWidget.resize()</code>.  This shouldn&#39;t be called directly</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Extension to make sure a deferred is in place before certain functions\nexecute, like making sure all the plugins are properly inserted.</p>\n"
    },
    "queryCommandEnabled": {
     "!type": "fn(cmd: ?) -> bool",
     "!doc": "<p>Returns true if specified editor command is enabled.\nUsed by the plugins to know when to highlight/not highlight buttons.</p>\n"
    },
    "queryCommandState": {
     "!type": "fn(command: ?) -> bool",
     "!doc": "<p>Check the state of a given command and returns true or false.</p>\n"
    },
    "queryCommandValue": {
     "!type": "fn(command: ?) -> bool",
     "!doc": "<p>Check the value of a given command. This matters most for\ncustom selections and complex values like font value setting.</p>\n"
    },
    "removeStyleSheet": {
     "!type": "fn(uri: +dojo)",
     "!doc": "<p>remove an external stylesheet for the editing area</p>\n"
    },
    "replaceValue": {
     "!type": "fn(html: string)",
     "!doc": "<p>over-ride of replaceValue to support custom undo and stack maintenance.</p>\n"
    },
    "resize": {
     "!type": "fn(size: ?)",
     "!doc": "<p>Resize the editor to the specified size, see <code>dijit/layout/_LayoutWidget.resize()</code></p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setDisabled": {
     "!type": "fn(disabled: bool)",
     "!doc": "<p>Deprecated, use set(&#39;disabled&#39;, ...) instead.</p>\n"
    },
    "setupDefaultShortcuts": {
     "!type": "fn()",
     "!doc": "<p>Add some default key handlers</p>\n"
    },
    "setValue": {
     "!type": "fn(html: string)",
     "!doc": "<p>This function sets the content. No undo history is preserved.\nUsers should use set(&#39;value&#39;, ...) instead.</p>\n"
    },
    "startup": {
     "!type": "fn()"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/Toolbar": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_Widget.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_keyNavCodes": {
     "!doc": "<p>Hash mapping key code (arrow keys and home/end key) to functions to handle those keys.\nUsually not used directly, as subclasses can instead override _onLeftArrow() etc.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_searchString": {},
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {},
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "focusedChild": {
     "!doc": "<p>The currently focused child widget, or null if there isn&#39;t one</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "multiCharSearchDuration": {
     "!doc": "<p>If multiple characters are typed where each keystroke happens within\nmultiCharSearchDuration of the previous keystroke,\nsearch for nodes matching all the keystrokes.\n\n</p>\n<p>For example, typing &quot;ab&quot; will search for entries starting with\n&quot;ab&quot; unless the delay between &quot;a&quot; and &quot;b&quot; is greater than multiCharSearchDuration.</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "searchContainerNode": {},
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Tab index of the container; same as HTML tabIndex attribute.\nNote then when user tabs into the container, focus is immediately\nmoved to the first item in the container.</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "addChild": {
     "!type": "fn(widget: +dijit, insertIndex?: number)"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "childSelector": {
     "!type": "fn(node: +Node)"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "connectKeyNavHandlers": {
     "!type": "fn(prevKeyCodes: +dojo, nextKeyCodes: +dojo)",
     "!doc": "<p>Deprecated.  You can call this in postCreate() to attach the keyboard handlers to the container,\nbut the preferred method is to override _onLeftArrow() and _onRightArrow(), or\n_onUpArrow() and _onDownArrow(), to call focusPrev() and focusNext().</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Default focus() implementation: focus the first child.</p>\n"
    },
    "focusChild": {
     "!type": "fn(widget: +dijit, last: bool)",
     "!doc": "<p>Focus specified child widget.</p>\n"
    },
    "focusFirstChild": {
     "!type": "fn()",
     "!doc": "<p>Focus the first focusable child in the container.</p>\n"
    },
    "focusLastChild": {
     "!type": "fn()",
     "!doc": "<p>Focus the last focusable child in the container.</p>\n"
    },
    "focusNext": {
     "!type": "fn()",
     "!doc": "<p>Focus the next widget</p>\n"
    },
    "focusPrev": {
     "!type": "fn()",
     "!doc": "<p>Focus the last focusable node in the previous widget\n(ex: go to the ComboButton icon section rather than button section)</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getIndexOfChild": {
     "!type": "fn(child: +dijit)",
     "!doc": "<p>Gets the index of the child in this container or -1 if not found</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "hasChildren": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if widget has child widgets, i.e. if this.containerNode contains widgets.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "removeChild": {
     "!type": "fn(widget: +dijit._Widget|number)",
     "!doc": "<p>Removes the passed widget instance from this widget but does\nnot destroy it.  You can also pass in an integer indicating\nthe index within the container to remove (ie, removeChild(5) removes the sixth widget).</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "startupKeyNavChildren": {
     "!type": "fn()"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/ToolbarSeparator": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_Widget.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {
     "!doc": "<p>Root CSS class of the widget (ex: dijitTextBox), used to construct CSS classes to indicate\nwidget state.</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "searchContainerNode": {},
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "isFocusable": {
     "!type": "fn() -> bool",
     "!doc": "<p>This widget isn&#39;t focusable, so pass along that fact.</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/layout/_LayoutWidget": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_Widget.prototype",
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTitleAttr": {},
    "_setTypeAttr": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {
     "!doc": "<p>This class name is applied to the widget&#39;s domNode\nand also may be used to generate names for sub nodes,\nfor example dijitTabContainer-content.</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "isLayoutContainer": {
     "!doc": "<p>Indicates that this widget is going to call resize() on its\nchildren widgets, setting their size, when they become visible.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "addChild": {
     "!type": "fn(child: +dijit, insertIndex?: +Integer)"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getIndexInParent": {
     "!type": "fn() -> number",
     "!doc": "<p>Returns the index of this widget within its container parent.\nIt returns -1 if the parent does not exist, or if the parent\nis not a dijit/_Container</p>\n"
    },
    "getIndexOfChild": {
     "!type": "fn(child: +dijit)",
     "!doc": "<p>Gets the index of the child in this container or -1 if not found</p>\n"
    },
    "getNextSibling": {
     "!type": "fn()",
     "!doc": "<p>Returns null if this is the last child of the parent,\notherwise returns the next element sibling to the &quot;right&quot;.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "getPreviousSibling": {
     "!type": "fn()",
     "!doc": "<p>Returns null if this is the first child of the parent,\notherwise returns the next element sibling to the &quot;left&quot;.</p>\n"
    },
    "hasChildren": {
     "!type": "fn() -> bool",
     "!doc": "<p>Returns true if widget has child widgets, i.e. if this.containerNode contains widgets.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "layout": {
     "!type": "fn()",
     "!doc": "<p>Widgets override this method to size and position their contents/children.\nWhen this is called this._contentBox is guaranteed to be set (see resize()).\n\n</p>\n<p>This is called after startup(), and also when the widget&#39;s size has been\nchanged.</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "removeChild": {
     "!type": "fn(child: +dijit)"
    },
    "resize": {
     "!type": "fn(changeSize?: ?, resultSize?: ?)",
     "!doc": "<p>Call this to resize a widget, or after its size has changed.</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Called after all the widgets have been instantiated and their\ndom nodes have been inserted somewhere under <body>.\n\n</p>\n<p>Widgets should override this method to do any initialization\ndependent on other widgets existing, and then call\nthis superclass method to finish things off.\n\n</p>\n<p>startup() in subclasses shouldn&#39;t do anything\nsize related because the size of the widget hasn&#39;t been set yet.</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/form/ToggleButton": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/form/Button.prototype",
    "_aria_attr": {},
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_onChangeActive": {
     "!doc": "<p>Indicates that changes to the value should call onChange() callback.\nThis is false during widget initialization, to avoid calling onChange()\nwhen the initial value is set.</p>\n"
    },
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIconClassAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTabIndexAttr": {},
    "_setTypeAttr": {},
    "_setValueAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "alt": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "aria-label": {},
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {},
    "checked": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.\nIn markup, specified as &quot;checked=&#39;checked&#39;&quot; or just &quot;checked&quot;.\nTrue if the button is depressed, or the checkbox is checked,\nor the radio button is selected, etc.</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "disabled": {
     "!doc": "<p>Should this widget respond to user input?\nIn markup, this is specified as &quot;disabled=&#39;disabled&#39;&quot;, or just &quot;disabled&quot;.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "iconClass": {
     "!doc": "<p>Class to apply to DOMNode in button to make it display an icon</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "intermediateChanges": {
     "!doc": "<p>Fires onChange for each value change or only on demand</p>\n"
    },
    "label": {
     "!doc": "<p>Content to display in button.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "name": {
     "!doc": "<p>Name used when submitting form; same as &quot;name&quot; attribute or plain HTML elements</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "scrollOnFocus": {
     "!doc": "<p>On focus, should this widget scroll into view?</p>\n"
    },
    "searchContainerNode": {},
    "showLabel": {
     "!doc": "<p>Set this to true to hide the label text and display only the icon.\n(If showLabel=false then iconClass must be specified.)\nEspecially useful for toolbars.\nIf showLabel=true, the label will become the title (a.k.a. tooltip/hint) of the icon.\n\n</p>\n<p>The exception case is for computers in high-contrast mode, where the label\nwill still be displayed, since the icon doesn&#39;t appear.</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "tabIndex": {
     "!doc": "<p>Order fields are traversed when user hits the tab key</p>\n"
    },
    "templatePath": {
     "!doc": "<p>Path to template (HTML file) for this widget relative to dojo.baseUrl.\nDeprecated: use templateString with require([... &quot;dojo/text!...&quot;], ...) instead</p>\n"
    },
    "templateString": {},
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "type": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "value": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "compare": {
     "!type": "fn(val1: +anything, val2: +anything) -> number",
     "!doc": "<p>Compare 2 values (as returned by get(&#39;value&#39;) for this widget).</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn(preserveDom: bool)",
     "!doc": "<p>Destroy this widget, but not its descendants.  Descendants means widgets inside of\nthis.containerNode.   Will also destroy any resources (including widgets) registered via this.own().\n\n</p>\n<p>This method will also destroy internal widgets such as those created from a template,\nassuming those widgets exist inside of this.domNode but outside of this.containerNode.\n\n</p>\n<p>For 2.0 it&#39;s planned that this method will also destroy descendant widgets, so apps should not\ndepend on the current ability to destroy a widget without destroying its descendants.   Generally\nthey should use destroyRecursive() for widgets with children.</p>\n"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Put focus on this widget</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "getValue": {
     "!type": "fn()",
     "!doc": "<p>Deprecated.  Use get(&#39;value&#39;) instead.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()"
    },
    "reset": {
     "!type": "fn()",
     "!doc": "<p>Reset the widget&#39;s value to what it was at initialization time</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setChecked": {
     "!type": "fn(checked: bool)",
     "!doc": "<p>Deprecated.  Use set(&#39;checked&#39;, true/false) instead.</p>\n"
    },
    "setDisabled": {
     "!type": "fn(disabled: bool)",
     "!doc": "<p>Deprecated.  Use set(&#39;disabled&#39;, ...) instead.</p>\n"
    },
    "setLabel": {
     "!type": "fn(content: string)",
     "!doc": "<p>Deprecated.  Use set(&#39;label&#39;, ...) instead.</p>\n"
    },
    "setValue": {
     "!type": "fn(value: string)",
     "!doc": "<p>Deprecated.  Use set(&#39;value&#39;, ...) instead.</p>\n"
    },
    "startup": {
     "!type": "fn()",
     "!doc": "<p>Processing after the DOM fragment is added to the document</p>\n"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/form/_ToggleButtonMixin": {
   "!type": "fn()",
   "prototype": {
    "_aria_attr": {},
    "checked": {
     "!doc": "<p>Corresponds to the native HTML <code>&lt;input&gt;</code> element&#39;s attribute.\nIn markup, specified as &quot;checked=&#39;checked&#39;&quot; or just &quot;checked&quot;.\nTrue if the button is depressed, or the checkbox is checked,\nor the radio button is selected, etc.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "reset": {
     "!type": "fn()",
     "!doc": "<p>Reset the widget&#39;s value to what it was at initialization time</p>\n"
    }
   }
  },
  "dijit/_editor/_Plugin": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/Destroyable.prototype",
    "button": {
     "!doc": "<p>Pointer to <code>dijit/form/Button</code> or other widget (ex: <code>dijit/form/FilteringSelect</code>)\nthat is added to the toolbar to control this plugin.\nIf not specified, will be created on initialization according to <code>buttonClass</code></p>\n"
    },
    "command": {
     "!doc": "<p>String like &quot;insertUnorderedList&quot;, &quot;outdent&quot;, &quot;justifyCenter&quot;, etc. that represents an editor command.\nPassed to editor.execCommand() if <code>useDefaultCommand</code> is true.</p>\n"
    },
    "disabled": {
     "!doc": "<p>Flag to indicate if this plugin has been disabled and should do nothing\nhelps control button state, among other things.  Set via the setter api.</p>\n"
    },
    "editor": {
     "!doc": "<p>Points to the parent editor</p>\n"
    },
    "iconClassPrefix": {
     "!doc": "<p>The CSS class name for the button node is formed from <code>iconClassPrefix</code> and <code>command</code></p>\n"
    },
    "useDefaultCommand": {
     "!doc": "<p>If true, this plugin executes by calling Editor.execCommand() with the argument specified in <code>command</code>.</p>\n"
    },
    "buttonClass": {
     "!type": "fn()",
     "!doc": "<p>Class of widget (ex: dijit.form.Button or dijit/form/FilteringSelect)\nthat is added to the toolbar to control this plugin.\nThis is used to instantiate the button, unless <code>button</code> itself is specified directly.</p>\n"
    },
    "connect": {
     "!type": "fn(o: ?, f: ?, tf: ?)",
     "!doc": "<p>Deprecated.  Use this.own() with dojo/on or dojo/aspect.instead.\n\n</p>\n<p>Make a connect.connect() that is automatically disconnected when this plugin is destroyed.\nSimilar to <code>dijit/_Widget.connect()</code>.</p>\n"
    },
    "destroy": {
     "!type": "fn()"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a plugin.</p>\n"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "set": {
     "!type": "fn(name: +attribute, value: +anything) -> fn()",
     "!doc": "<p>Set a property on a plugin</p>\n"
    },
    "setEditor": {
     "!type": "fn(editor: +dijit)",
     "!doc": "<p>Tell the plugin which Editor it is associated with.</p>\n"
    },
    "setToolbar": {
     "!type": "fn(toolbar: +dijit)",
     "!doc": "<p>Tell the plugin to add it&#39;s controller widget (often a button)\nto the toolbar.  Does nothing if there is no controller widget.</p>\n"
    },
    "updateState": {
     "!type": "fn()",
     "!doc": "<p>Change state of the plugin to respond to events in the editor.</p>\n"
    }
   },
   "registry": {}
  },
  "dijit/_editor/_Plugin.registry": {
   "bidiSupport": {
    "!type": "fn(args: ?)"
   },
   "bidisupport": {
    "!type": "fn(args: ?)"
   },
   "bold": {
    "!type": "fn(args: ?)"
   },
   "colorTableCell": {
    "!type": "fn(args: ?)"
   },
   "copy": {
    "!type": "fn(args: ?)"
   },
   "createLink": {
    "!type": "fn()"
   },
   "cut": {
    "!type": "fn(args: ?)"
   },
   "delete": {
    "!type": "fn(args: ?)"
   },
   "deleteTableColumn": {
    "!type": "fn(args: ?)"
   },
   "deleteTableRow": {
    "!type": "fn(args: ?)"
   },
   "foreColor": {
    "!type": "fn(args: ?)"
   },
   "fullscreen": {
    "!type": "fn(args: ?)"
   },
   "fullScreen": {
    "!type": "fn(args: ?)"
   },
   "hiliteColor": {
    "!type": "fn(args: ?)"
   },
   "indent": {
    "!type": "fn(args: ?)"
   },
   "insertHorizontalRule": {
    "!type": "fn(args: ?)"
   },
   "insertImage": {
    "!type": "fn()"
   },
   "insertOrderedList": {
    "!type": "fn(args: ?)"
   },
   "insertTable": {
    "!type": "fn(args: ?)"
   },
   "insertTableColumnAfter": {
    "!type": "fn(args: ?)"
   },
   "insertTableColumnBefore": {
    "!type": "fn(args: ?)"
   },
   "insertTableRowAfter": {
    "!type": "fn(args: ?)"
   },
   "insertTableRowBefore": {
    "!type": "fn(args: ?)"
   },
   "insertUnorderedList": {
    "!type": "fn(args: ?)"
   },
   "italic": {
    "!type": "fn(args: ?)"
   },
   "justifyCenter": {
    "!type": "fn(args: ?)"
   },
   "justifyFull": {
    "!type": "fn(args: ?)"
   },
   "justifyLeft": {
    "!type": "fn(args: ?)"
   },
   "justifyRight": {
    "!type": "fn(args: ?)"
   },
   "localimage": {
    "!type": "fn(args: ?)"
   },
   "localImage": {
    "!type": "fn(args: ?)"
   },
   "LocalImage": {
    "!type": "fn(args: ?)"
   },
   "modifyTable": {
    "!type": "fn(args: ?)"
   },
   "newPage": {
    "!type": "fn(args: ?)"
   },
   "newpage": {
    "!type": "fn(args: ?)"
   },
   "outdent": {
    "!type": "fn(args: ?)"
   },
   "paste": {
    "!type": "fn(args: ?)"
   },
   "print": {
    "!type": "fn()"
   },
   "redo": {
    "!type": "fn(args: ?)"
   },
   "removeFormat": {
    "!type": "fn(args: ?)"
   },
   "selectAll": {
    "!type": "fn(args: ?)"
   },
   "strikethrough": {
    "!type": "fn(args: ?)"
   },
   "subscript": {
    "!type": "fn(args: ?)"
   },
   "superscript": {
    "!type": "fn(args: ?)"
   },
   "tabIndent": {
    "!type": "fn()"
   },
   "tableContextMenu": {
    "!type": "fn(args: ?)"
   },
   "toggleDir": {
    "!type": "fn()"
   },
   "underline": {
    "!type": "fn(args: ?)"
   },
   "undo": {
    "!type": "fn(args: ?)"
   },
   "unlink": {
    "!type": "fn(args: ?)"
   },
   "viewSource": {
    "!type": "fn(args: ?)"
   },
   "viewsource": {
    "!type": "fn(args: ?)"
   },
   "|": {
    "!type": "fn()"
   }
  },
  "dijit/_editor/plugins/EnterKeyHandling": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_editor/_Plugin.prototype",
    "blockNodeForEnter": {
     "!doc": "<p>This property decides the behavior of Enter key. It can be either P,\nDIV, BR, or empty (which means disable this feature). Anything else\nwill trigger errors.  The default is &#39;BR&#39;\n\n</p>\n<p>See class description for more details.</p>\n"
    },
    "blockNodes": {
     "!doc": "<p>Regex for testing if a given tag is a block level (display:block) tag</p>\n"
    },
    "bogusHtmlContent": {
     "!doc": "<p>HTML to stick into a new empty block</p>\n"
    },
    "button": {
     "!doc": "<p>Pointer to <code>dijit/form/Button</code> or other widget (ex: <code>dijit/form/FilteringSelect</code>)\nthat is added to the toolbar to control this plugin.\nIf not specified, will be created on initialization according to <code>buttonClass</code></p>\n"
    },
    "command": {
     "!doc": "<p>String like &quot;insertUnorderedList&quot;, &quot;outdent&quot;, &quot;justifyCenter&quot;, etc. that represents an editor command.\nPassed to editor.execCommand() if <code>useDefaultCommand</code> is true.</p>\n"
    },
    "disabled": {
     "!doc": "<p>Flag to indicate if this plugin has been disabled and should do nothing\nhelps control button state, among other things.  Set via the setter api.</p>\n"
    },
    "editor": {
     "!doc": "<p>Points to the parent editor</p>\n"
    },
    "iconClassPrefix": {
     "!doc": "<p>The CSS class name for the button node is formed from <code>iconClassPrefix</code> and <code>command</code></p>\n"
    },
    "useDefaultCommand": {
     "!doc": "<p>If true, this plugin executes by calling Editor.execCommand() with the argument specified in <code>command</code>.</p>\n"
    },
    "buttonClass": {
     "!type": "fn()",
     "!doc": "<p>Class of widget (ex: dijit.form.Button or dijit/form/FilteringSelect)\nthat is added to the toolbar to control this plugin.\nThis is used to instantiate the button, unless <code>button</code> itself is specified directly.</p>\n"
    },
    "connect": {
     "!type": "fn(o: ?, f: ?, tf: ?)",
     "!doc": "<p>Deprecated.  Use this.own() with dojo/on or dojo/aspect.instead.\n\n</p>\n<p>Make a connect.connect() that is automatically disconnected when this plugin is destroyed.\nSimilar to <code>dijit/_Widget.connect()</code>.</p>\n"
    },
    "destroy": {
     "!type": "fn()"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a plugin.</p>\n"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "set": {
     "!type": "fn(name: +attribute, value: +anything) -> fn()",
     "!doc": "<p>Set a property on a plugin</p>\n"
    },
    "setEditor": {
     "!type": "fn(editor: ?)"
    },
    "setToolbar": {
     "!type": "fn(toolbar: +dijit)",
     "!doc": "<p>Tell the plugin to add it&#39;s controller widget (often a button)\nto the toolbar.  Does nothing if there is no controller widget.</p>\n"
    },
    "updateState": {
     "!type": "fn()",
     "!doc": "<p>Change state of the plugin to respond to events in the editor.</p>\n"
    }
   }
  },
  "dijit/_editor/RichText": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/_Widget.prototype",
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_editorCommandsLocalized": {},
    "_focusManager": {},
    "_local2NativeFormatNames": {},
    "_mozSettingProps": {},
    "_NAME_CONTENT_SEP": {
     "!doc": "<p>USed to separate name from content.  Just a colon isn&#39;t safe.</p>\n"
    },
    "_native2LocalFormatNames": {},
    "_qcaCache": {},
    "_SEPARATOR": {
     "!doc": "<p>Used to concat contents from multiple editors into a single string,\nso they can be saved into a single <code>&lt;textarea&gt;</code> node.  See &quot;name&quot; attribute.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTypeAttr": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "_updateTimer": {},
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {},
    "captureEvents": {
     "!doc": "<p>Events which should be connected to the underlying editing\narea, events in this array will be addListener with\ncapture=true.</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<p>In templated widgets, &quot;containerNode&quot; is set via a\ndata-dojo-attach-point assignment.\n\n</p>\n<p>containerNode must be defined for any widget that accepts innerHTML\n(like ContentPane or BorderContainer or even Button), and conversely\nis null for widgets that don&#39;t, like TextBox.</p>\n"
    },
    "cssStateNodes": {
     "!doc": "<p>Subclasses may define a cssStateNodes property that lists sub-nodes within the widget that\nneed CSS classes applied on mouse hover/press and focus.\n\n</p>\n<p>Each entry in this optional hash is a an attach-point name (like &quot;upArrowButton&quot;) mapped to a CSS class name\n(like &quot;dijitUpArrowButton&quot;). Example:\n\n</p>\n<pre><code>{\n    &quot;upArrowButton&quot;: &quot;dijitUpArrowButton&quot;,\n    &quot;downArrowButton&quot;: &quot;dijitDownArrowButton&quot;\n}</code></pre>\n<p>The above will set the CSS class dijitUpArrowButton to the this.upArrowButton DOMNode when it\n\n</p>\n<p>is hovered, etc.</p>\n"
    },
    "dir": {
     "!doc": "<p>Bi-directional support, as defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-dir\">HTML DIR</a>\nattribute. Either left-to-right &quot;ltr&quot; or right-to-left &quot;rtl&quot;.  If undefined, widgets renders in page&#39;s\ndefault direction.</p>\n"
    },
    "disabled": {
     "!doc": "<p>The editor is disabled; the text cannot be changed.</p>\n"
    },
    "disableSpellCheck": {
     "!doc": "<p>When true, disables the browser&#39;s native spell checking, if supported.\nWorks only in Firefox.</p>\n"
    },
    "domNode": {
     "!doc": "<p>This is our visible representation of the widget! Other DOM\nNodes may by assigned to other properties, usually through the\ntemplate system&#39;s data-dojo-attach-point syntax, but the domNode\nproperty is the canonical &quot;top level&quot; node in widget UI.</p>\n"
    },
    "events": {
     "!doc": "<p>events which should be connected to the underlying editing area</p>\n"
    },
    "focused": {
     "!doc": "<p>This widget or a widget it contains has focus, or is &quot;active&quot; because\nit was recently clicked.</p>\n"
    },
    "focusOnLoad": {
     "!doc": "<p>Focus into this widget when the page is loaded</p>\n"
    },
    "height": {
     "!doc": "<p>Set height to fix the editor at a specific height, with scrolling.\nBy default, this is 300px.  If you want to have the editor always\nresizes to accommodate the content, use AlwaysShowToolbar plugin\nand set height=&quot;&quot;.  If this editor is used within a layout widget,\nset height=&quot;100%&quot;.</p>\n"
    },
    "hovering": {
     "!doc": "<p>True if cursor is over this widget</p>\n"
    },
    "id": {
     "!doc": "<p>A unique, opaque ID string that can be assigned by users or by the\nsystem. If the developer passes an ID which is known not to be\nunique, the specified ID is ignored and the system-generated ID is\nused instead.</p>\n"
    },
    "inheritWidth": {
     "!doc": "<p>whether to inherit the parent&#39;s width or simply use 100%</p>\n"
    },
    "isClosed": {},
    "isLoaded": {},
    "isTabIndent": {
     "!doc": "<p>Make tab key and shift-tab indent and outdent rather than navigating.\nCaution: sing this makes web pages inaccessible to users unable to use a mouse.</p>\n"
    },
    "lang": {
     "!doc": "<p>Rarely used.  Overrides the default Dojo locale used to render this widget,\nas defined by the <a href=\"http://www.w3.org/TR/html401/struct/dirlang.html#adef-lang\">HTML LANG</a> attribute.\nValue must be among the list of locales specified during by the Dojo bootstrap,\nformatted according to <a href=\"http://www.ietf.org/rfc/rfc3066.txt\">RFC 3066</a> (like en-us).</p>\n"
    },
    "minHeight": {
     "!doc": "<p>The minimum height that the editor should have.</p>\n"
    },
    "name": {
     "!doc": "<p>Specifies the name of a (hidden) <code>&lt;textarea&gt;</code> node on the page that&#39;s used to save\nthe editor content on page leave.   Used to restore editor contents after navigating\nto a new page and then hitting the back button.</p>\n"
    },
    "onLoadDeferred": {
     "!doc": "<p>Deferred which is fired when the editor finishes loading.\nCall myEditor.onLoadDeferred.then(callback) it to be informed\nwhen the rich-text area initialization is finalized.</p>\n"
    },
    "ownerDocument": {
     "!doc": "<p>The document this widget belongs to.  If not specified to constructor, will default to\nsrcNodeRef.ownerDocument, or if no sourceRef specified, then to the document global</p>\n"
    },
    "srcNodeRef": {
     "!doc": "<p>pointer to original DOM node</p>\n"
    },
    "style": {
     "!doc": "<p>HTML style attributes as cssText string or name/value hash</p>\n"
    },
    "styleSheets": {
     "!doc": "<p>semicolon (&quot;;&quot;) separated list of css files for the editing area</p>\n"
    },
    "title": {
     "!doc": "<p>HTML title attribute.\n\n</p>\n<p>For form widgets this specifies a tooltip to display when hovering over\nthe widget (just like the native HTML title attribute).\n\n</p>\n<p>For TitlePane or for when this widget is a child of a TabContainer, AccordionContainer,\netc., it&#39;s used to specify the tab label, accordion pane title, etc.  In this case it&#39;s\ninterpreted as HTML.</p>\n"
    },
    "tooltip": {
     "!doc": "<p>When this widget&#39;s title attribute is used to for a tab label, accordion pane title, etc.,\nthis specifies the tooltip to appear when the mouse is hovered over that text.</p>\n"
    },
    "updateInterval": {},
    "addKeyHandler": {
     "!type": "fn(key: string|number, ctrl: bool, shift: bool, handler: fn())",
     "!doc": "<p>Add a handler for a keyboard shortcut</p>\n"
    },
    "addStyleSheet": {
     "!type": "fn(uri: +dojo)",
     "!doc": "<p>add an external stylesheet for the editing area</p>\n"
    },
    "attr": {
     "!type": "fn(name: string|?, value?: ?)",
     "!doc": "<p>This method is deprecated, use get() or set() directly.</p>\n"
    },
    "blur": {
     "!type": "fn()",
     "!doc": "<p>Remove focus from this instance.</p>\n"
    },
    "buildRendering": {
     "!type": "fn()",
     "!doc": "<p>Construct the UI for this widget, setting this.domNode.\nMost widgets will mixin <code>dijit._TemplatedMixin</code>, which implements this method.</p>\n"
    },
    "connect": {
     "!type": "fn(obj: ?|?, event: string|fn(), method: string|fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(on(...)) or this.own(aspect.after(...)) instead.\n\n</p>\n<p>Connects specified obj/event to specified method of this object\nand registers for disconnect() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.connect, except with the\nimplicit use of this widget as the target object.\nEvents connected with <code>this.connect</code> are disconnected upon\ndestruction.</p>\n"
    },
    "defer": {
     "!type": "fn(fcn: fn(), delay?: number)",
     "!doc": "<p>Wrapper to setTimeout to avoid deferred functions executing\nafter the originating widget has been destroyed.\nReturns an object handle with a remove method (that returns null) (replaces clearTimeout).</p>\n"
    },
    "destroy": {
     "!type": "fn()"
    },
    "destroyDescendants": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Recursively destroy the children of this widget and their\ndescendants.</p>\n"
    },
    "destroyRecursive": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroy this widget and its descendants</p>\n"
    },
    "destroyRendering": {
     "!type": "fn(preserveDom?: bool)",
     "!doc": "<p>Destroys the DOM nodes associated with this widget.</p>\n"
    },
    "disconnect": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Disconnects handle created by <code>connect</code>.</p>\n"
    },
    "emit": {
     "!type": "fn(type: string, eventObj?: ?, callbackArgs?: [])",
     "!doc": "<p>Used by widgets to signal that a synthetic event occurred, ex:\n\n</p>\n<pre><code>myWidget.emit(&quot;attrmodified-selectedChildWidget&quot;, {}).</code></pre>\n<p>Emits an event on this.domNode named type.toLowerCase(), based on eventObj.\nAlso calls onType() method, if present, and returns value from that method.\nBy default passes eventObj to callback, but will pass callbackArgs instead, if specified.\nModifies eventObj by adding missing parameters (bubbles, cancelable, widget).</p>\n"
    },
    "execCommand": {
     "!type": "fn(command: string, argument: ?)",
     "!doc": "<p>Executes a command in the Rich Text area</p>\n"
    },
    "focus": {
     "!type": "fn()",
     "!doc": "<p>Move focus to this editor</p>\n"
    },
    "get": {
     "!type": "fn(name: ?)",
     "!doc": "<p>Get a property from a widget.</p>\n"
    },
    "getChildren": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all direct children of this widget, i.e. all widgets underneath this.containerNode whose parent\nis this widget.   Note that it does not return all descendants, but rather just direct children.\nAnalogous to <a href=\"https://developer.mozilla.org/en-US/docs/DOM/Node.childNodes\">Node.childNodes</a>,\nexcept containing widgets rather than DOMNodes.\n\n</p>\n<p>The result intentionally excludes internally created widgets (a.k.a. supporting widgets)\noutside of this.containerNode.\n\n</p>\n<p>Note that the array returned is a simple array.  Application code should not assume\nexistence of methods like forEach().</p>\n"
    },
    "getDescendants": {
     "!type": "fn() -> []",
     "!doc": "<p>Returns all the widgets contained by this, i.e., all widgets underneath this.containerNode.\nThis method should generally be avoided as it returns widgets declared in templates, which are\nsupposed to be internal/hidden, but it&#39;s left here for back-compat reasons.</p>\n"
    },
    "getFooterHeight": {
     "!type": "fn()",
     "!doc": "<p>A function for obtaining the height of the footer node</p>\n"
    },
    "getHeaderHeight": {
     "!type": "fn()",
     "!doc": "<p>A function for obtaining the height of the header node</p>\n"
    },
    "getNodeChildrenHtml": {
     "!type": "fn(dom: +Node)",
     "!doc": "<p>Deprecated.   Use dijit/_editor/html::getChildrenHtml() instead.</p>\n"
    },
    "getNodeHtml": {
     "!type": "fn(node: +Node)",
     "!doc": "<p>Deprecated.   Use dijit/_editor/html::_getNodeHtml() instead.</p>\n"
    },
    "getParent": {
     "!type": "fn()",
     "!doc": "<p>Returns the parent widget of this widget.</p>\n"
    },
    "isFocusable": {
     "!type": "fn()",
     "!doc": "<p>Return true if this widget can currently be focused\nand false if not</p>\n"
    },
    "isLeftToRight": {
     "!type": "fn()",
     "!doc": "<p>Return this widget&#39;s explicit or implicit orientation (true for LTR, false for RTL)</p>\n"
    },
    "on": {
     "!type": "fn(type: string|fn(), func: fn())"
    },
    "own": {
     "!type": "fn()",
     "!doc": "<p>Track specified handles and remove/destroy them when this instance is destroyed, unless they were\nalready removed/destroyed manually.</p>\n"
    },
    "placeAt": {
     "!type": "fn(reference: string|+Node|+DocumentFragment|+dijit, position?: string|number) -> +dijit",
     "!doc": "<p>Place this widget somewhere in the DOM based\non standard domConstruct.place() conventions.</p>\n"
    },
    "postCreate": {
     "!type": "fn()"
    },
    "postMixInProperties": {
     "!type": "fn()",
     "!doc": "<p>Called after the parameters to the widget have been read-in,\nbut before the widget template is instantiated. Especially\nuseful to set properties that are referenced in the widget\ntemplate.</p>\n"
    },
    "queryCommandEnabled": {
     "!type": "fn(command: string) -> bool",
     "!doc": "<p>Check whether a command is enabled or not.</p>\n"
    },
    "queryCommandState": {
     "!type": "fn(command: ?) -> bool",
     "!doc": "<p>Check the state of a given command and returns true or false.</p>\n"
    },
    "queryCommandValue": {
     "!type": "fn(command: ?) -> bool",
     "!doc": "<p>Check the value of a given command. This matters most for\ncustom selections and complex values like font value setting.</p>\n"
    },
    "removeStyleSheet": {
     "!type": "fn(uri: +dojo)",
     "!doc": "<p>remove an external stylesheet for the editing area</p>\n"
    },
    "replaceValue": {
     "!type": "fn(html: string)",
     "!doc": "<p>This function set the content while trying to maintain the undo stack\n(now only works fine with Moz, this is identical to setValue in all\nother browsers)</p>\n"
    },
    "set": {
     "!type": "fn(name: ?, value: ?) -> fn()",
     "!doc": "<p>Set a property on a widget</p>\n"
    },
    "setAttribute": {
     "!type": "fn(attr: string, value: +anything)",
     "!doc": "<p>Deprecated.  Use set() instead.</p>\n"
    },
    "setDisabled": {
     "!type": "fn(disabled: bool)",
     "!doc": "<p>Deprecated, use set(&#39;disabled&#39;, ...) instead.</p>\n"
    },
    "setupDefaultShortcuts": {
     "!type": "fn()",
     "!doc": "<p>Add some default key handlers</p>\n"
    },
    "setValue": {
     "!type": "fn(html: string)",
     "!doc": "<p>This function sets the content. No undo history is preserved.\nUsers should use set(&#39;value&#39;, ...) instead.</p>\n"
    },
    "startup": {
     "!type": "fn()"
    },
    "subscribe": {
     "!type": "fn(t: string, method: fn())",
     "!doc": "<p>Deprecated, will be removed in 2.0, use this.own(topic.subscribe()) instead.\n\n</p>\n<p>Subscribes to the specified topic and calls the specified method\nof this object and registers for unsubscribe() on widget destroy.\n\n</p>\n<p>Provide widget-specific analog to dojo.subscribe, except with the\nimplicit use of this widget as the target object.</p>\n"
    },
    "toString": {
     "!type": "fn() -> string",
     "!doc": "<p>Returns a string that represents the widget.</p>\n"
    },
    "uninitialize": {
     "!type": "fn() -> bool",
     "!doc": "<p>Deprecated. Override destroy() instead to implement custom widget tear-down\nbehavior.</p>\n"
    },
    "unsubscribe": {
     "!type": "fn(handle: ?)",
     "!doc": "<p>Deprecated, will be removed in 2.0, use handle.remove() instead.\n\n</p>\n<p>Unsubscribes handle created by this.subscribe.\nAlso removes handle from this widget&#39;s list of subscriptions</p>\n"
    },
    "watch": {
     "!type": "fn(name?: string, callback: fn())",
     "!doc": "<p>Watches a property for changes</p>\n"
    }
   }
  },
  "dijit/selection": {
   "collapse": {
    "!type": "fn(beginning: bool)",
    "!doc": "<p>Function to collapse (clear), the current selection</p>\n"
   },
   "getAncestorElement": {
    "!type": "fn(tagName: string)",
    "!doc": "<p>Return the parent element of the current selection which is of\ntype tagName (or one of the other specified tagName)</p>\n"
   },
   "getBookmark": {
    "!type": "fn()",
    "!doc": "<p>Retrieves a bookmark that can be used with moveToBookmark to reselect the currently selected range.</p>\n"
   },
   "getParentElement": {
    "!type": "fn()",
    "!doc": "<p>Get the parent element of the current selection</p>\n"
   },
   "getParentOfType": {
    "!type": "fn(node: +Node, tags: [string])",
    "!doc": "<p>Function to locate a parent node that matches one of a set of tags</p>\n"
   },
   "getSelectedElement": {
    "!type": "fn()",
    "!doc": "<p>Retrieves the selected element (if any), just in the case that\na single element (object like and image or a table) is\nselected.</p>\n"
   },
   "getSelectedHtml": {
    "!type": "fn()",
    "!doc": "<p>Return the html text of the current selection or null if unavailable</p>\n"
   },
   "getSelectedText": {
    "!type": "fn()",
    "!doc": "<p>Return the text (no html tags) included in the current selection or null if no text is selected</p>\n"
   },
   "getType": {
    "!type": "fn()",
    "!doc": "<p>Get the selection type (like doc.select.type in IE).</p>\n"
   },
   "hasAncestorElement": {
    "!type": "fn(tagName: string)",
    "!doc": "<p>Check whether current selection has a  parent element which is\nof type tagName (or one of the other specified tagName)</p>\n"
   },
   "inSelection": {
    "!type": "fn(node: ?)",
    "!doc": "<p>This function determines if &#39;node&#39; is\nin the current selection.</p>\n"
   },
   "isCollapsed": {
    "!type": "fn()",
    "!doc": "<p>Returns true if there is no text selected</p>\n"
   },
   "isTag": {
    "!type": "fn(node: +Node, tags: [string])",
    "!doc": "<p>Function to determine if a node is one of an array of tags.</p>\n"
   },
   "moveToBookmark": {
    "!type": "fn(bookmark: ?)",
    "!doc": "<p>Moves current selection to a bookmark.</p>\n"
   },
   "remove": {
    "!type": "fn()",
    "!doc": "<p>Function to delete the currently selected content from the document.</p>\n"
   },
   "selectElement": {
    "!type": "fn(element: +Node, nochangefocus?: bool)",
    "!doc": "<p>clear previous selection and select element (including all its children)</p>\n"
   },
   "selectElementChildren": {
    "!type": "fn(element: +Node, nochangefocus?: bool)",
    "!doc": "<p>clear previous selection and select the content of the node\n(excluding the node itself)</p>\n"
   }
  },
  "dijit/_editor/range": {
   "BlockTagNames": {},
   "ie": {},
   "adjacentNoneTextNode": {
    "!type": "fn(startnode: ?, next: ?) -> []"
   },
   "atBeginningOfContainer": {
    "!type": "fn(container: +Node, node: +Node, offset: number) -> bool"
   },
   "atEndOfContainer": {
    "!type": "fn(container: +Node, node: +Node, offset: number) -> bool"
   },
   "create": {
    "!type": "fn(win?: +Window)"
   },
   "getAncestor": {
    "!type": "fn(node: +Node, regex?: +RegEx, root?: +Node) -> +Node"
   },
   "getBlockAncestor": {
    "!type": "fn(node: +Node, regex?: +RegEx, root?: +Node)"
   },
   "getCommonAncestor": {
    "!type": "fn(n1: ?, n2: ?, root: ?)"
   },
   "getIndex": {
    "!type": "fn(node: +Node, parent: +Node)"
   },
   "getNode": {
    "!type": "fn(index: [], parent: +Node) -> +Node"
   },
   "getSelection": {
    "!type": "fn(window: +Window, ignoreUpdate?: bool)"
   },
   "W3CRange": {
    "!type": "fn()"
   }
  },
  "dijit/_editor/range.ie": {
   "cachedSelection": {},
   "decomposeControlRange": {
    "!type": "fn(range: ?) -> []"
   },
   "decomposeTextRange": {
    "!type": "fn(range: ?) -> []"
   },
   "getEndPoint": {
    "!type": "fn(range: ?, end: ?) -> []"
   },
   "selection": {
    "!type": "fn(window: ?)"
   },
   "setEndPoint": {
    "!type": "fn(range: ?, container: ?, offset: ?)"
   },
   "setRange": {
    "!type": "fn(range: ?, startContainer: ?, startOffset: ?, endContainer: ?, endOffset: ?, collapsed: ?)"
   }
  },
  "dijit/_editor/range.W3CRange": {
   "!type": "fn()",
   "prototype": {
    "cloneRange": {
     "!type": "fn()"
    },
    "deleteContents": {
     "!type": "fn()"
    },
    "detach": {
     "!type": "fn()"
    },
    "getBookmark": {
     "!type": "fn()"
    },
    "setEnd": {
     "!type": "fn(node: ?, offset: ?)"
    },
    "setEndAfter": {
     "!type": "fn(node: ?, offset: ?)"
    },
    "setEndBefore": {
     "!type": "fn(node: ?, offset: ?)"
    },
    "setStart": {
     "!type": "fn(node: ?, offset: ?)"
    },
    "setStartAfter": {
     "!type": "fn(node: ?, offset: ?)"
    },
    "setStartBefore": {
     "!type": "fn(node: ?, offset: ?)"
    }
   }
  },
  "dijit/_editor/html": {
   "escapeXml": {
    "!type": "fn(str: string, noSingleQuotes?: bool)",
    "!doc": "<p>Adds escape sequences for special characters in XML: <code>&amp;&lt;&gt;&quot;&#39;</code>.\nOptionally skips escapes for single quotes.</p>\n"
   },
   "getChildrenHtml": {
    "!type": "fn(node: +Node)",
    "!doc": "<p>Returns the html content of a DomNode&#39;s children</p>\n"
   },
   "getChildrenHtmlHelper": {
    "!type": "fn(dom: +Node, output: [string])",
    "!doc": "<p>Pushes the html content of a DomNode&#39;s children into out[]</p>\n"
   },
   "getNodeHtml": {
    "!type": "fn(node: +Node)",
    "!doc": "<p>Return string representing HTML for node and it&#39;s children</p>\n"
   },
   "getNodeHtmlHelper": {
    "!type": "fn(node: +Node, output: [string]) -> string",
    "!doc": "<p>Pushes array of strings into output[] which represent HTML for node and it&#39;s children</p>\n"
   }
  },
  "dijit/Fieldset": {
   "!type": "fn()",
   "prototype": {
    "!proto": "dijit/TitlePane.prototype",
    "_attachEvents": {
     "!doc": "<p>List of connections associated with data-dojo-attach-event=... in the\ntemplate</p>\n"
    },
    "_attachPoints": {
     "!doc": "<p>List of widget attribute names associated with data-dojo-attach-point=... in the\ntemplate, ex: [&quot;containerNode&quot;, &quot;labelNode&quot;]</p>\n"
    },
    "_attrPairNames": {},
    "_blankGif": {
     "!doc": "<p>Path to a blank 1x1 image.\nUsed by <code>&lt;img&gt;</code> nodes in templates that really get their image via CSS background-image.</p>\n"
    },
    "_focusManager": {},
    "_rendered": {
     "!doc": "<p>Not normally use, but this flag can be set by the app if the server has already rendered the template,\ni.e. already inlining the template for the widget into the main page.   Reduces _TemplatedMixin to\njust function like _AttachMixin.</p>\n"
    },
    "_setClassAttr": {},
    "_setDirAttr": {},
    "_setIdAttr": {},
    "_setLangAttr": {},
    "_setTitleAttr": {},
    "_setTooltipAttr": {},
    "_setTypeAttr": {},
    "_skipNodeCache": {},
    "_started": {
     "!doc": "<p>startup() has completed.</p>\n"
    },
    "active": {
     "!doc": "<p>True if mouse was pressed while over this widget, and hasn&#39;t been released yet</p>\n"
    },
    "attachScope": {
     "!doc": "<p>Object to which attach points and events will be scoped.  Defaults\nto &#39;this&#39;.</p>\n"
    },
    "attributeMap": {
     "!doc": "<p>Deprecated. Instead of attributeMap, widget should have a _setXXXAttr attribute\nfor each XXX attribute to be mapped to the DOM.\n\n</p>\n<p>attributeMap sets up a &quot;binding&quot; between attributes (aka properties)\nof the widget and the widget&#39;s DOM.\nChanges to widget attributes listed in attributeMap will be\nreflected into the DOM.\n\n</p>\n<p>For example, calling set(&#39;title&#39;, &#39;hello&#39;)\non a TitlePane will automatically cause the TitlePane&#39;s DOM to update\nwith the new title.\n\n</p>\n<p>attributeMap is a hash where the key is an attribute of the widget,\nand the value reflects a binding to a:\n\n</p>\n<ul>\n<li><p>DOM node attribute</p>\n<p>  focus: {node: &quot;focusNode&quot;, type: &quot;attribute&quot;}\nMaps this.focus to this.focusNode.focus</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerHTML</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerHTML&quot; }\nMaps this.title to this.titleNode.innerHTML</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node innerText</p>\n<p>  title: { node: &quot;titleNode&quot;, type: &quot;innerText&quot; }\nMaps this.title to this.titleNode.innerText</p>\n</li>\n</ul>\n<ul>\n<li><p>DOM node CSS class</p>\n<p>  myClass: { node: &quot;domNode&quot;, type: &quot;class&quot; }\nMaps this.myClass to this.domNode.className</p>\n</li>\n</ul>\n<p>If the value is an array, then each element in the array matches one of the\nformats of the above list.\n\n</p>\n<p>There are also some shorthands for backwards compatibility:\n\n</p>\n<ul>\n<li>string --&gt; { node: string, type: &quot;attribute&quot; }, for example:</li>\n</ul>\n<pre><code>&quot;focusNode&quot; ---&gt; { node: &quot;focusNode&quot;, type: &quot;attribute&quot; }</code></pre>\n<ul>\n<li>&quot;&quot; --&gt; { node: &quot;domNode&quot;, type: &quot;attribute&quot; }</li>\n</ul>\n"
    },
    "baseClass": {
     "!doc": "<p>The root className to use for the various states of this widget</p>\n"
    },
    "class": {},
    "containerNode": {
     "!doc": "<p>Designates where children of the source DOM node will be placed.\n&quot;Children&quot; in this case refers to both DOM nodes and widgets.\nFor example, for myWidget:\n\n\n</p>\n<pre><code>&lt;div data-dojo-type=myWidget&gt;\n    &lt;b&gt; here&#39;s a plain DOM node\n    &lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n    &lt;i&gt; and another plain DOM node &lt;/i&gt;\n&lt;/div&gt;</code></pre>\n<p>containerNode would point to:\n\n\n</p>\n<pre><code>&lt;b&gt; here&#39;s a plain DOM node\n&lt;span data-dojo-type=subWidget&gt;and a widget&lt;/span&gt;\n&lt;i&gt; and another plain DOM node &lt;/i&gt;</code></pre>\n<